package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.model.CampaignDayBudgetNotificationStatus;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromPropertyToField;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithDayBudgetTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithDayBudget> {
    private static final JooqMapper<CampaignWithDayBudget> MAPPER = createMapper();

    @Autowired
    public CampaignWithDayBudgetTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithDayBudget> getTypeClass() {
        return CampaignWithDayBudget.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithDayBudget>> appliedChanges) {
        updateHelperAggregator.getOrCreate(Campaigns.CAMPAIGNS.CID).processUpdateAll(MAPPER, appliedChanges);
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithDayBudget campaign) {
        insertHelperAggregator.getOrCreate(Campaigns.CAMPAIGNS).add(MAPPER, campaign);
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(MAPPER, campaign);
    }

    @Override
    public void fillFromRecord(CampaignWithDayBudget campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    private static JooqMapper<CampaignWithDayBudget> createMapper() {
        return JooqMapperBuilder.<CampaignWithDayBudget>builder()
                .map(property(CampaignWithDayBudget.WALLET_ID, CAMPAIGNS.WALLET_CID))
                .map(convertibleProperty(CampaignWithDayBudget.TYPE, CAMPAIGNS.TYPE, CampaignType::fromSource,
                        CampaignType::toSource))
                .map(property(CampaignWithDayBudget.DAY_BUDGET, CAMPAIGNS.DAY_BUDGET))
                .map(convertibleProperty(CampaignWithDayBudget.DAY_BUDGET_SHOW_MODE, CAMPAIGNS.DAY_BUDGET_SHOW_MODE,
                        DayBudgetShowMode::fromSource, DayBudgetShowMode::toSource))
                .map(convertibleProperty(CampaignWithDayBudget.DAY_BUDGET_DAILY_CHANGE_COUNT,
                        CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT, Long::intValue, Integer::longValue))
                .map(convertibleProperty(CampaignWithDayBudget.DAY_BUDGET_NOTIFICATION_STATUS,
                        CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS,
                        CampaignDayBudgetNotificationStatus::fromSource, CampaignDayBudgetNotificationStatus::toSource))
                .writeField(CAMP_OPTIONS.DAY_BUDGET_LAST_CHANGE,
                        fromPropertyToField(CampaignWithDayBudget.DAY_BUDGET_LAST_CHANGE)
                                .by(RepositoryUtils::setCurrentLocalDateTimeIfShould))
                .readProperty(CampaignWithDayBudget.DAY_BUDGET_LAST_CHANGE,
                        fromField(CAMP_OPTIONS.DAY_BUDGET_LAST_CHANGE))
                .writeField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME,
                        fromPropertyToField(CampaignWithDayBudget.DAY_BUDGET_STOP_TIME)
                                .by(RepositoryUtils::zeroableDateTimeToDb))
                .readProperty(CampaignWithDayBudget.DAY_BUDGET_STOP_TIME, fromField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME))
                .build();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(CAMP_OPTIONS, JoinType.LEFT_OUTER_JOIN, CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID)));
    }
}
