package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithDialog;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.dbschema.ppc.tables.records.CampDialogsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_DIALOGS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithDialogTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithDialog> {
    private static final JooqMapper<CampaignWithDialog> MAPPER = createCampaignWithDialogMapper();

    @Autowired
    CampaignWithDialogTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    private static JooqMapper<CampaignWithDialog> createCampaignWithDialogMapper() {
        return JooqMapperBuilder.<CampaignWithDialog>builder()
                .map(property(CampaignWithDialog.ID, CAMP_DIALOGS.CID))
                .map(property(CampaignWithDialog.CLIENT_DIALOG_ID, CAMP_DIALOGS.CLIENT_DIALOG_ID))
                .build();
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public void updateAdditionTables(DSLContext context,
                                     RestrictedCampaignsUpdateOperationContainer updateParameters,
                                     Collection<AppliedChanges<CampaignWithDialog>> appliedChanges) {
        List<CampaignWithDialog> changedCampaigns = filterAndMapList(
                appliedChanges,
                ac -> ac.changed(CampaignWithDialog.CLIENT_DIALOG_ID),
                AppliedChanges::getModel);

        List<Long> idsToDelete = filterAndMapList(changedCampaigns,
                campaign -> campaign.getClientDialogId() == null,
                CampaignWithDialog::getId);
        context.deleteFrom(CAMP_DIALOGS).where(CAMP_DIALOGS.CID.in(idsToDelete)).execute();

        List<CampaignWithDialog> campaignsToUpdate = filterList(changedCampaigns,
                campaign -> campaign.getClientDialogId() != null);
        insertCampaignsDialogs(context, campaignsToUpdate);
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                       Collection<CampaignWithDialog> campaigns) {
        List<CampaignWithDialog> campaignsToInsert = filterList(campaigns,
                campaign -> campaign.getClientDialogId() != null);
        insertCampaignsDialogs(context, campaignsToInsert);
    }

    private void insertCampaignsDialogs(DSLContext context, List<CampaignWithDialog> campaignsToUpdate) {
        InsertHelper<CampDialogsRecord> insertHelper = new InsertHelper<>(context, CAMP_DIALOGS);
        insertHelper.addAll(MAPPER, campaignsToUpdate);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CAMP_DIALOGS.CLIENT_DIALOG_ID, MySQLDSL.values(CAMP_DIALOGS.CLIENT_DIALOG_ID));
        }
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(CAMP_DIALOGS, JoinType.LEFT_OUTER_JOIN, CAMP_DIALOGS.CID.eq(CAMPAIGNS.CID)));
    }

    @Override
    public <M extends CampaignWithDialog> void fillFromRecord(M campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    @Override
    public Class<CampaignWithDialog> getTypeClass() {
        return CampaignWithDialog.class;
    }
}
