package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.converter.ExperimentsConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithExperiments;
import ru.yandex.direct.dbschema.ppc.tables.RetargetingConditions;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithExperimentsTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithExperiments> {
    private static final RetargetingConditions RETARGETING_CONDITIONS_SECTIONS =
            RETARGETING_CONDITIONS.as("retageting_condition_exp_sections");
    private static final RetargetingConditions RETARGETING_CONDITIONS_AB_GOALS =
            RETARGETING_CONDITIONS.as("retageting_condition_exp_abgoals");
    private static final JooqMapper<CampaignWithExperiments> MAPPER = createCampaignWithExperimentsMapper();

    @Autowired
    public CampaignWithExperimentsTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(
                new JoinQuery(
                        RETARGETING_CONDITIONS_SECTIONS,
                        JoinType.LEFT_OUTER_JOIN,
                        CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID.eq(RETARGETING_CONDITIONS_SECTIONS.RET_COND_ID)
                ),
                new JoinQuery(
                        RETARGETING_CONDITIONS_AB_GOALS,
                        JoinType.LEFT_OUTER_JOIN,
                        CAMPAIGNS.AB_SEGMENT_RET_COND_ID.eq(RETARGETING_CONDITIONS_AB_GOALS.RET_COND_ID)
                )
        );
    }

    @Override
    public <M extends CampaignWithExperiments> void fillFromRecord(M campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    @Override
    public Class<CampaignWithExperiments> getTypeClass() {
        return CampaignWithExperiments.class;
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithExperiments>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(MAPPER, appliedChanges);
        updateHelperAggregator.getOrCreate(CAMPAIGNS.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithExperiments campaign) {
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(MAPPER, campaign);
        insertHelperAggregator.getOrCreate(CAMPAIGNS).add(MAPPER, campaign);
    }

    private static JooqMapper<CampaignWithExperiments> createCampaignWithExperimentsMapper() {
        return JooqMapperBuilder.<CampaignWithExperiments>builder()
                .map(property(CampaignWithExperiments.ID, CAMPAIGNS.CID))
                .map(property(CampaignWithExperiments.AB_SEGMENT_RETARGETING_CONDITION_ID,
                        CAMPAIGNS.AB_SEGMENT_RET_COND_ID))
                .map(property(CampaignWithExperiments.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID,
                        CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID))
                .readProperty(
                        CampaignWithExperiments.SECTION_IDS,
                        fromField(RETARGETING_CONDITIONS_SECTIONS.CONDITION_JSON)
                                .by(ExperimentsConverter::sectionIdFromDb))
                .readProperty(
                        CampaignWithExperiments.AB_SEGMENT_GOAL_IDS,
                        fromField(RETARGETING_CONDITIONS_AB_GOALS.CONDITION_JSON)
                                .by(ExperimentsConverter::abSegmentGoalsFromDb))
                .build();
    }
}
