package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.Record;
import org.jooq.Row2;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithFavorite;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.usercampaignsfavorite.repository.UserCampaignsFavoriteRepository;
import ru.yandex.direct.dbschema.ppc.tables.records.UserCampaignsFavoriteRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.dbschema.ppc.Tables.USER_CAMPAIGNS_FAVORITE;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Обработка добавления галки "самая важная" для кампании.
 * Изменение (updateAdditionTables) пока что используется только в API5 Campaigns
 * Добавление (insertToAdditionTables) используется в API5 Campaigns и в новом копировании.
 */
@Component
@ParametersAreNonnullByDefault
public class CampaignWithFavoriteTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithFavorite> {

    private final UserCampaignsFavoriteRepository userCampaignsFavoriteRepository;

    @Autowired
    protected CampaignWithFavoriteTypeSupport(DslContextProvider dslContextProvider,
                                              UserCampaignsFavoriteRepository userCampaignsFavoriteRepository) {
        super(dslContextProvider);
        this.userCampaignsFavoriteRepository = userCampaignsFavoriteRepository;
    }

    @Override
    public Class<CampaignWithFavorite> getTypeClass() {
        return CampaignWithFavorite.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return emptySet();
    }

    @Override
    public <M extends CampaignWithFavorite> void fillFromRecord(M campaign, Record record) {
    }

    @Override
    public void updateAdditionTables(DSLContext context,
                                     RestrictedCampaignsUpdateOperationContainer updateCampaignParametersContainer,
                                     Collection<AppliedChanges<CampaignWithFavorite>> appliedChanges) {
        Map<Long, Set<Long>> rowsToInsert = StreamEx.of(appliedChanges)
                .mapToEntry(this::getUidsToInsertForCampaign)
                .mapKeys(x -> x.getModel().getId())
                .filterValues(x -> !x.isEmpty())
                .toMap();
        insertIntoUserCampaignsFavorite(context, rowsToInsert);

        Map<Long, Set<Long>> rowsToDelete = StreamEx.of(appliedChanges)
                .mapToEntry(this::getUidsToDeleteForCampaign)
                .mapKeys(x -> x.getModel().getId())
                .filterValues(x -> !x.isEmpty())
                .toMap();
        deleteFromUserCampaignsFavorite(context, rowsToDelete);
    }

    @Nonnull
    private Set<Long> getUidsToInsertForCampaign(AppliedChanges<CampaignWithFavorite> appliedChanges) {
        Set<Long> oldUids = appliedChanges.getOldValue(CampaignWithFavorite.FAVORITE_FOR_UIDS);
        Set<Long> newUids = appliedChanges.getNewValue(CampaignWithFavorite.FAVORITE_FOR_UIDS);
        return Sets.difference(nvl(newUids, emptySet()), nvl(oldUids, emptySet()));
    }

    @Nonnull
    private Set<Long> getUidsToDeleteForCampaign(AppliedChanges<CampaignWithFavorite> appliedChanges) {
        Set<Long> oldUids = appliedChanges.getOldValue(CampaignWithFavorite.FAVORITE_FOR_UIDS);
        Set<Long> newUids = appliedChanges.getNewValue(CampaignWithFavorite.FAVORITE_FOR_UIDS);
        return Sets.difference(nvl(oldUids, emptySet()), nvl(newUids, emptySet()));
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                       Collection<CampaignWithFavorite> campaigns) {
        Map<Long, Set<Long>> favoriteCampaignIdToUserIds = StreamEx.of(campaigns)
                .mapToEntry(CampaignWithFavorite::getFavoriteForUids)
                .removeValues(CollectionUtils::isEmpty)
                .mapKeys(CampaignWithFavorite::getId)
                .flatMapValues(Collection::stream)
                .grouping(Collectors.toSet());
        if (favoriteCampaignIdToUserIds.isEmpty()) {
            return;
        }

        insertIntoUserCampaignsFavorite(context, favoriteCampaignIdToUserIds);
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<CampaignWithFavorite> campaigns) {
        List<Long> campaignIds = mapList(campaigns, CampaignWithFavorite::getId);
        Map<Long, Set<Long>> favoriteCampIdsToUserIds =
                userCampaignsFavoriteRepository.getUserIdsByCampaignIds(dslContext, campaignIds);
        campaigns.forEach(campaign -> campaign.setFavoriteForUids(favoriteCampIdsToUserIds.get(campaign.getId())));
    }


    private void insertIntoUserCampaignsFavorite(DSLContext context, Map<Long, Set<Long>> favoriteCampaignIdToUserIds) {
        InsertValuesStep2<UserCampaignsFavoriteRecord, Long, Long> insertStep =
                context.insertInto(USER_CAMPAIGNS_FAVORITE,
                        USER_CAMPAIGNS_FAVORITE.CID,
                        USER_CAMPAIGNS_FAVORITE.UID);
        EntryStream.of(favoriteCampaignIdToUserIds)
                .flatMapValues(Collection::stream)
                .forKeyValue(insertStep::values);
        insertStep.execute();
    }

    private void deleteFromUserCampaignsFavorite(DSLContext context, Map<Long, Set<Long>> favoriteCampaignIdToUserIds) {
        List<Row2<Long, Long>> keys = EntryStream.of(favoriteCampaignIdToUserIds)
                .flatMapValues(Collection::stream)
                .mapKeyValue(DSL::row)
                .toList();
        context.deleteFrom(USER_CAMPAIGNS_FAVORITE)
                .where(DSL.row(USER_CAMPAIGNS_FAVORITE.CID, USER_CAMPAIGNS_FAVORITE.UID).in(keys))
                .execute();
    }
}
