package ru.yandex.direct.core.entity.campaign.repository.type;

import java.time.Duration;
import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithMeaningfulGoalsTypesSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithMeaningfulGoals> {
    public static final int MAPPER_EXPIRE_TIME = 10;

    private final JooqMapper<CampaignWithMeaningfulGoals> mapperWithDoubleValue;
    private final JooqMapper<CampaignWithMeaningfulGoals> mapperWithStringValue;

    private final PpcProperty<Boolean> meaningfulGoalValueSerializeAsStringProperty;


    @Autowired
    public CampaignWithMeaningfulGoalsTypesSupport(DslContextProvider dslContextProvider,
                                                   PpcPropertiesSupport ppcPropertiesSupport) {
        super(dslContextProvider);

        meaningfulGoalValueSerializeAsStringProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.MEANINGFUL_GOAL_VALUE_SERIALIZE_AS_STRING,
                        Duration.ofSeconds(10L));

        mapperWithDoubleValue = createCampaignWithMeaningfulGoalsMapper(false);
        mapperWithStringValue = createCampaignWithMeaningfulGoalsMapper(true);
    }

    @Override
    public Collection<Field<?>> getFields() {
        return mapperWithStringValue.getFieldsToRead();
    }

    @Override
    public <M extends CampaignWithMeaningfulGoals> void fillFromRecord(M campaign, Record record) {
        getMapper().fromDb(record, campaign);
    }

    @Override
    public Class<CampaignWithMeaningfulGoals> getTypeClass() {
        return CampaignWithMeaningfulGoals.class;
    }

    private static JooqMapper<CampaignWithMeaningfulGoals> createCampaignWithMeaningfulGoalsMapper(
            boolean isMeaningfulGoalValueSerializeAsString) {
        return JooqMapperBuilder.<CampaignWithMeaningfulGoals>builder()
                .map(property(CampaignWithMeaningfulGoals.ID, Tables.CAMP_OPTIONS.CID))
                .map(convertibleProperty(CampaignWithMeaningfulGoals.MEANINGFUL_GOALS,
                        Tables.CAMP_OPTIONS.MEANINGFUL_GOALS,
                        CampaignConverter::meaningfulGoalsFromDb,
                        meaningfulGoals ->
                                CampaignConverter.meaningfulGoalsToDb(
                                        meaningfulGoals,
                                        isMeaningfulGoalValueSerializeAsString
                                )))
                .readProperty(CampaignWithMeaningfulGoals.RAW_MEANINGFUL_GOALS,
                        fromField(CAMP_OPTIONS.MEANINGFUL_GOALS))
                .build();
    }

    private JooqMapper<CampaignWithMeaningfulGoals> getMapper() {
        Boolean isMeaningfulGoalValueSerializeAsString =
                meaningfulGoalValueSerializeAsStringProperty.getOrDefault(false);
        return isMeaningfulGoalValueSerializeAsString ? mapperWithStringValue : mapperWithDoubleValue;
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithMeaningfulGoals>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(getMapper(), appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator,
                             CampaignWithMeaningfulGoals campaign) {
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(getMapper(), campaign);
    }

}
