package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurer;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurerSystem;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeasurers;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.measurers.repository.CampMeasurersRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_MEASURERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithMeasurersTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithMeasurers> {
    private final CampMeasurersRepository measurersRepository;
    public final JooqMapperWithSupplier<CampaignMeasurer> mapper = createMapper();

    @Autowired
    public CampaignWithMeasurersTypeSupport(DslContextProvider dslContextProvider,
                                            CampMeasurersRepository measurersRepository) {
        super(dslContextProvider);
        this.measurersRepository = measurersRepository;
    }

    @Override
    public Class<CampaignWithMeasurers> getTypeClass() {
        return CampaignWithMeasurers.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return Collections.emptySet();
    }

    @Override
    public void fillFromRecord(CampaignWithMeasurers campaign, Record record) {
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                       Collection<CampaignWithMeasurers> campaigns) {
        var insertHelper = new InsertHelper<>(context, CAMP_MEASURERS);
        insertHelper.addAll(mapper, campaignMeasurers(campaigns));
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    public void updateAdditionTables(DSLContext context, RestrictedCampaignsUpdateOperationContainer updateParameters,
                                     Collection<AppliedChanges<CampaignWithMeasurers>> appliedChanges) {
        var toDelete = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(CampaignWithMeasurers.MEASURERS))
                .flatMap(ac -> {
                    Long cid = ac.getModel().getId();
                    Set<CampaignMeasurerSystem> curMeasurersTypes = StreamEx.of(ac.getNewValue(CampaignWithMeasurers.MEASURERS))
                            .map(CampaignMeasurer::getMeasurerSystem)
                            .toSet();
                    return StreamEx.of(nvl(ac.getOldValue(CampaignWithMeasurers.MEASURERS), emptyList()))
                            .filter(it -> !curMeasurersTypes.contains(it.getMeasurerSystem()))
                            .map(it -> it.withCid(cid));
                }).toList();
        if (!toDelete.isEmpty()) {
            Condition condition = StreamEx.of(toDelete)
                    .map(m -> CAMP_MEASURERS.CID.in(m.getCid())
                            .and(CAMP_MEASURERS.MEASURER_SYSTEM.eq(CampaignMeasurerSystem.toSource(m.getMeasurerSystem()))))
                    .reduce(Condition::or)
                    .orElse(DSL.falseCondition());
            context.deleteFrom(CAMP_MEASURERS)
                    .where(condition)
                    .execute();
        }
        var toUpdate = campaignMeasurers(StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(CampaignWithMeasurers.MEASURERS))
                .map(AppliedChanges::getModel)
                .toList()
        );
        if (toUpdate.isEmpty()) {
            return;
        }
        var insertHelper = new InsertHelper<>(context, CAMP_MEASURERS);
        insertHelper.addAll(mapper, toUpdate)
                .onDuplicateKeyUpdate()
                .set(CAMP_MEASURERS.PARAMS, MySQLDSL.values(CAMP_MEASURERS.PARAMS))
                .execute();
    }

    private List<CampaignMeasurer> campaignMeasurers(Collection<CampaignWithMeasurers> campaigns) {
        return StreamEx.of(campaigns)
                .flatMap(c -> {
                    List<CampaignMeasurer> mrs = nvl(c.getMeasurers(), emptyList());
                    StreamEx.of(mrs)
                            .forEach(it -> it.setCid(c.getId()));
                    return mrs.stream();
                }).toList();
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<CampaignWithMeasurers> campaigns) {
        List<Long> campaignIds = mapList(campaigns, CampaignWithMeasurers::getId);
        var campIdsToMeasurers = measurersRepository.getMeasurersByCampaignIds(dslContext, campaignIds);
        campaigns.forEach(campaign -> campaign.setMeasurers(nvl(campIdsToMeasurers.get(campaign.getId()), emptyList())));
    }

    private static JooqMapperWithSupplier<CampaignMeasurer> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampaignMeasurer::new)
                .map(property(CampaignMeasurer.CID, CAMP_MEASURERS.CID))
                .map(convertibleProperty(
                        CampaignMeasurer.MEASURER_SYSTEM,
                        CAMP_MEASURERS.MEASURER_SYSTEM,
                        CampaignMeasurerSystem::fromSource,
                        CampaignMeasurerSystem::toSource))
                .map(convertibleProperty(
                        CampaignMeasurer.PARAMS,
                        CAMP_MEASURERS.PARAMS,
                        dbValue -> dbValue == null ? "" : dbValue,
                        javaValue -> "".equals(javaValue) ? null : javaValue))
                .build();
    }
}
