package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.converter.MobileContentConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMobileContent;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsMobileContentRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MOBILE_CONTENT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithMobileContentTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithMobileContent> {
    private static final JooqMapper<CampaignWithMobileContent> MAPPER = createMapper();

    @Autowired
    public CampaignWithMobileContentTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithMobileContent> getTypeClass() {
        return CampaignWithMobileContent.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(CAMPAIGNS_MOBILE_CONTENT, JoinType.LEFT_OUTER_JOIN,
                CAMPAIGNS_MOBILE_CONTENT.CID.eq(CAMPAIGNS.CID)));
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithMobileContent>> appliedChanges) {
    }

    @Override
    public void updateAdditionTables(DSLContext context, RestrictedCampaignsUpdateOperationContainer updateParameters,
                                     Collection<AppliedChanges<CampaignWithMobileContent>> appliedChanges) {
        List<CampaignWithMobileContent> changedCampaignsMobileContent =
                filterAndMapList(appliedChanges,
                        ac -> ac.changed(CampaignWithMobileContent.MOBILE_APP_ID) ||
                                ac.changed(CampaignWithMobileContent.IS_INSTALLED_APP) ||
                                ac.changed(CampaignWithMobileContent.DEVICE_TYPE_TARGETING) ||
                                ac.changed(CampaignWithMobileContent.NETWORK_TARGETING) ||
                                ac.changed(CampaignWithMobileContent.ALTERNATIVE_APP_STORES),
                        AppliedChanges::getModel);

        List<Long> idsToDelete = mapList(changedCampaignsMobileContent, CampaignWithMobileContent::getId);
        deleteCampaignMobileContent(context, idsToDelete);

        addOrUpdateMobileContent(context, changedCampaignsMobileContent);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithMobileContent campaign) {
    }

    @Override
    public void insertToAdditionTables(DSLContext context, RestrictedCampaignsAddOperationContainer addModelParametersContainer,
                                       Collection<CampaignWithMobileContent> models) {
        addOrUpdateMobileContent(context, models);
    }

    @Override
    public void fillFromRecord(CampaignWithMobileContent campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    private static JooqMapper<CampaignWithMobileContent> createMapper() {
        return JooqMapperBuilder.<CampaignWithMobileContent>builder()
                .map(property(CampaignWithMobileContent.ID, CAMPAIGNS_MOBILE_CONTENT.CID))
                .map(property(CampaignWithMobileContent.MOBILE_APP_ID, CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID))
                .map(convertibleProperty(CampaignWithMobileContent.IS_INSTALLED_APP,
                        CAMPAIGNS_MOBILE_CONTENT.IS_INSTALLED_APP,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong))
                .map(convertibleProperty(CampaignWithMobileContent.DEVICE_TYPE_TARGETING,
                        CAMPAIGNS_MOBILE_CONTENT.DEVICE_TYPE_TARGETING,
                        MobileContentConverter::deviceTypeTargetingFromDb,
                        MobileContentConverter::deviceTypeTargetingToDb))
                .map(convertibleProperty(
                        CampaignWithMobileContent.NETWORK_TARGETING, CAMPAIGNS_MOBILE_CONTENT.NETWORK_TARGETING,
                        MobileContentConverter::networkTargetingsFromDb,
                        MobileContentConverter::networkTargetingsToDb))
                .map(convertibleProperty(CampaignWithMobileContent.ALTERNATIVE_APP_STORES,
                        CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES,
                        MobileContentConverter::altAppStoresFromDb,
                        MobileContentConverter::altAppStoresToDb))
                .build();
    }


    private void addOrUpdateMobileContent(DSLContext context, Collection<CampaignWithMobileContent> campaignsToUpdate) {
        InsertHelper<CampaignsMobileContentRecord> insertHelper = new InsertHelper<>(context, CAMPAIGNS_MOBILE_CONTENT);
        insertHelper.addAll(MAPPER, campaignsToUpdate);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID,
                            MySQLDSL.values(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID));
        }
        insertHelper.executeIfRecordsAdded();
    }

    private void deleteCampaignMobileContent(DSLContext context, List<Long> idsToDelete) {
        if (!idsToDelete.isEmpty()) {
            context.deleteFrom(CAMPAIGNS_MOBILE_CONTENT)
                    .where(CAMPAIGNS_MOBILE_CONTENT.CID.in(idsToDelete))
                    .execute();
        }
    }
}
