package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrganization;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignPermalinksRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGN_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithOrganizationTypeSupport extends
        AbstractCampaignRepositoryTypeSupport<CampaignWithOrganization> {

    private static final JooqMapper<CampaignWithOrganization> MAPPER = createMapper();

    @Autowired
    public CampaignWithOrganizationTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithOrganization> getTypeClass() {
        return CampaignWithOrganization.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(CAMPAIGN_PERMALINKS,
                JoinType.LEFT_OUTER_JOIN,
                CAMPAIGN_PERMALINKS.CID.eq(CAMPAIGNS.CID)));
    }

    @Override
    public void fillFromRecord(CampaignWithOrganization campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithOrganization>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void updateAdditionTables(DSLContext context, RestrictedCampaignsUpdateOperationContainer updateParameters,
                                     Collection<AppliedChanges<CampaignWithOrganization>> appliedChanges) {
        List<CampaignWithOrganization> changedCampaigns = filterAndMapList(
                appliedChanges,
                ac -> ac.changed(CampaignWithOrganization.DEFAULT_CHAIN_ID) ||
                        ac.changed(CampaignWithOrganization.DEFAULT_PERMALINK_ID),
                AppliedChanges::getModel);

        List<Long> idsToDelete = filterAndMapList(changedCampaigns,
                campaign -> campaign.getDefaultChainId() == null && campaign.getDefaultPermalinkId() == null,
                CampaignWithOrganization::getId);
        deleteCampaignPermalinks(context, idsToDelete);

        List<CampaignWithOrganization> campaignsToUpdate = filterList(changedCampaigns,
                campaign -> campaign.getDefaultChainId() != null ||
                        campaign.getDefaultPermalinkId() != null
        );
        updateCampaignPermalinks(context, campaignsToUpdate);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithOrganization campaign) {
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(MAPPER, campaign);
    }

    @Override
    public void insertToAdditionTables(DSLContext context, RestrictedCampaignsAddOperationContainer addModelParametersContainer,
                                       Collection<CampaignWithOrganization> models) {
        List<CampaignWithOrganization> campaignsToInsert =
                filterList(models, campaign -> campaign.getDefaultChainId() != null ||
                        campaign.getDefaultPermalinkId() != null);
        updateCampaignPermalinks(context, campaignsToInsert);
    }

    private static JooqMapper<CampaignWithOrganization> createMapper() {
        return JooqMapperBuilder.<CampaignWithOrganization>builder()
                .writeField(CAMPAIGN_PERMALINKS.CID, fromProperty(CampaignWithOrganization.ID))
                .map(convertibleProperty(CampaignWithOrganization.DEFAULT_CHAIN_ID, CAMPAIGN_PERMALINKS.CHAIN_ID,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .map(convertibleProperty(CampaignWithOrganization.DEFAULT_PERMALINK_ID, CAMPAIGN_PERMALINKS.PERMALINK_ID,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .build();
    }

    private void deleteCampaignPermalinks(DSLContext context, List<Long> idsToDelete) {
        if (!idsToDelete.isEmpty()) {
            context.deleteFrom(CAMPAIGN_PERMALINKS).where(CAMPAIGN_PERMALINKS.CID.in(idsToDelete)).execute();
        }
    }

    private void updateCampaignPermalinks(DSLContext context, List<CampaignWithOrganization> campaignsToUpdate) {
        InsertHelper<CampaignPermalinksRecord> insertHelper = new InsertHelper<>(context, CAMPAIGN_PERMALINKS);
        insertHelper.addAll(MAPPER, campaignsToUpdate);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CAMPAIGN_PERMALINKS.PERMALINK_ID, MySQLDSL.values(CAMPAIGN_PERMALINKS.PERMALINK_ID))
                    .set(CAMPAIGN_PERMALINKS.CHAIN_ID, MySQLDSL.values(CAMPAIGN_PERMALINKS.CHAIN_ID));
        }
        insertHelper.executeIfRecordsAdded();
    }
}
