package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithPhone;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGN_PHONES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPhoneTypeSupport
        extends AbstractCampaignRepositoryTypeSupport<CampaignWithPhone> {

    private static final JooqMapper<CampaignWithPhone> MAPPER = createMapper();

    @Autowired
    public CampaignWithPhoneTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithPhone> getTypeClass() {
        return CampaignWithPhone.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(
                CAMPAIGN_PHONES,
                JoinType.LEFT_OUTER_JOIN,
                CAMPAIGN_PHONES.CID.eq(CAMPAIGNS.CID)
        ));
    }

    @Override
    public void fillFromRecord(CampaignWithPhone campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    @Override
    public void updateAdditionTables(DSLContext context, RestrictedCampaignsUpdateOperationContainer updateParameters,
                                     Collection<AppliedChanges<CampaignWithPhone>> appliedChanges) {

        List<Long> idsToDelete = filterAndMapList(appliedChanges,
                ac -> ac.deleted(CampaignWithPhone.DEFAULT_TRACKING_PHONE_ID),
                ac -> ac.getModel().getId());

        if (!idsToDelete.isEmpty()) {
            context.deleteFrom(CAMPAIGN_PHONES).where(CAMPAIGN_PHONES.CID.in(idsToDelete)).execute();
        }

        List<CampaignWithPhone> campaignsToUpdate = filterAndMapList(appliedChanges,
                ac -> ac.changedAndNotDeleted(CampaignWithPhone.DEFAULT_TRACKING_PHONE_ID),
                AppliedChanges::getModel);

        updateCampaignPhones(context, campaignsToUpdate);
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       RestrictedCampaignsAddOperationContainer addModelParametersContainer,
                                       Collection<CampaignWithPhone> models) {
        List<CampaignWithPhone> campaignsToInsert =
                filterList(models, campaign -> campaign.getDefaultTrackingPhoneId() != null);
        updateCampaignPhones(context, campaignsToInsert);
    }

    private static JooqMapper<CampaignWithPhone> createMapper() {
        return JooqMapperBuilder.<CampaignWithPhone>builder()
                .writeField(CAMPAIGN_PHONES.CID, fromProperty(CampaignWithPhone.ID))
                .map(property(CampaignWithPhone.DEFAULT_TRACKING_PHONE_ID,
                        CAMPAIGN_PHONES.CLIENT_PHONE_ID))
                .build();
    }

    private void updateCampaignPhones(DSLContext context, List<CampaignWithPhone> campaignsToUpdate) {
        var insertHelper = new InsertHelper<>(context, CAMPAIGN_PHONES);
        insertHelper.addAll(MAPPER, campaignsToUpdate);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CAMPAIGN_PHONES.CLIENT_PHONE_ID, MySQLDSL.values(CAMPAIGN_PHONES.CLIENT_PHONE_ID));
        }
        insertHelper.executeIfRecordsAdded();
    }
}
