package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.converter.PriceFlightConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsCpmYndxFrontpageRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithPricePackageTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithPricePackage> {

    private static final JooqMapper<CampaignWithPricePackage> MAPPER = createMapper();

    @Autowired
    public CampaignWithPricePackageTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithPricePackage> getTypeClass() {
        return CampaignWithPricePackage.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(CAMPAIGNS_CPM_PRICE, JoinType.LEFT_OUTER_JOIN,
                CAMPAIGNS_CPM_PRICE.CID.eq(CAMPAIGNS.CID)));
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithPricePackage>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMPAIGNS_CPM_PRICE.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithPricePackage campaign) {
        insertHelperAggregator.getOrCreate(CAMPAIGNS_CPM_PRICE).add(MAPPER, campaign);
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                       Collection<CampaignWithPricePackage> campaigns) {
        addFrontpagShowTypesData(context, campaigns);
    }

    /**
     * TODO: временно для прайсовых записываем данные в campaigns_cpm_yndx_frontpage (как для frontpage кампаний)
     * TODO: Удалить в DIRECT-109162
     */
    private void addFrontpagShowTypesData(DSLContext context, Collection<CampaignWithPricePackage> campaigns) {
        if (campaigns.isEmpty()) {
            return;
        }

        var viewTypeToFrontpageType = Map.of(
                ViewType.DESKTOP, "frontpage",
                ViewType.MOBILE, "frontpage_mobile",
                ViewType.NEW_TAB, "browser_new_tab"
        );
        InsertValuesStep2<CampaignsCpmYndxFrontpageRecord, Long, String> insertStep =
                context.insertInto(CAMPAIGNS_CPM_YNDX_FRONTPAGE,
                        CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID,
                        CAMPAIGNS_CPM_YNDX_FRONTPAGE.ALLOWED_FRONTPAGE_TYPES
                );
        campaigns.forEach(campaign -> {
            var frontpageTypes = campaign.getFlightTargetingsSnapshot().getViewTypes().stream()
                    .filter(viewTypeToFrontpageType::containsKey)
                    .map(viewTypeToFrontpageType::get)
                    .collect(Collectors.joining(","));
            if (!frontpageTypes.isEmpty()) {
                insertStep.values(campaign.getId(), frontpageTypes);
            }
        });

        insertStep.execute();
    }

    @Override
    public void fillFromRecord(CampaignWithPricePackage campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    private static JooqMapper<CampaignWithPricePackage> createMapper() {
        return JooqMapperBuilder.<CampaignWithPricePackage>builder()
                .map(property(CampaignWithPricePackage.ID, CAMPAIGNS_CPM_PRICE.CID))
                .map(property(CampaignWithPricePackage.PRICE_PACKAGE_ID, CAMPAIGNS_CPM_PRICE.PACKAGE_ID))
                .map(convertibleProperty(CampaignWithPricePackage.FLIGHT_TARGETINGS_SNAPSHOT,
                        CAMPAIGNS_CPM_PRICE.TARGETINGS_SNAPSHOT,
                        PriceFlightConverter::targetingsSnapshotFromDbFormat,
                        PriceFlightConverter::targetingsSnapshotToDbFormat))
                .map(property(CampaignWithPricePackage.FLIGHT_ORDER_VOLUME, CAMPAIGNS_CPM_PRICE.ORDER_VOLUME))
                .map(convertibleProperty(CampaignWithPricePackage.FLIGHT_STATUS_APPROVE,
                        CAMPAIGNS_CPM_PRICE.STATUS_APPROVE,
                        PriceFlightConverter::statusApproveFromDbFormat,
                        PriceFlightConverter::statusApproveToDbFormat))
                .map(booleanProperty(CampaignWithPricePackage.IS_DRAFT_APPROVE_ALLOWED,
                        CAMPAIGNS_CPM_PRICE.IS_DRAFT_APPROVE_ALLOWED))
                .map(convertibleProperty(CampaignWithPricePackage.FLIGHT_STATUS_CORRECT,
                        CAMPAIGNS_CPM_PRICE.STATUS_CORRECT,
                        PriceFlightConverter::statusCorrectFromDbFormat,
                        PriceFlightConverter::statusCorrectToDbFormat))
                .map(convertibleProperty(CampaignWithPricePackage.FLIGHT_REASON_INCORRECT,
                        CAMPAIGNS_CPM_PRICE.REASON_INCORRECT,
                        PriceFlightConverter::reasonIncorrectFromDbFormat,
                        PriceFlightConverter::reasonIncorrectToDbFormat))
                .readProperty(CampaignWithPricePackage.AUCTION_PRIORITY,
                        fromField(CAMPAIGNS_CPM_PRICE.AUCTION_PRIORITY))
                .readProperty(CampaignWithPricePackage.AUTO_PROLONGATION,
                        fromField(CAMPAIGNS.STRATEGY_DATA).by(PriceFlightConverter::autoProlongationtFromDbFormat))
                .build();
    }

}
