package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithStrategyTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithStrategy> {
    private static final JooqMapper<CampaignWithStrategy> MAPPER = createCampaignWithStrategyMapper();

    @Autowired
    public CampaignWithStrategyTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public void fillFromRecord(CampaignWithStrategy campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    @Override
    public Class<CampaignWithStrategy> getTypeClass() {
        return CampaignWithStrategy.class;
    }

    private static JooqMapper<CampaignWithStrategy> createCampaignWithStrategyMapper() {
        return JooqMapperBuilder.<CampaignWithStrategy>builder()
                .map(property(CommonCampaign.ID, CAMPAIGNS.CID))
                // ppc.campaigns
                .writeField(CAMPAIGNS.AUTOBUDGET,
                        fromProperty(CampaignWithStrategy.STRATEGY)
                                .by(strategy -> CampaignsAutobudget.toSource(strategy.getAutobudget())))
                .writeField(CAMPAIGNS.STRATEGY_NAME,
                        fromProperty(CampaignWithStrategy.STRATEGY)
                                .by(strategy -> StrategyName.toSource(strategy.getStrategyName())))
                .writeField(CAMPAIGNS.PLATFORM,
                        fromProperty(CampaignWithStrategy.STRATEGY)
                                .by(strategy -> CampaignsPlatform.toSource(strategy.getPlatform())))
                .writeField(CAMPAIGNS.STRATEGY_DATA,
                        fromProperty(CampaignWithStrategy.STRATEGY)
                                .by(strategy -> CampaignMappings.strategyDataToDb(strategy.getStrategyData())))
                // ppc.camp_options
                .writeFieldExplicitly(CAMP_OPTIONS.STRATEGY,
                        fromProperty(CampaignWithStrategy.STRATEGY)
                                .by(strategy -> CampOptionsStrategy.toSource(strategy.getStrategy())))
                .readProperty(CampaignWithStrategy.STRATEGY,
                        fromFields(List.of(CAMPAIGNS.AUTOBUDGET, CAMPAIGNS.STRATEGY_NAME,
                                CAMPAIGNS.PLATFORM, CAMPAIGNS.STRATEGY_DATA, CAMP_OPTIONS.STRATEGY))
                                .by(CampaignConverter::strategyFromDb))
                .build();
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithStrategy>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMPAIGNS.CID).processUpdateAll(MAPPER, appliedChanges);
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator,
                             CampaignWithStrategy campaign) {
        insertHelperAggregator.getOrCreate(CAMPAIGNS).add(MAPPER, campaign);
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(MAPPER, campaign);
    }

}
