package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignCanBeUniversal;
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignSupportedOnTouch;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAllowedOnAdultContent;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithEnableCompanyInfo;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithNewIosVersion;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalAloneTrafaret;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalHasTurboSmarts;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalMeaningfulGoalsValuesFromMetrika;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalRequireFiltrationByDontShowDomains;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalTitleSubstitution;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrderPhraseLengthPrecedence;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSimplifiedStrategyView;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSkadNetwork;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTurboApp;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsSendaccnews;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperties;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromPropertyToField;

@Component
@ParametersAreNonnullByDefault
public class CommonCampaignTypeSupport extends AbstractCampaignRepositoryTypeSupport<CommonCampaign> {
    private static final JooqMapper<CommonCampaign> MAPPER = createCommonCampaignMapper();

    private final CampaignRepository campaignRepository;

    @Autowired
    protected CommonCampaignTypeSupport(DslContextProvider dslContextProvider,
                                        CampaignRepository campaignRepository) {
        super(dslContextProvider);
        this.campaignRepository = campaignRepository;
    }

    @Override
    public Class<CommonCampaign> getTypeClass() {
        return CommonCampaign.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(
                CAMP_OPTIONS, JoinType.JOIN, CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID)
        ));
    }

    @Override
    public <M extends CommonCampaign> void fillFromRecord(M campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    @SuppressWarnings("java:S3252")
    private static JooqMapper<CommonCampaign> createCommonCampaignMapper() {
        return JooqMapperBuilder.<CommonCampaign>builder()
                .map(property(CommonCampaign.ID, CAMPAIGNS.CID))
                .map(property(CommonCampaign.UID, CAMPAIGNS.UID))
                .map(property(CommonCampaign.AGENCY_UID, CAMPAIGNS.AGENCY_UID))
                .map(property(CommonCampaign.MANAGER_UID, CAMPAIGNS.MANAGER_UID))
                .map(property(CommonCampaign.AGENCY_ID, CAMPAIGNS.AGENCY_ID))
                .map(property(CommonCampaign.ORDER_ID, CAMPAIGNS.ORDER_ID))
                .map(property(CommonCampaign.PRODUCT_ID, CAMPAIGNS.PRODUCT_ID))

                .map(convertibleProperty(CommonCampaign.CURRENCY, CAMPAIGNS.CURRENCY,
                        CampaignMappings::currencyCodeFromDb, CampaignMappings::currencyCodeToDb))
                .map(property(CommonCampaign.SUM_TO_PAY, CAMPAIGNS.SUM_TO_PAY))
                .map(property(CommonCampaign.SUM, CAMPAIGNS.SUM))
                .map(property(CommonCampaign.SUM_SPENT, CAMPAIGNS.SUM_SPENT))
                .map(property(CommonCampaign.SUM_LAST, CAMPAIGNS.SUM_LAST))
                // поле без записи в базу, чтобы не испортить данные получаемые от баланса
                // обновляем значение только в NotifyOrder
                .readProperty(CommonCampaign.PAID_BY_CERTIFICATE,
                        fromField(CAMPAIGNS.PAID_BY_CERTIFICATE).by(CampaignMappings::paidByCertificateFromDb))
                .map(property(CommonCampaign.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
                .map(property(CommonCampaign.WALLET_ID, CAMPAIGNS.WALLET_CID))
                .map(property(CommonCampaign.NAME, CAMPAIGNS.NAME))
                .map(property(CommonCampaign.START_DATE, CAMPAIGNS.START_TIME))
                .readProperty(CommonCampaign.END_DATE, fromField(CAMPAIGNS.FINISH_TIME))
                .writeField(CAMPAIGNS.FINISH_TIME, fromPropertyToField(CommonCampaign.END_DATE)
                        .by(RepositoryUtils::zeroableDateToDb))
                .map(convertibleProperty(CommonCampaign.DISABLED_IPS, CAMPAIGNS.DISABLED_IPS,
                        CampaignMappings::disabledIpsFromDbToList, CampaignMappings::disabledIpsToDb))
                .map(convertibleProperty(CommonCampaign.TYPE, CAMPAIGNS.TYPE, CampaignType::fromSource,
                        CampaignType::toSource))
                .writeField(CAMPAIGNS.OPTS, fromProperties(CommonCampaign.class,
                        List.of(CampaignWithOptionalTitleSubstitution.HAS_TITLE_SUBSTITUTION,
                                CommonCampaign.HAS_EXTENDED_GEO_TARGETING,
                                CampaignWithEnableCompanyInfo.ENABLE_COMPANY_INFO,
                                CommonCampaign.ENABLE_CPC_HOLD,
                                CampaignWithOptionalAloneTrafaret.IS_ALONE_TRAFARET_ALLOWED,
                                CampaignWithOptionalHasTurboSmarts.HAS_TURBO_SMARTS,
                                CampaignWithTurboApp.HAS_TURBO_APP,
                                CampaignSupportedOnTouch.IS_TOUCH,
                                CommonCampaign.IS_VIRTUAL,
                                CampaignWithSimplifiedStrategyView.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED,
                                CampaignWithOptionalRequireFiltrationByDontShowDomains.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS,
                                CampaignCanBeUniversal.IS_UNIVERSAL,
                                CampaignWithOrderPhraseLengthPrecedence.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED,
                                CampaignWithOptionalMeaningfulGoalsValuesFromMetrika.IS_MEANINGFUL_GOALS_VALUES_FROM_METRIKA_ENABLED,
                                CampaignWithNewIosVersion.IS_NEW_IOS_VERSION_ENABLED,
                                CampaignWithSkadNetwork.IS_SKAD_NETWORK_ENABLED,
                                CampaignWithAllowedOnAdultContent.IS_ALLOWED_ON_ADULT_CONTENT,
                                CampaignWithBrandLift.IS_BRAND_LIFT_HIDDEN,
                                CommonCampaign.IS_S2S_TRACKING_ENABLED,
                                CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED,
                                CommonCampaign.USE_CURRENT_REGION,
                                CommonCampaign.USE_REGULAR_REGION,
                                CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED,
                                CommonCampaign.IS_WW_MANAGED_ORDER))
                        .by(CampaignConverter::campaignOptsToDb))
                // TODO DIRECT-106999 перенести считывание opts-флагов в отдельные TypeSupport-ы
                .readProperty(CommonCampaign.HAS_TITLE_SUBSTITUTION,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasTitleSubstituteFromDb))
                .readProperty(CommonCampaign.HAS_EXTENDED_GEO_TARGETING,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasExtendedGeoTargetingFromDb))
                .readProperty(CommonCampaign.USE_CURRENT_REGION,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::useCurrentRegionFromDb))
                .readProperty(CommonCampaign.USE_REGULAR_REGION,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::useRegularRegionFromDb))
                .readProperty(CommonCampaign.IS_SKAD_NETWORK_ENABLED,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isSkadNetworkEnabled))
                .readProperty(CommonCampaign.ENABLE_COMPANY_INFO,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::enableCompanyInfoFromDb))
                .readProperty(CommonCampaign.ENABLE_CPC_HOLD,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasEnableCpcHoldFromDb))
                .readProperty(CommonCampaign.HAS_TURBO_APP,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasTurboAppFromDb))
                .readProperty(CommonCampaign.IS_VIRTUAL,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isVirtualFromDb))
                .readProperty(CommonCampaign.IS_UNIVERSAL,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::isUniversalFromDb))

                .map(convertibleProperty(CommonCampaign.SMS_TIME, CAMP_OPTIONS.SMS_TIME,
                        CampaignMappings::smsTimeFromDb,
                        CampaignMappings::smsTimeToDb))
                .map(convertibleProperty(CommonCampaign.SMS_FLAGS, CAMP_OPTIONS.SMS_FLAGS,
                        CampaignMappings::smsFlagsFromDb,
                        CampaignMappings::smsFlagsToDb))
                .map(property(CommonCampaign.EMAIL, CAMP_OPTIONS.EMAIL))
                .map(property(CommonCampaign.FIO, CAMP_OPTIONS.FIO))
                .map(integerProperty(CommonCampaign.WARNING_BALANCE, CAMP_OPTIONS.MONEY_WARNING_VALUE))
                .map(booleanProperty(CommonCampaign.ENABLE_SEND_ACCOUNT_NEWS, CAMP_OPTIONS.SEND_ACC_NEWS,
                        CampOptionsSendaccnews.class))
                .writeField(CAMP_OPTIONS.EMAIL_NOTIFICATIONS, fromProperties(CommonCampaign.class,
                        List.of(CommonCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT))
                        .by(CampaignConverter::emailNotificationsToDb))
                .readProperty(CommonCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT,
                        fromField(CAMP_OPTIONS.EMAIL_NOTIFICATIONS).by(CampaignConverter::hasPausedByDayBudgetFromDb))
                .map(convertibleProperty(CommonCampaign.STATUS_ARCHIVED, CAMPAIGNS.ARCHIVED,
                        CampaignMappings::archivedFromDb, CampaignMappings::archivedToDb))
                .map(convertibleProperty(CommonCampaign.STATUS_MODERATE, CAMPAIGNS.STATUS_MODERATE,
                        CampaignStatusModerate::fromSource, CampaignStatusModerate::toSource))
                .map(convertibleProperty(CommonCampaign.STATUS_POST_MODERATE, CAMP_OPTIONS.STATUS_POST_MODERATE,
                        CampaignStatusPostmoderate::fromSource, CampaignStatusPostmoderate::toSource))
                .map(convertibleProperty(CommonCampaign.STATUS_BS_SYNCED, CAMPAIGNS.STATUS_BS_SYNCED,
                        CampaignStatusBsSynced::fromSource, CampaignStatusBsSynced::toSource))
                .map(convertibleProperty(CommonCampaign.STATUS_ACTIVE, CAMPAIGNS.STATUS_ACTIVE,
                        CampaignConverter::campaignStatusActiveFromDb, CampaignConverter::campaignStatusActiveToDb))
                .map(convertibleProperty(CommonCampaign.STATUS_SHOW, CAMPAIGNS.STATUS_SHOW,
                        CampaignConverter::campaignStatusShowFromDb, CampaignConverter::campaignStatusShowToDb))
                .map(convertibleProperty(CommonCampaign.STATUS_EMPTY, CAMPAIGNS.STATUS_EMPTY,
                        CampaignConverter::campaignStatusEmptyFromDb, CampaignConverter::campaignStatusEmptyToDb))
                .map(property(CommonCampaign.AUTOBUDGET_FORECAST_DATE, CAMPAIGNS.AUTOBUDGET_FORECAST_DATE))
                .readProperty(CommonCampaign.LAST_CHANGE, fromField(CAMPAIGNS.LAST_CHANGE))
                .readProperty(CommonCampaign.CREATE_TIME, fromField(CAMP_OPTIONS.CREATE_TIME))
                .writeField(CAMPAIGNS.LAST_CHANGE, fromPropertyToField(CommonCampaign.LAST_CHANGE)
                        .by(RepositoryUtils::setCurrentLocalDateTimeIfShould))
                .map(convertibleProperty(CommonCampaign.COPIED_FROM, CAMPAIGNS.COPIED_FROM,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .map(convertibleProperty(CommonCampaign.BANNERS_PER_PAGE, CAMP_OPTIONS.BANNERS_PER_PAGE,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .map(convertibleProperty(CommonCampaign.SOURCE, CAMPAIGNS.SOURCE, CampaignSource::fromSource,
                        CampaignConverter::campaignSourceOrDefaultToDb))
                .map(convertibleProperty(CommonCampaign.METATYPE, CAMPAIGNS.METATYPE, CampaignMetatype::fromSource,
                        CampaignConverter::campaignMetatypeOrDefaultToDb))
                .build();
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CommonCampaign>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMPAIGNS.CID).processUpdateAll(MAPPER, appliedChanges);
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CommonCampaign campaign) {
        insertHelperAggregator.getOrCreate(CAMPAIGNS).add(MAPPER, campaign);
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(MAPPER, campaign);
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                       Collection<CommonCampaign> campaigns) {
        campaignRepository.addCampaignsForServicing(context, campaigns);
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<CommonCampaign> campaigns) {
        Set<Long> campaignIdsForServicing = campaignRepository.getCampaignIdsForServicing(dslContext, campaigns);
        campaigns.forEach(c -> {
            c.setIsServiceRequested(campaignIdsForServicing.contains(c.getId()));
        });
    }
}
