package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.WALLET_CAMPAIGNS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromPropertyToField;

@Component
@ParametersAreNonnullByDefault
public class WalletCampaignTypeSupport extends AbstractCampaignRepositoryTypeSupport<WalletTypedCampaign> {
    private static final JooqMapper<WalletTypedCampaign> MAPPER = createWalletCampaignMapper();

    @Autowired
    public WalletCampaignTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<WalletTypedCampaign> getTypeClass() {
        return WalletTypedCampaign.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(WALLET_CAMPAIGNS, JoinType.LEFT_OUTER_JOIN,
                WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.CID)));
    }

    @Override
    public <M extends WalletTypedCampaign> void fillFromRecord(M campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    private static JooqMapper<WalletTypedCampaign> createWalletCampaignMapper() {
        return JooqMapperBuilder.<WalletTypedCampaign>builder()
                .map(property(WalletTypedCampaign.ID, WALLET_CAMPAIGNS.WALLET_CID))
                .readProperty(WalletTypedCampaign.WALLET_ON_OFF_TIME, fromField(WALLET_CAMPAIGNS.ONOFF_DATE))
                .writeField(WALLET_CAMPAIGNS.ONOFF_DATE, fromPropertyToField(WalletTypedCampaign.WALLET_ON_OFF_TIME)
                        .by(RepositoryUtils::setCurrentLocalDateTimeIfShould))
                .readProperty(WalletTypedCampaign.WALLET_INFO_LAST_CHANGE, fromField(WALLET_CAMPAIGNS.LAST_CHANGE))
                .writeField(WALLET_CAMPAIGNS.LAST_CHANGE, fromPropertyToField(WalletTypedCampaign.LAST_CHANGE)
                        .by(RepositoryUtils::setCurrentLocalDateTimeIfShould))
                .map(property(WalletTypedCampaign.TOTAL_SUM, WALLET_CAMPAIGNS.TOTAL_SUM))
                .map(convertibleProperty(WalletTypedCampaign.IS_AUTO_PAY_ON, WALLET_CAMPAIGNS.AUTOPAY_MODE,
                        CampaignMappings::isAutoPayFromDb, CampaignMappings::isAutoPayToDb))
                .map(property(WalletTypedCampaign.TOTAL_CHIPS_COST, WALLET_CAMPAIGNS.TOTAL_CHIPS_COST))
                .map(convertibleProperty(WalletTypedCampaign.IS_SUM_AGGREGATED, WALLET_CAMPAIGNS.IS_SUM_AGGREGATED,
                        CampaignConverter::isSumAggregatedFromDb, CampaignConverter::isSumAggregatedToDb))
                .build();
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, WalletTypedCampaign campaign) {
        insertHelperAggregator.getOrCreate(WALLET_CAMPAIGNS).add(MAPPER, campaign);
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<WalletTypedCampaign>> appliedChanges) {
        updateHelperAggregator.getOrCreate(WALLET_CAMPAIGNS.WALLET_CID).processUpdateAll(MAPPER, appliedChanges);
    }
}
