package ru.yandex.direct.core.entity.campaign.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.advq.query.ast.WordKind;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.adgroup.service.UpdateAdGroupMinusKeywordsOperation;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignNewMinusKeywords;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.UpdateCampaignValidationService;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.KeywordProcessingUtils;
import ru.yandex.direct.core.entity.keyword.processing.KeywordStopwordsFixer;
import ru.yandex.direct.core.entity.keyword.repository.KeywordCacheRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.phrase.NormalizedPhrasePair;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.utils.FunctionalUtils;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.model.ModelChanges.propertyModifier;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция добавления минус-фраз в кампанию.
 * <p>
 * Структура добавления аналогична {@link UpdateAdGroupMinusKeywordsOperation}
 */
public class AppendCampaignMinusKeywordsOperation extends CampaignsUpdateOperation {
    private final CampaignRepository campaignRepository;
    private final KeywordCacheRepository keywordCacheRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final KeywordStopwordsFixer keywordStopwordsFixer;
    private final KeywordNormalizer keywordNormalizer;
    private int shard;

    /**
     * Добавляемые (а не суммарные) минус-фразы
     * в нормальной форме для каждой обновляемой кампании.
     */
    private Map<Long, List<String>> normalMinusKeywords;

    /**
     * Реально добавленное число фраз для каждой обновляемой кампании.
     */
    private Map<Long, Integer> actualAddedMinusKeywordsCount;

    /**
     * Суммарная длина всех минус-фраз после добавления
     * для каждой обновляемой кампании (за вычетом спец-символов).
     */
    private Map<Long, Integer> sumMinusKeywordsLength;


    /**
     * Исходные значения фраз для кампаний
     */
    private Map<Long, List<NormalizedPhrasePair>> origPhrases;

    /**
     * Отображение 'исходная фраза' => 'нормализованная фраза'
     */
    private Map<String, String> origPhraseToNormPhrase;

    public AppendCampaignMinusKeywordsOperation(
            List<CampaignNewMinusKeywords> campaignNewMinusKeywords,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            KeywordCacheRepository keywordCacheRepository,
            UpdateCampaignValidationService updateCampaignValidationService,
            MinusKeywordPreparingTool minusKeywordPreparingTool,
            KeywordStopwordsFixer keywordStopwordsFixer,
            KeywordNormalizer keywordNormalizer,
            long operatorUid, ClientId clientId, int shard) {
        super(Applicability.FULL, createModelChanges(campaignNewMinusKeywords), campaignRepository, adGroupRepository,
                bannerCommonRepository, updateCampaignValidationService, minusKeywordPreparingTool,
                MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD, operatorUid, clientId,
                shard);
        this.campaignRepository = campaignRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.keywordCacheRepository = keywordCacheRepository;
        this.keywordStopwordsFixer = keywordStopwordsFixer;
        this.keywordNormalizer = keywordNormalizer;
        this.shard = shard;
    }

    private static List<ModelChanges<Campaign>> createModelChanges(List<CampaignNewMinusKeywords> newMinusKeywords) {
        Function<CampaignNewMinusKeywords, ModelChanges<Campaign>> toModelChanges =
                mk -> ModelChanges.build(mk.getId(), Campaign.class, Campaign.MINUS_KEYWORDS, mk.getMinusKeywords());
        return mapList(newMinusKeywords, toModelChanges);
    }

    /**
     * Возвращает добавляемые минус-фразы в нормальной форме
     * с предварительно зафиксированными стоп-словами
     * для каждой обновляемой кампании.
     * Ключами являются id кампаний.
     * <p>
     * Метод актуален после выполнения {@link Operation#prepare()},
     * если валидация списка ModelChanges прошла успешно.
     * В противном случае будет сгенерировано исключение {@code IllegalStateException}.
     */
    public Map<Long, List<String>> getNormalMinusKeywords() {
        checkState(normalMinusKeywords != null,
                "prepare() is not called or validation of ModelChanges failed");
        return normalMinusKeywords;
    }

    /**
     * Для всех обновляемых кампаний возвращает количество минус-фраз,
     * которое будет реально добавлено в кампанию, оно равно размеру
     * добавляемых минус-фраз минус количество удаленных дубликатов.
     * Ключами являются id кампаний.
     * <p>
     * Метод актуален после выполнения {@link Operation#prepare()},
     * если валидация списка ModelChanges прошла успешно.
     * В противном случае будет сгенерировано исключение {@code IllegalStateException}.
     */
    public Map<Long, Integer> getActualAddedMinusKeywordsCount() {
        checkState(actualAddedMinusKeywordsCount != null,
                "prepare() is not called or validation of AppliedChanges failed");
        return actualAddedMinusKeywordsCount;
    }

    /**
     * Для всех обновляемых кампаний возвращает длину минус-фраз (без учета спец. символов),
     * которые будут выставлены в кампаниях в базе после обновления (имеющиеся + новые - дубликаты).
     * Ключами являются id кампаний.
     * <p>
     * Метод актуален после выполнения {@link Operation#prepare()},
     * если валидация списка AppliedChanges прошла успешно.
     * В противном случае будет сгенерировано исключение {@code IllegalStateException}.
     */
    public Map<Long, Integer> getSumMinusKeywordsLength() {
        checkState(sumMinusKeywordsLength != null,
                "prepare() is not called or validation of AppliedChanges failed");
        return sumMinusKeywordsLength;
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<Campaign> modelChangesValidatedStep) {
        Collection<ModelChanges<Campaign>> validModelChanges = modelChangesValidatedStep.getValidModelChanges();

        preprocessNewMinusKeywords(validModelChanges);
        rememberNewMinusKeywordsInNormalForm(validModelChanges);
        appendNewMinusKeywordsToExisting(validModelChanges);
        removeDuplicatesAndSortMinusKeywords(validModelChanges);
    }

    @Override
    protected void beforeExecution(ExecutionStep<Campaign> executionStep) {
        super.beforeExecution(executionStep);
        Collection<AppliedChanges<Campaign>> appliedChangesCollection = executionStep.getAppliedChangesForExecution();
        rememberMinusKeywordsLength(appliedChangesCollection);
    }

    @Override
    protected void afterExecution(ExecutionStep<Campaign> executionStep) {
        super.afterExecution(executionStep);
        saveAddedMinusKeywordsToCache();
    }

    private void preprocessNewMinusKeywords(Collection<ModelChanges<Campaign>> validModelChanges) {
        Function<ModelChanges<Campaign>, List<NormalizedPhrasePair>> createNormalizedPhrasePair =
                modelChanges ->
                        FunctionalUtils.mapList(modelChanges.getChangedProp(Campaign.MINUS_KEYWORDS),
                                phrase -> new NormalizedPhrasePair(phrase, null));
        origPhrases = StreamEx.of(validModelChanges.stream())
                .toMap(ModelChanges::getId, createNormalizedPhrasePair);

        validModelChanges.forEach(propertyModifier(Campaign.MINUS_KEYWORDS, minusKeywordPreparingTool::preprocess));
    }

    /**
     * Запоминает добавляемые минус-фразы, приведя их к нормальной форме после фиксации стоп-слов.
     * Вызывается в тот момент, когда в ModelChanges находятся еще только добавляемые минус-фразы,
     * т.е. до того, когда будут извлечены имеющиеся минус-фразы и объединены с добавляемыми.
     *
     * @param validModelChanges список валидных {@link ModelChanges}.
     */
    private void rememberNewMinusKeywordsInNormalForm(Collection<ModelChanges<Campaign>> validModelChanges) {
        Function<ModelChanges<Campaign>, List<String>> getUnquotedNormalizedMinusKeywords = modelChanges -> {
            List<String> minusKeywords = modelChanges.getChangedProp(Campaign.MINUS_KEYWORDS);
            minusKeywords = keywordStopwordsFixer.unquoteAndFixStopwords(minusKeywords, WordKind.PLUS);
            List<String> normalizedWords = keywordNormalizer.normalizeKeywords(minusKeywords);

            StreamEx.of(origPhrases.get(modelChanges.getId()))
                    .zipWith(normalizedWords.stream())
                    .forEach(el -> el.getKey().setNormalizedPhrase(el.getValue()));

            return normalizedWords;
        };
        normalMinusKeywords = StreamEx.of(validModelChanges)
                .mapToEntry(getUnquotedNormalizedMinusKeywords)
                .mapKeys(ModelChanges::getId)
                .toMap();

        origPhraseToNormPhrase = StreamEx.ofValues(origPhrases)
                .flatMapToEntry(
                        phraseInfoList -> StreamEx.of(phraseInfoList)
                                .toMap(NormalizedPhrasePair::getOrigPhrase, NormalizedPhrasePair::getNormalizedPhrase, (el1, el2) -> el1)
                )
                .toMap((el1, el2) -> el1);
    }

    private void appendNewMinusKeywordsToExisting(Collection<ModelChanges<Campaign>> validModelChanges) {
        List<Long> campaignIds = mapList(validModelChanges, ModelChanges::getId);
        Map<Long, List<String>> campaignIdToExistingMinusKeywordsMap =
                campaignRepository.getMinusKeywordsByCampaignIds(shard, campaignIds);

        BiFunction<Long, List<String>, List<String>> appendMinusKeywords = (id, newMinusKeywords) -> {
            List<String> existingMinusKeywords =
                    campaignIdToExistingMinusKeywordsMap.computeIfAbsent(id, id1 -> new ArrayList<>());
            List<String> updatedMinusKeywords = new ArrayList<>(existingMinusKeywords);
            updatedMinusKeywords.addAll(newMinusKeywords);
            return updatedMinusKeywords;
        };

        validModelChanges.forEach(propertyModifier(Campaign.MINUS_KEYWORDS, appendMinusKeywords));
    }

    /**
     * Удаляет дубликаты из полного списка минус-фраз (имеющиеся + добавляемые), затем сортирует его.
     * <p>
     * Помимо удаления дубликатов и сортировки, вычисляет количество реально добавляемых минус-фраз
     * для каждой кампании. Эту информацию затем можно получить с помощью метода
     * {@link #getActualAddedMinusKeywordsCount}.
     * <p>
     * К моменту вызова в ModelChanges должны лежать полные списки минус-фраз.
     */
    private void removeDuplicatesAndSortMinusKeywords(Collection<ModelChanges<Campaign>> validModelChanges) {
        Map<Long, Integer> minusKeywordsCountBeforeDeduplicating = StreamEx.of(validModelChanges)
                .mapToEntry(mc -> mc.getChangedProp(Campaign.MINUS_KEYWORDS).size())
                .mapKeys(ModelChanges::getId)
                .toMap();

        validModelChanges
                .forEach(propertyModifier(Campaign.MINUS_KEYWORDS, minusKeywordPreparingTool::removeDuplicatesAndSort));

        Function<ModelChanges<Campaign>, Integer> calcActualAddedMinusKeywordsCount = modelChanges -> {
            long id = modelChanges.getId();
            int newMinusKeywordsCount = normalMinusKeywords.get(id).size();
            int allBeforeDeduplicatingCount = minusKeywordsCountBeforeDeduplicating.get(id);
            int allAfterDeduplicatingCount = modelChanges.getChangedProp(Campaign.MINUS_KEYWORDS).size();
            int removedDuplicatesCount = allBeforeDeduplicatingCount - allAfterDeduplicatingCount;
            int actualNewMinusKeywordsCount = newMinusKeywordsCount - removedDuplicatesCount;
            return actualNewMinusKeywordsCount > 0 ? actualNewMinusKeywordsCount : 0;
        };

        actualAddedMinusKeywordsCount = StreamEx.of(validModelChanges)
                .mapToEntry(calcActualAddedMinusKeywordsCount)
                .mapKeys(ModelChanges::getId)
                .toMap();
    }

    /**
     * Запоминает суммарную длину минус-фраз (без учета спец. символов),
     * которые будут выставлены в кампаниях в базе после обновления.
     */
    private void rememberMinusKeywordsLength(Collection<AppliedChanges<Campaign>> validAppliedChanges) {
        Function<Campaign, Integer> calcLength = campaign ->
                KeywordProcessingUtils.getLengthWithoutSpecSymbolsAndSpaces(campaign.getMinusKeywords());
        sumMinusKeywordsLength = StreamEx.of(validAppliedChanges)
                .map(AppliedChanges::getModel)
                .mapToEntry(Campaign::getId, calcLength)
                .toMap();
    }

    private void saveAddedMinusKeywordsToCache() {
        keywordCacheRepository.addCampaignMinusKeywords(shard, normalMinusKeywords);
    }

    public Map<String, String> getOrigPhraseToNormPhrase() {
        return origPhraseToNormPhrase;
    }
}
