package ru.yandex.direct.core.entity.campaign.service;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

@ParametersAreNonnullByDefault
@Service
public class BaseCampaignService implements EntityService<BaseCampaign, Long> {
    private ShardHelper shardHelper;
    private CampaignTypedRepository campaignRepository;
    private UserRepository userRepository;
    private CampaignOperationService operationService;

    public BaseCampaignService(ShardHelper shardHelper,
                               CampaignTypedRepository campaignRepository,
                               UserRepository userRepository,
                               CampaignOperationService operationService) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.userRepository = userRepository;
        this.operationService = operationService;
    }

    @Override
    public List<BaseCampaign> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        return List.copyOf(campaignRepository.getTypedCampaigns(
                shard, campaignRepository.getClientCampaignIds(shard, clientId, ids)));
    }

    /**
     * Сервис добавления для копирования
     */
    @Override
    public MassResult add(
            ClientId clientId, Long operatorUid, List<BaseCampaign> entities, Applicability applicability) {
        if (applicability == Applicability.FULL) {
            throw new UnsupportedOperationException("New campaign service does not support full applicability.");
        }

        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long uid = userRepository.getChiefUidByClientId(shard, clientId.asLong());
        CampaignOptions options = new CampaignOptions.Builder().build();
        return operationService
                .createRestrictedCampaignAddOperation(entities, operatorUid, UidAndClientId.of(uid, clientId), options)
                .prepareAndApply();
    }

    /**
     * Сервис добавления для копирования
     */
    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer, List<BaseCampaign> entities,
                                 Applicability applicability) {
        if (applicability == Applicability.FULL) {
            throw new UnsupportedOperationException("New campaign service does not support full applicability.");
        }

        Long chiefUid = copyContainer.getClientTo().getChiefUid();
        CampaignOptions options = new CampaignOptions.Builder()
                .withCopy(true)
                .withCopyInOneClient(!copyContainer.isCopyingBetweenClients())
                .withAllowNullDeviceTypeTargeting(true)
                .withAllowNullNetworkTargeting(true)
                .withAllowZeroMobileAppId(true)
                .build();
        return operationService
                .createRestrictedCampaignAddOperation(
                        entities,
                        copyContainer.getOperatorUid(),
                        UidAndClientId.of(chiefUid, copyContainer.getClientIdTo()),
                        options)
                .prepareAndApply();
    }
}
