package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueObjType;
import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueuePriority;
import ru.yandex.direct.core.entity.balance.model.BalanceNotificationInfo;
import ru.yandex.direct.core.entity.balance.service.BalanceInfoQueueService;
import ru.yandex.direct.core.entity.balanceaggrmigration.container.BalanceAggregateMigrationParams;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.BillingAggregateCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.time.LocalDate.now;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.common.util.RepositoryUtils.NOW_PLACEHOLDER;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CAMPAIGN_WARNING_BALANCE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_SMS_TIME_INTERVAL;
import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.BALANCE_AGGREGATE_MIGRATION;
import static ru.yandex.direct.core.entity.product.ProductConstants.AGGREGATE_NAME_BY_PRODUCT_TYPE;
import static ru.yandex.direct.core.entity.product.ProductConstants.BILLING_AGGREGATES_AUTO_CREATE_TYPES;
import static ru.yandex.direct.core.entity.product.ProductConstants.PRODUCT_TYPE_BY_CAMPAIGN_TYPES;
import static ru.yandex.direct.core.entity.product.ProductConstants.getSpecialProductTypesByCampaignType;
import static ru.yandex.direct.feature.FeatureName.AGGREGATED_SUMS_FOR_OLD_CLIENTS;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class BillingAggregateService {
    private static final Logger logger = LoggerFactory.getLogger(BillingAggregateService.class);
    //Perl
    //https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/Model/BillingAggregate/Manager.pm#L46.
    private static final String NEW_BILLING_AGGREGATE_LOCK_PREFIX = "NEW_BILLING_AGGREGATE";

    //Словарь uid, которым запрещено создавать биллинговые агрегаты.
    //Нужен для автотеста на неотправку биллинговых агрегатов в БК
    static final Set<Long> UIDS_TO_DISABLE_BILLING_AGGREGATES = Set.of(
            783297797L //at-transport-archived-ba
    );

    public static final Set<CampaignType> CAMPAIGN_TYPES_WITH_BILLING_AGGREGATES =
            Sets.difference(CampaignTypeKinds.ALL, CampaignTypeKinds.WITHOUT_BILLING_AGGREGATES);

    private final BalanceInfoQueueService balanceInfoQueueService;
    private final BsResyncService bsResyncService;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;
    private final ClientService clientService;
    private final DbQueueRepository dbQueueRepository;
    private final DslContextProvider ppcDslContextProvider;
    private final FeatureService featureService;
    private final ProductService productService;
    private final ShardHelper shardHelper;

    @Autowired
    public BillingAggregateService(BalanceInfoQueueService balanceInfoQueueService, BsResyncService bsResyncService,
                                   CampaignTypedRepository campaignTypedRepository,
                                   CampaignModifyRepository campaignModifyRepository, ClientService clientService,
                                   DbQueueRepository dbQueueRepository, DslContextProvider ppcDslContextProvider,
                                   FeatureService featureService,
                                   ProductService productService, ShardHelper shardHelper) {
        this.balanceInfoQueueService = balanceInfoQueueService;
        this.bsResyncService = bsResyncService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
        this.clientService = clientService;
        this.dbQueueRepository = dbQueueRepository;
        this.featureService = featureService;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.productService = productService;
        this.shardHelper = shardHelper;
    }

    /**
     * Проверяем, доступно ли автозаведение агрегатов для клиента
     */
    boolean autoCreateEnabled(Client client) {
        //Если клиент фишечный, то выключено
        if (client.getWorkCurrency() == CurrencyCode.YND_FIXED) {
            return false;
        }

        //Если клиент агентский и на агентство стоит запрет, то выключено
        if (isValidId(client.getAgencyClientId()) && featureService.isEnabledForClientId(
                ClientId.fromLong(client.getAgencyClientId()), FeatureName.DISABLE_BILLING_AGGREGATES)) {
            return false;
        }

        //Если на клиента стоит запрет, то выключено
        if (featureService.isEnabledForClientId(ClientId.fromLong(client.getId()),
                FeatureName.DISABLE_BILLING_AGGREGATES)) {
            return false;
        }

        //Если для uid'а стоит запрет, то выключено
        if (UIDS_TO_DISABLE_BILLING_AGGREGATES.contains(client.getChiefUid())) {
            return false;
        }
        return true;
    }

    /**
     * Смотрим для каких продуктов у клиента еще нет агрегатов
     */
    Set<Product> getMissingProducts(Client client, Set<CampaignType> campaignTypes) {
        ClientId clientId = ClientId.fromLong(client.getClientId());

        Set<Product> neededProductByCampaignType = productService.calculateProductsForCampaigns(campaignTypes,
                client.getWorkCurrency(), client.getUsesQuasiCurrency());

        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, ? extends BaseCampaign> billingAggregates =
                campaignTypedRepository.getClientsTypedCampaignsByType(shard, clientId,
                        Set.of(CampaignType.BILLING_AGGREGATE));

        List<Long> existingProductIds =
                StreamEx.of(billingAggregates.values())
                        .select(BillingAggregateCampaign.class)
                        .map(BillingAggregateCampaign::getProductId)
                        .toList();

        return StreamEx.of(neededProductByCampaignType)
                .remove(product -> existingProductIds.contains(product.getId()))
                .filter(product -> BILLING_AGGREGATES_AUTO_CREATE_TYPES.contains(product.getType()))
                .toSet();
    }

    /**
     * Проверяет, есть ли под ОС биллинговые агрегаты тех продуктов, которые могут быть
     * использованы в кампаниях
     * Если нужно (включен ОС, клиент попал под фичу, включено автосоздание для типов продукта),
     * досоздает агрегаты.
     * Если какие-то агрегаты были созданы, кладет кампании, которые могут использовать этот агрегат,
     * в ленивую очередь переотправки в БК.
     */
    public List<Long> createBillingAggregates(ClientId clientId, Long operatorUid, Long chiefUid,
                                              Collection<? extends CampaignWithCampaignType> campaigns,
                                              Long walletId) {
        Client client = checkNotNull(clientService.getClient(clientId));
        checkArgument(walletId > 0);
        int shard = shardHelper.getShardByClientId(clientId);
        WalletTypedCampaign wallet = (WalletTypedCampaign) campaignTypedRepository.getTypedCampaigns(shard,
                List.of(walletId)).stream()
                .findAny()
                .orElseThrow(() -> new IllegalStateException("wallet with id: " + walletId + " was not found"));

        if (autoCreateEnabled(client)) {
            Set<CampaignType> campaignTypes = listToSet(campaigns, CampaignWithCampaignType::getType);
            Set<Product> missingProducts = getMissingProducts(client, campaignTypes);
            if (!missingProducts.isEmpty()) {
                List<BillingAggregateCampaign> toCreate = getNewBillingAggregatesForClient(missingProducts,
                        wallet);
                RestrictedCampaignsAddOperationContainer addCampaignParametersContainer =
                        RestrictedCampaignsAddOperationContainer.create(shard, operatorUid, clientId,
                                wallet.getUid(), chiefUid);

                List<Long> createdAggregateIds = StreamEx.of(toCreate)
                        .map(billingAggregateCampaign ->
                                createBillingAggregate(addCampaignParametersContainer, billingAggregateCampaign,
                                        getLockName(clientId, billingAggregateCampaign.getProductId())))
                        .flatMap(Collection::stream)
                        .toList();

                logger.info("Client has camps with types {} under wallet, creating billing aggregates with product " +
                                "types {}", campaignTypes.stream()
                                .map(CampaignType::name)
                                .collect(joining(", ")),
                        mapSet(missingProducts, Product::getType).stream()
                                .map(ProductType::name)
                                .collect(joining(", "))
                );

                Set<CampaignType> campaignTypesToResync = getCampaignTypesToResync(
                        client.getWorkCurrency(),
                        mapList(missingProducts, Product::getType));

                Map<Long, ? extends BaseCampaign> potentionalCampaignsToResync =
                        campaignTypedRepository.getClientsTypedCampaignsByType(shard, clientId, campaignTypesToResync);

                Set<Long> createdCampaignIds = listToSet(campaigns, CampaignWithCampaignType::getId);
                List<BsResyncItem> bsResyncItems = EntryStream.of(potentionalCampaignsToResync)
                        .mapValues(o -> (CommonCampaign) o)
                        .filterValues(commonCampaign -> commonCampaign.getWalletId().equals(walletId) &&
                                !createdCampaignIds.contains(commonCampaign.getId()))
                        .keys()
                        .map(id -> new BsResyncItem(BsResyncPriority.ON_BILLING_AGGREGATE_ADDED, id))
                        .toList();

                bsResyncService.addObjectsToResync(bsResyncItems);
                logger.info("Resyncing camps with types {}", campaignTypesToResync.stream()
                        .map(CampaignType::name)
                        .collect(joining(", ")));

                boolean hasAggregatedSumsForOldClients = featureService.isEnabledForClientId(clientId,
                        AGGREGATED_SUMS_FOR_OLD_CLIENTS);

                if (!nvl(wallet.getIsSumAggregated(), false) && hasAggregatedSumsForOldClients) {
                    addWalletSumMigrationJob(shard, clientId, operatorUid, walletId);
                }

                return createdAggregateIds;
            }
        }

        return emptyList();
    }

    private static Set<CampaignType> getCampaignTypesToResync(CurrencyCode currencyCode, List<ProductType> affectedProductTypes) {
        var specialProductTypesByCampaignType = getSpecialProductTypesByCampaignType(currencyCode);
        return EntryStream.of(PRODUCT_TYPE_BY_CAMPAIGN_TYPES)
                .filter(entry -> {
                    boolean campaignHasBillingAggregate =
                            CAMPAIGN_TYPES_WITH_BILLING_AGGREGATES.contains(entry.getKey());
                    boolean campaignTypeAffected =
                            campaignHasBillingAggregate && affectedProductTypes.contains(entry.getValue());
                    boolean specialProductNeedsToBeChecked =
                            specialProductTypesByCampaignType.containsKey(entry.getKey());
                    boolean additionalCheckTypeConfirmed =
                            specialProductNeedsToBeChecked && CollectionUtils.intersection(affectedProductTypes,
                                    specialProductTypesByCampaignType.get(entry.getKey())).size() > 0;

                    return campaignTypeAffected || additionalCheckTypeConfirmed;
                })
                .keys()
                .toSet();
    }

    private static List<BillingAggregateCampaign> getNewBillingAggregatesForClient(Collection<Product> products,
                                                                                   WalletTypedCampaign wallet) {

        LocalDate now = now();
        return mapList(products, product -> new BillingAggregateCampaign()
                .withOrderId(0L)
                .withUid(wallet.getUid())
                .withClientId(wallet.getClientId())
                .withAgencyId(wallet.getAgencyId())
                .withAgencyUid(wallet.getAgencyUid())
                .withManagerUid(wallet.getManagerUid())
                .withFio(wallet.getFio())
                .withEmail(wallet.getEmail())
                .withCurrency(wallet.getCurrency())
                .withStatusBsSynced(CampaignStatusBsSynced.NO)
                .withStatusActive(false)
                .withStatusEmpty(false)
                .withStatusModerate(CampaignStatusModerate.YES)
                .withStatusPostModerate(CampaignStatusPostmoderate.ACCEPTED)
                .withStatusShow(true)
                .withStatusArchived(false)
                //Тут в перле был заложен механизм локализации, но реальных переводов нет, в базе все на русском
                //TODO: что если нет имени?
                .withName(AGGREGATE_NAME_BY_PRODUCT_TYPE.get(product.getType()))
                .withProductId(product.getId())
                .withType(CampaignType.BILLING_AGGREGATE)
                .withWalletId(wallet.getId())
                .withSum(BigDecimal.ZERO)
                .withSumToPay(BigDecimal.ZERO)
                .withSumLast(BigDecimal.ZERO)
                .withSumSpent(BigDecimal.ZERO)
                .withStartDate(now)
                .withLastChange(NOW_PLACEHOLDER)
                .withEnablePausedByDayBudgetEvent(true)
                .withEnableSendAccountNews(true)
                .withSmsTime(DEFAULT_SMS_TIME_INTERVAL)
                .withTimeZoneId(0L)
                .withFio(wallet.getFio())
                .withWarningBalance(DEFAULT_CAMPAIGN_WARNING_BALANCE)
                .withSmsFlags(EnumSet.of(SmsFlag.PAUSED_BY_DAY_BUDGET_SMS))
                .withEmail(wallet.getEmail())
                .withIsServiceRequested(false)
                .withIsSumAggregated(wallet.getIsSumAggregated())
        );
    }

    private static String getLockName(ClientId clientId, Long productId) {
        return StreamEx
                .of(NEW_BILLING_AGGREGATE_LOCK_PREFIX,
                        clientId,
                        productId
                )
                .joining("_");
    }

    private List<Long> createBillingAggregate(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                              BillingAggregateCampaign billingAggregate,
                                              String lockName) {

        DSLContext context = ppcDslContextProvider.ppc(addCampaignParametersContainer.getShard());
        try {
            Integer lockResult = context
                    .select(SqlUtils.mysqlGetLock(lockName, Duration.ofSeconds(0))).fetchOne().component1();
            checkState(Objects.equals(lockResult, 1), "failed to get mysql lock with name: " + lockName);
            boolean alreadyExists =
                    campaignTypedRepository.getClientsTypedCampaignsByType(addCampaignParametersContainer.getShard(),
                            addCampaignParametersContainer.getClientId(), Set.of(CampaignType.BILLING_AGGREGATE))
                            .values()
                            .stream()
                            .anyMatch(o -> ((BillingAggregateCampaign) o).getProductId()
                                    .equals(billingAggregate.getProductId()));

            if (alreadyExists) {
                return emptyList();
            }

            List<Long> createdAggregateId =
                    campaignModifyRepository.addBillingAggregate(addCampaignParametersContainer, billingAggregate);

            addToBalanceQueue(context, addCampaignParametersContainer.getOperatorUid(), createdAggregateId);
            return createdAggregateId;
        } finally {
            context.select(SqlUtils.mysqlReleaseLock(lockName)).execute();
        }
    }

    private void addToBalanceQueue(DSLContext context, Long operatorUid, List<Long> billingAggregateIds) {
        List<BalanceNotificationInfo> balanceNotificationInfos = StreamEx.of(billingAggregateIds)
                .map(aLong -> new BalanceNotificationInfo()
                        .withCidOrUid(aLong)
                        .withObjType(BalanceInfoQueueObjType.CID)
                        .withOperatorUid(operatorUid)
                        .withPriority(BalanceInfoQueuePriority.PRIORITY_CAMP_ON_NEW_BILLING_AGGREGATE))
                .toList();

        balanceInfoQueueService.addToBalanceInfoQueue(context, balanceNotificationInfos);
    }

    private void addWalletSumMigrationJob(int shard, ClientId clientId, Long operatorUid, Long walletId) {
        BalanceAggregateMigrationParams params = new BalanceAggregateMigrationParams()
                .withWalletId(walletId)
                .withIsRollback(false);

        Long jobId = dbQueueRepository.insertJob(shard, BALANCE_AGGREGATE_MIGRATION, clientId, operatorUid, params)
                .getId();

        logger.info("job {} was inserted in db queue. clientId: {}; uid: {}; walletId: {}, type {};",
                jobId, clientId.asLong(), operatorUid, walletId, false);
    }

}
