package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStopReason;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.currency.service.CurrencyRateService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.currency.Money.MONEY_MATH_CONTEXT;
import static ru.yandex.direct.utils.CommonUtils.max;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.misc.lang.ObjectUtils.min;

@ParametersAreNonnullByDefault
@Service
public class BrandSurveyConditionsService {

    public static final Map<String, Double> INTERFACE_DAILY_THRESHOLDS =
            Map.of(
                    "RUB", 70_000d,
                    "BYN", 584d,
                    "KZT", 107_000d,
                    "USD", 490d,
                    "TRY", 12_835d,
                    "CHF", 870d,
                    "EUR", 490d,
                    "UZS", 850_000d
            );

    public static final Map<String, Double> INTERFACE_DAILY_THRESHOLDS_NEW_AGE =
            Map.of(
                    "RUB", 70_000d,
                    "BYN", 584d,
                    "KZT", 107_000d,
                    "USD", 490d,
                    "TRY", 12_835d,
                    "CHF", 870d,
                    "EUR", 490d,
                    "UZS", 850_000d
            );

    public static final Map<String, Double> INTERFACE_TOTAL_THRESHOLDS =
            Map.of(
                    "RUB", 1_000_000d,
                    "BYN", 8_500d,
                    "KZT", 1_500_000d,
                    "TRY", 180_000d,
                    "CHF", 12_000d,
                    "UZS", 12_000_000d,
                    "USD", 7_000d,
                    "EUR", 7_000d
            );

    public static final Map<String, Double> INTERFACE_TOTAL_THRESHOLDS_NEW_AGE =
            Map.of(
                    "RUB", 1_000_000d,
                    "BYN", 8_500d,
                    "KZT", 1_500_000d,
                    "TRY", 180_000d,
                    "CHF", 12_000d,
                    "UZS", 12_000_000d,
                    "USD", 7_000d,
                    "EUR", 7_000d
            );

    private final CurrencyRateService currencyRateService;

    public BrandSurveyConditionsService(CurrencyRateService currencyRateService) {
        this.currencyRateService = currencyRateService;
    }

    public BrandSurveyStatus getBrandSurveyStatus(
            DbStrategy campaignStrategy,
            LocalDate campaignStartDate,
            @Nullable LocalDate campaignEndDate,
            BigDecimal dayBudget,
            CurrencyCode currencyCode,
            boolean totalBudgetWarnEnabled,
            LocalDate brandSurveyBudgetDatePropVal,
            boolean isBrandLiftHidden,
            List<Campaign> prevCampsForBrandlift//другие РК для брендлифта. Влияют на валидацию
    ) {
        Long period = getEstimatedPeriod(campaignStartDate, campaignEndDate, campaignStrategy);
        BigDecimal budget = getEstimatedBudget(campaignStrategy, dayBudget);
        List<BudgetCondition> budgetConditions = StreamEx.of(prevCampsForBrandlift)
                .map(it -> new BudgetCondition(getEstimatedPeriod(it.getStartTime(), it.getFinishTime(), it.getStrategy()),
                        getEstimatedBudget(it.getStrategy(), it.getDayBudget())))
                .append(new BudgetCondition(period, budget))
                .toList();

        var stopReasons = checkConditions(budgetConditions, currencyCode,
                isBrandLiftHidden, totalBudgetWarnEnabled, brandSurveyBudgetDatePropVal);

        return new BrandSurveyStatus()
                // добавлено для консистентности данных, на фронте статус не отображается
                .withSurveyStatusDaily(stopReasons.isEmpty() ? SurveyStatus.DRAFT : SurveyStatus.UNFEASIBLE)
                .withReasonIds(emptyList())
                .withSumSpentByDay(budget)
                .withSumSpentByTotalPeriod(budget.multiply(BigDecimal.valueOf(period)))
                .withBrandSurveyStopReasonsDaily(stopReasons);
    }

    public BrandSurveyStatus getCpmPriceBrandSurveyStatus(
            LocalDate campaignStartDate,
            LocalDate campaignEndDate,
            PricePackage pricePackage,
            BigDecimal budget,
            Boolean autoProlongation,
            boolean totalBudgetWarnEnabled,
            LocalDate brandSurveyBudgetDatePropVal,
            boolean isBrandLiftHidden,
            List<Campaign> prevCampsForBrandlift//другие РК для брендлифта. Влияют на валидацию
    ) {
        long period = getEstimatedCpmPricePeriod(campaignStartDate, campaignEndDate, pricePackage, autoProlongation);
        BigDecimal dayBudget = budget.divide(BigDecimal.valueOf(period), MONEY_MATH_CONTEXT);
        List<BudgetCondition> budgetConditions = StreamEx.of(prevCampsForBrandlift)
                .map(it -> new BudgetCondition(getEstimatedPeriod(it.getStartTime(), it.getFinishTime(), it.getStrategy()),
                        getEstimatedBudget(it.getStrategy(), it.getDayBudget())))
                .append(new BudgetCondition(period, dayBudget))
                .toList();
        var stopReasons = checkConditions(budgetConditions,
                pricePackage.getCurrency(),
                isBrandLiftHidden, totalBudgetWarnEnabled, brandSurveyBudgetDatePropVal);

        return new BrandSurveyStatus()
                // На фронте на данный момент показывается только значение поля brandSurveyStopReasonsDaily
                // остальные для ненулевых значений заполняются
                .withBrandSurveyStopReasonsDaily(stopReasons)
                .withSurveyStatusDaily(stopReasons.isEmpty() ? SurveyStatus.DRAFT : SurveyStatus.UNFEASIBLE)
                .withReasonIds(emptyList())
                .withSumSpentByTotalPeriod(budget)
                .withSumSpentByDay(dayBudget);
    }

    private class BudgetCondition {
        public long period;
        public BigDecimal budget;

        public BudgetCondition(long period, BigDecimal budget) {
            this.period = period;
            this.budget = budget;
        }
    }

    public Long getEstimatedPeriod(
            LocalDate campaignStartDate, @Nullable LocalDate campaignEndDate, DbStrategy strategy) {
        var startDate = campaignStartDate;
        var endDate = campaignEndDate;
        var strategyData = strategy.getStrategyData();
        startDate = max(startDate, nvl(strategyData.getStart(), LocalDate.MIN));

        switch (strategy.getStrategyName()) {
            case CPM_DEFAULT:
                if (campaignEndDate == null) {
                    return 1L;
                }
                break;

            case AUTOBUDGET_AVG_CPV:
            case AUTOBUDGET_MAX_IMPRESSIONS:
            case AUTOBUDGET_MAX_REACH:
                if (campaignEndDate == null) {
                    return 7L;
                }
                break;

            case AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD:
            case AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD:
            case AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD:

                if (strategyData.getAutoProlongation() == 1) {
                    endDate = max(nvl(endDate, LocalDate.MIN), strategyData.getFinish());
                } else {
                    endDate = min(nvl(endDate, LocalDate.MAX), strategyData.getFinish());
                }
                break;
            default:
                throw new IllegalStateException("Unexpected brandsurvey strategy");
        }

        return ChronoUnit.DAYS.between(startDate, endDate) + 1;
    }

    public long getEstimatedCpmPricePeriod(LocalDate campaignStartDate,
            LocalDate campaignEndDate,
            PricePackage pricePackage,
            Boolean autoProlongation) {

        LocalDate endDate = autoProlongation ?
                max(campaignEndDate, pricePackage.getDateEnd()) :
                min(campaignEndDate, pricePackage.getDateEnd());

        return ChronoUnit.DAYS.between(campaignStartDate, endDate) + 1;
    }

    private BigDecimal getEstimatedBudget(DbStrategy strategy, BigDecimal dayBudget) {
        var strategyPeriod = 0L;
        var strategyBudget = BigDecimal.ZERO;

        var strategyName = strategy.getStrategyName();
        var strategyData = strategy.getStrategyData();

        switch (strategyName) {
            case CPM_DEFAULT:
                strategyBudget = dayBudget;
                strategyPeriod = 1;
                break;

            case AUTOBUDGET_AVG_CPV:
            case AUTOBUDGET_MAX_IMPRESSIONS:
            case AUTOBUDGET_MAX_REACH:
                strategyBudget = strategyData.getSum();
                strategyPeriod = 7;
                break;

            case AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD:
            case AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD:
            case AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD:
                strategyBudget = strategyData.getBudget();
                strategyPeriod =
                        ChronoUnit.DAYS.between(strategyData.getStart(), strategyData.getFinish()) + 1;
        }

        return strategyBudget.divide(BigDecimal.valueOf(strategyPeriod), MONEY_MATH_CONTEXT);
    }

    private Set<BrandSurveyStopReason> checkConditions(
            List<BudgetCondition> budgetConditions, CurrencyCode currencyCode,
            boolean isBrandLiftHidden, boolean totalBudgetWarnEnabled, LocalDate brandSurveyBudgetDatePropVal) {
        var stopReasons = EnumSet.noneOf(BrandSurveyStopReason.class);

        // для скрытого Brand Lift'а не проверяем размер бюджета
        if (isBrandLiftHidden) {
            return stopReasons;
        }

        Double budget = StreamEx.of(budgetConditions)
                .mapToDouble(it -> it.budget.doubleValue())
                .sum();
        double budgetThreshold = brandSurveyBudgetThresholdDaily(
                currencyCode.getCurrency(), null, brandSurveyBudgetDatePropVal);
        double totalThreshold = brandSurveyBudgetThreshold(
                currencyCode.getCurrency(), null, brandSurveyBudgetDatePropVal);

        if (!INTERFACE_DAILY_THRESHOLDS.containsKey(currencyCode.name())) {
            Money budgetMoney = currencyRateService.convertMoney(Money.valueOf(budget, currencyCode), CurrencyCode.RUB);
            budget = budgetMoney.bigDecimalValue().doubleValue();
        }

        BigDecimal totalBudget = StreamEx.of(budgetConditions)
                .map(it -> it.budget.multiply(BigDecimal.valueOf(it.period), MONEY_MATH_CONTEXT))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        if (!INTERFACE_DAILY_THRESHOLDS.containsKey(currencyCode.name())) {
            Money budgetMoney = currencyRateService.convertMoney(Money.valueOf(totalBudget, currencyCode), CurrencyCode.RUB);
            totalBudget = budgetMoney.bigDecimalValue();
        }

        if(budget < budgetThreshold) {
            stopReasons.add(totalBudgetWarnEnabled ? BrandSurveyStopReason.LOW_DAILY_BUDGET
                    : BrandSurveyStopReason.LOW_BUDGET);
        }
        if (totalBudget.doubleValue() < totalThreshold) {
            stopReasons.add(totalBudgetWarnEnabled ? BrandSurveyStopReason.LOW_TOTAL_BUDGET
                    :BrandSurveyStopReason.LOW_BUDGET);
        }

        return stopReasons;
    }

    public Double brandSurveyBudgetThreshold(Currency currency, LocalDateTime createTime,
                                             LocalDate brandSurveyBudgetDatePropVal) {
        //Если createTime не nul и проверти есть и createTime позже проперти, то используем новые значения
        if (brandSurveyBudgetDatePropVal != null
                && nvl(createTime, LocalDateTime.now()).isAfter(brandSurveyBudgetDatePropVal.atStartOfDay())) {
            return INTERFACE_TOTAL_THRESHOLDS_NEW_AGE.getOrDefault(currency.getCode().name(),
                    INTERFACE_TOTAL_THRESHOLDS_NEW_AGE.get("RUB"));
        }
        return INTERFACE_TOTAL_THRESHOLDS.getOrDefault(currency.getCode().name(),
                INTERFACE_TOTAL_THRESHOLDS.get("RUB"));
    }

    public Double brandSurveyBudgetThresholdDaily(Currency currency, LocalDateTime createTime,
                                                  LocalDate brandSurveyBudgetDatePropVal) {
        if (brandSurveyBudgetDatePropVal != null
                && nvl(createTime, LocalDateTime.now()).isAfter(brandSurveyBudgetDatePropVal.atStartOfDay())) {
            return INTERFACE_DAILY_THRESHOLDS_NEW_AGE.getOrDefault(currency.getCode().name(),
                    INTERFACE_DAILY_THRESHOLDS_NEW_AGE.get("RUB"));
        }
        return INTERFACE_DAILY_THRESHOLDS.getOrDefault(currency.getCode().name(),
                INTERFACE_DAILY_THRESHOLDS.get("RUB"));
    }
}
