package ru.yandex.direct.core.entity.campaign.service;

import java.net.IDN;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.jooq.tools.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.container.DeleteCampMetrikaCountersRequest;
import ru.yandex.direct.core.entity.campaign.container.UpdateCampMetrikaCountersRequest;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.repository.CampMetrikaCountersRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CounterIdWithDomain;
import ru.yandex.direct.core.entity.metrika.model.MetrikaCounterByDomain;
import ru.yandex.direct.core.entity.metrika.repository.CalltrackingNumberClicksRepository;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCounterByDomainRepository;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterPermission;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters;
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.request.GetExistentCountersRequest;
import ru.yandex.direct.metrika.client.model.request.GrantAccessRequestStatusesRequest;
import ru.yandex.direct.metrika.client.model.request.RequestGrantsObjectType;
import ru.yandex.direct.metrika.client.model.request.RequestGrantsPermissionType;
import ru.yandex.direct.metrika.client.model.request.RequestGrantsRequest;
import ru.yandex.direct.metrika.client.model.request.RequestGrantsRequestItem;
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.metrika.client.model.response.GrantAccessRequestStatus;
import ru.yandex.direct.metrika.client.model.response.GrantAccessRequestStatusesResponse;
import ru.yandex.direct.metrika.client.model.response.RequestGrantsResponse;
import ru.yandex.direct.metrika.client.model.response.UserCounters;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.common.db.PpcPropertyNames.DOMAINS_NOT_ALLOWED_FOR_UNAVAILABLE_COUNTERS;
import static ru.yandex.direct.common.db.PpcPropertyNames.METRIKA_COUNTER_LIFETIME_IN_DAYS_FOR_AUTOCOMPLETE;
import static ru.yandex.direct.common.db.PpcPropertyNames.METRIKA_COUNTER_WITH_WEAK_RESTRICTIONS_LIFETIME_IN_DAYS;
import static ru.yandex.direct.core.entity.campaign.CampaignUtils.CPM_TYPES;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCounterSource;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCountersWithAdditionalInformation;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.cantAddOrDeleteMetrikaCountersToPerformanceCampaign;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.maxMetrikaCountersListSize;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.metrikaCounterIsUnavailable;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.metrikaCountersUnsupportedCampType;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.mustContainMetrikaCounters;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.Collectors.nullFriendlyMapCollector;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
public class CampMetrikaCountersService {
    private static final Logger logger = LoggerFactory.getLogger(CampMetrikaCountersService.class);

    private final MetrikaClient metrikaClient;
    private final RbacService rbacService;
    private final FeatureService featureService;

    private final ShardHelper shardHelper;
    private final CalltrackingNumberClicksRepository calltrackingNumberClicksRepository;
    private final CampMetrikaCountersRepository campMetrikaCountersRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final CampaignRepository campaignRepository;
    private final MetrikaCounterByDomainRepository metrikaCounterByDomainRepository;
    private final MetrikaCampaignRepository metrikaCampaignRepository;
    private final StrategyTypedRepository strategyTypedRepository;

    private final PpcProperty<Integer> metrikaCounterLifetimeInDaysProperty;
    private final PpcProperty<Integer> metrikaCounterWithWeakRestrictionsLifetimeInDaysProperty;
    private final PpcProperty<Set<String>> domainsNotAllowedForUnavailableCounters;

    public static final int MAX_METRIKA_COUNTERS_COUNT = 100;
    public static final Long MAX_METRIKA_COUNTER_ID = (long) Integer.MAX_VALUE;
    public static final Set<MetrikaCounterPermission> COUNTER_PERMISSIONS_TREATED_AS_EDITABLE =
            Set.of(MetrikaCounterPermission.EDIT, MetrikaCounterPermission.OWN);
    public static final Set<MetrikaCounterSource> TECHNICAL_COUNTER_SOURCES = Set.of(
            MetrikaCounterSource.SPRAV, MetrikaCounterSource.TURBO,
            MetrikaCounterSource.MARKET, MetrikaCounterSource.EDA);

    @Autowired
    public CampMetrikaCountersService(MetrikaClient metrikaClient,
                                      RbacService rbacService,
                                      FeatureService featureService,
                                      ShardHelper shardHelper,
                                      CalltrackingNumberClicksRepository calltrackingNumberClicksRepository,
                                      CampMetrikaCountersRepository campMetrikaCountersRepository,
                                      BannerCommonRepository bannerCommonRepository,
                                      CampaignRepository campaignRepository,
                                      MetrikaCounterByDomainRepository metrikaCounterByDomainRepository,
                                      MetrikaCampaignRepository metrikaCampaignRepository,
                                      StrategyTypedRepository strategyTypedRepository,
                                      PpcPropertiesSupport ppcPropertiesSupport) {
        this.metrikaClient = metrikaClient;
        this.rbacService = rbacService;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
        this.calltrackingNumberClicksRepository = calltrackingNumberClicksRepository;
        this.campMetrikaCountersRepository = campMetrikaCountersRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.campaignRepository = campaignRepository;
        this.metrikaCounterByDomainRepository = metrikaCounterByDomainRepository;
        this.metrikaCampaignRepository = metrikaCampaignRepository;
        this.strategyTypedRepository = strategyTypedRepository;

        metrikaCounterLifetimeInDaysProperty =
                ppcPropertiesSupport.get(METRIKA_COUNTER_LIFETIME_IN_DAYS_FOR_AUTOCOMPLETE, Duration.ofMinutes(5));
        metrikaCounterWithWeakRestrictionsLifetimeInDaysProperty =
                ppcPropertiesSupport.get(METRIKA_COUNTER_WITH_WEAK_RESTRICTIONS_LIFETIME_IN_DAYS,
                        Duration.ofMinutes(5));
        domainsNotAllowedForUnavailableCounters =
                ppcPropertiesSupport.get(DOMAINS_NOT_ALLOWED_FOR_UNAVAILABLE_COUNTERS, Duration.ofMinutes(5));
    }

    /**
     * Добавление счетчиков метрики в кампании
     * Во все указанные кампании в поле "Счетчик метрики" дописывается один или несколько указанных счетчиков.
     * Если в РК уже указаны какие-то счетчики, то новые дописываются справа через запятую.
     * в performance кампании нельзя добавить счетчики (будет ошибка при валидации)
     *
     * @param clientId id клиента
     * @param request  запрос на добавление (содержит номера кампаний и счетчики, которые нужно добавить)
     */
    public Result<UpdateCampMetrikaCountersRequest> addCampMetrikaCounters(ClientId clientId,
                                                                           UpdateCampMetrikaCountersRequest request,
                                                                           Applicability applicability) {
        ValidationResult<UpdateCampMetrikaCountersRequest, Defect> preValidationResult =
                validateUpdateCountersRequest(clientId, request, true);
        if (preValidationResult.hasAnyErrors()) {
            logger.debug("can not add metrika_counters: selection criteria contains errors");
            return Result.broken(preValidationResult);
        }

        List<Long> cids = request.getCids();
        List<Long> metrikaCounters = request.getMetrikaCounters();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, List<Long>> oldMetrikaCountersByCid =
                campMetrikaCountersRepository.getMetrikaCountersByCids(shard, cids);

        Map<Long, List<Long>> newMetrikaCountersByCid = EntryStream.of(oldMetrikaCountersByCid)
                .mapValues(counters -> StreamEx.of(counters).append(metrikaCounters).distinct().toList())
                .toMap();

        ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validationResult =
                validateAdd(shard, clientId, request, newMetrikaCountersByCid);
        List<Long> validCids = getValidCidsFromUpdateValidationResult(cids, validationResult);

        if (isFull(applicability) && validCids.size() != cids.size()) {
            logger.debug("can not add metrika_counters: there are invalid cids");
            return Result.broken(validationResult);
        }
        Map<Long, List<Long>> newMetrikaCountersByValidCid = EntryStream.of(newMetrikaCountersByCid)
                .filterKeys(validCids::contains)
                .toMap();

        Map<Long, List<Long>> changedValidMetrikaCountersByCid = EntryStream.of(newMetrikaCountersByValidCid)
                .filterKeyValue((cid, counters) -> !counters.equals(oldMetrikaCountersByCid.get(cid)))
                .toMap();

        if (!changedValidMetrikaCountersByCid.isEmpty()) {
            var counterIds = StreamEx.of(changedValidMetrikaCountersByCid.values())
                    .flatMap(Collection::stream)
                    .toSet();
            Map<Long, MetrikaCounterWithAdditionalInformation> availableCountersById =
                    listToMap(getAvailableCountersByClientId(clientId, counterIds),
                            MetrikaCounterWithAdditionalInformation::getId, identity());

            Map<Long, List<MetrikaCounter>> countersForUpdate =
                    getCampMetrikaCountersForUpdate(changedValidMetrikaCountersByCid, availableCountersById,
                            Collections.emptyMap(), emptySet());
            if (featureService.isEnabledForClientId(clientId,
                    FeatureName.TOGETHER_UPDATING_STRATEGY_AND_CAMPAIGN_METRIKA_COUNTERS)) {
                updateMetrikaCounterForStrategies(shard, clientId, countersForUpdate, validCids,
                        newMetrikaCountersByValidCid);
            } else {
                campMetrikaCountersRepository.updateMetrikaCounters(
                        shard,
                        countersForUpdate
                );
            }

            // DIRECT-23646: счетчики меняются на кампанию, а передаются в БК в баннерах
            // чтобы счетчики передавались в БК меняем statusBsSynced у баннеров
            bannerCommonRepository.updateStatusBsSyncedByCampaignIds(shard, countersForUpdate.keySet(),
                    StatusBsSynced.NO);
        }
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        logger.debug("metrika counters were added successfully");
        return Result.successful(null, validationResult);
    }


    /**
     * Удаление переданного списка счетчиков метрики в кампании
     * Во все указанные кампании в поле "Счетчик метрики" удаляется один или несколько указанных счетчиков.
     * Порядок оставщихся счетчиков сохраняется
     * Из performance кампаний нельзя удалить счетчик (для этих кампаний будет ошибка при валидации)
     *
     * @param clientId id клиента
     * @param request  запрос на удаление (содержит номера кампаний и счетчики, которые нужно удалить)
     */
    public Result<UpdateCampMetrikaCountersRequest> removeCampMetrikaCounters(ClientId clientId,
                                                                              UpdateCampMetrikaCountersRequest request,
                                                                              Applicability applicability) {
        ValidationResult<UpdateCampMetrikaCountersRequest, Defect> preValidationResult =
                validateUpdateCountersRequest(clientId, request, false);
        if (preValidationResult.hasAnyErrors()) {
            logger.debug("can not remove metrika_counters: selection criteria contains errors");
            return Result.broken(preValidationResult);
        }

        List<Long> cids = request.getCids();
        Set<Long> deletingMetrikaCounterIds = Set.copyOf(request.getMetrikaCounters());

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, List<Long>> oldMetrikaCountersByCid =
                campMetrikaCountersRepository.getMetrikaCountersByCids(shard, cids);
        Map<Long, Set<Long>> oldSpravCountersByCid = metrikaCampaignRepository.getSpravCountersByCampaignId(shard,
                cids);

        Map<Long, List<Long>> newMetrikaCountersByCid = EntryStream.of(oldMetrikaCountersByCid)
                .mapValues(counters -> filterList(counters, not(deletingMetrikaCounterIds::contains)))
                .toMap();

        ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validationResult =
                validateRemove(shard, clientId, request, newMetrikaCountersByCid, oldMetrikaCountersByCid,
                        oldSpravCountersByCid);
        List<Long> validCids = getValidCidsFromUpdateValidationResult(cids, validationResult);

        if (isFull(applicability) && validCids.size() != cids.size()) {
            logger.debug("can not remove metrika_counters: there are invalid cids");
            return Result.broken(validationResult);
        }

        Map<Long, List<Long>> newMetrikaCountersByValidCid = EntryStream.of(newMetrikaCountersByCid)
                .filterKeys(validCids::contains)
                .toMap();

        Map<Long, List<Long>> changedValidMetrikaCountersByCid = EntryStream.of(newMetrikaCountersByValidCid)
                .filterKeyValue((cid, counters) -> !counters.equals(oldMetrikaCountersByCid.get(cid)))
                .toMap();

        if (!changedValidMetrikaCountersByCid.isEmpty()) {
            // обновляем счетчики на кампаниях, у которых остались счетчики после вычета
            Map<Long, List<Long>> updateMetrikaCountersByCid = EntryStream.of(changedValidMetrikaCountersByCid)
                    .removeValues(List::isEmpty)
                    .toMap();

            var counterIds = StreamEx.of(updateMetrikaCountersByCid.values())
                    .flatMap(Collection::stream)
                    .toSet();
            Map<Long, MetrikaCounterWithAdditionalInformation> availableCountersById =
                    listToMap(getAvailableCountersByClientId(clientId, counterIds),
                            MetrikaCounterWithAdditionalInformation::getId, identity());

            Map<Long, List<MetrikaCounter>> countersForUpdate =
                    getCampMetrikaCountersForUpdate(updateMetrikaCountersByCid, availableCountersById,
                            Collections.emptyMap(), emptySet());

            if (featureService.isEnabledForClientId(clientId,
                    FeatureName.TOGETHER_UPDATING_STRATEGY_AND_CAMPAIGN_METRIKA_COUNTERS)) {
                updateMetrikaCounterForStrategies(shard, clientId, countersForUpdate, validCids,
                        newMetrikaCountersByValidCid);
            } else {
                campMetrikaCountersRepository.updateMetrikaCounters(
                        shard,
                        countersForUpdate);
            }

            // удаляем счетчики на кампаниях, у которых не осталось счетчиков после вычета
            Set<Long> campaignIdsForDeleteAllMetrikaCounters =
                    EntryStream.of(changedValidMetrikaCountersByCid)
                            .filterValues(List::isEmpty)
                            .keys().toSet();

            campMetrikaCountersRepository.deleteMetrikaCounters(shard, campaignIdsForDeleteAllMetrikaCounters);

            // DIRECT-23646: счетчики меняются на кампанию, а передаются в БК в баннерах
            // чтобы счетчики передавались в БК меняем statusBsSynced у баннеров
            bannerCommonRepository.updateStatusBsSyncedByCampaignIds(shard, countersForUpdate.keySet(),
                    StatusBsSynced.NO);
        }
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        logger.debug("metrika counters were removed successfully");
        return Result.successful(null, validationResult);
    }


    /**
     * Замена счетчиков метрики в кампаниях
     * У performance кампаний будет установлен первый счётчик из списка metrikaCounters (тк в них может быть только
     * один счетчик)
     * для performance кампаний при валидации проверяется доступность счетчика, при сохранении вычисляется флаг
     * has_ecommerce
     * В результате замены не должны пропадать счётчики справочника, если такое случается - будет ошибка валидации.
     *
     * @param clientId id клиента
     * @param request  запрос, в котором содержатся номера кампаний и счетчики, которые нужно проставить
     */
    public Result<UpdateCampMetrikaCountersRequest> replaceCampMetrikaCounters(ClientId clientId,
                                                                               UpdateCampMetrikaCountersRequest request,
                                                                               Applicability applicability) {
        ValidationResult<UpdateCampMetrikaCountersRequest, Defect> preValidationResult =
                validateUpdateCountersRequest(clientId, request, true);
        if (preValidationResult.hasAnyErrors()) {
            logger.debug("can not replace metrika_counters: selection criteria contains errors");
            return Result.broken(preValidationResult);
        }

        List<Long> cids = request.getCids();
        List<Long> metrikaCounters = StreamEx.of(request.getMetrikaCounters()).distinct().toList();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Set<Long> performanceIds = getPerformanceIds(clientId, cids);
        Map<Long, List<Long>> oldMetrikaCountersByCid =
                campMetrikaCountersRepository.getMetrikaCountersByCids(shard, cids);
        Map<Long, Set<Long>> oldSpravCountersByCid = metrikaCampaignRepository.getSpravCountersByCampaignId(shard,
                cids);

        //для performance кампаний будет установлен первый счётчик из переданных
        Map<Long, List<Long>> newMetrikaCountersByCid = EntryStream.of(oldMetrikaCountersByCid)
                .mapToValue((cid, counters) -> performanceIds.contains(cid)
                        ? Collections.singletonList(metrikaCounters.get(0)) : metrikaCounters)
                .toMap();

        ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validationResult =
                validateReplace(shard, clientId, request, performanceIds, oldSpravCountersByCid,
                        newMetrikaCountersByCid);
        List<Long> validCids = getValidCidsFromUpdateValidationResult(cids, validationResult);

        if (isFull(applicability) && validCids.size() != cids.size()) {
            logger.debug("can not replace metrika_counters: there are invalid cids");
            return Result.broken(validationResult);
        }

        //если у счетчиков поменялась сортировка, считаем их тоже разными, т.к. порядок важен клиенту
        Map<Long, List<Long>> newMetrikaCountersByValidCid = EntryStream.of(newMetrikaCountersByCid)
                .filterKeys(validCids::contains)
                .toMap();

        Map<Long, List<Long>> changedValidMetrikaCountersByCid = EntryStream.of(newMetrikaCountersByValidCid)
                .filterKeyValue((cid, counters) -> !counters.equals(oldMetrikaCountersByCid.get(cid)))
                .toMap();

        if (!changedValidMetrikaCountersByCid.isEmpty()) {
            //Для перфоманс кампаний вычислим на счетчике признак has_ecommerce
            Set<Long> performanceCounters = StreamEx.of(performanceIds)
                    .flatCollection(changedValidMetrikaCountersByCid::get).toSet();
            Map<Long, Boolean> ecommerceFlagByCounter = checkCountersEcommerce(performanceCounters);

            var counterIds = StreamEx.of(changedValidMetrikaCountersByCid.values())
                    .flatMap(Collection::stream)
                    .toSet();
            Map<Long, MetrikaCounterWithAdditionalInformation> availableCountersById =
                    listToMap(getAvailableCountersByClientId(clientId, counterIds),
                            MetrikaCounterWithAdditionalInformation::getId, identity());

            Map<Long, List<MetrikaCounter>> countersForUpdate = getCampMetrikaCountersForUpdate(
                    changedValidMetrikaCountersByCid, availableCountersById, ecommerceFlagByCounter, performanceIds);

            if (featureService.isEnabledForClientId(clientId,
                    FeatureName.TOGETHER_UPDATING_STRATEGY_AND_CAMPAIGN_METRIKA_COUNTERS)) {
                updateMetrikaCounterForStrategies(shard, clientId, countersForUpdate, validCids,
                        newMetrikaCountersByValidCid);
            } else {
                campMetrikaCountersRepository.updateMetrikaCounters(
                        shard,
                        countersForUpdate);
            }
            // DIRECT-23646: чтобы счетчики передавались в БК меняем statusBsSynced у баннеров
            bannerCommonRepository.updateStatusBsSyncedByCampaignIds(shard, countersForUpdate.keySet(),
                    StatusBsSynced.NO);
        }
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        logger.debug("metrika counters were replaced successfully");
        return Result.successful(null, validationResult);
    }

    /**
     * Удаление всех счетчиков метрики в кампаниях
     * Из performance кампаний нельзя удалить счетчик (для этих кампаний будет ошибка при валидации)
     *
     * @param clientId id клиента
     * @param request  запрос, в котором содержатся номера кампаний, на которых нужно удалить счетчики
     */
    public Result<DeleteCampMetrikaCountersRequest> deleteAllCampMetrikaCounters(ClientId clientId,
                                                                                 DeleteCampMetrikaCountersRequest request,
                                                                                 Applicability applicability) {
        ValidationResult<DeleteCampMetrikaCountersRequest, Defect> preValidationResult =
                validateDeleteCountersRequest(request);
        if (preValidationResult.hasAnyErrors()) {
            logger.debug("can not delete metrika_counters: selection criteria contains errors");
            return Result.broken(preValidationResult);
        }
        List<Long> cids = request.getCids();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, List<Long>> campMetrikaCounters = campMetrikaCountersRepository.getMetrikaCountersByCids(shard, cids);

        ValidationResult<DeleteCampMetrikaCountersRequest, Defect> validationResult =
                validateDelete(shard, clientId, request);
        List<Long> validCids = getValidCidsFromDeleteValidationResult(cids, validationResult);

        if (isFull(applicability) && validCids.size() != cids.size()) {
            logger.debug("can not delete metrika_counters: there are invalid cids");
            return Result.broken(validationResult);
        }

        Map<Long, List<Long>> newValidMetrikaCounters = EntryStream.of(campMetrikaCounters)
                .filterKeys(validCids::contains)
                .toMap();

        Set<Long> validCidsWithMetrikaCounters = EntryStream.of(newValidMetrikaCounters)
                .removeValues(Collection::isEmpty)
                .keys().toSet();

        if (!validCidsWithMetrikaCounters.isEmpty()) {
            if (featureService.isEnabledForClientId(clientId,
                    FeatureName.TOGETHER_UPDATING_STRATEGY_AND_CAMPAIGN_METRIKA_COUNTERS)) {

                Map<Long, List<Long>> newListOfMetrikaCountersByCid = EntryStream.of(campMetrikaCounters)
                        .filterKeys(not(validCids::contains))
                        .toMap();

                updateMetrikaCounterForStrategies(
                        shard,
                        clientId,
                        emptyMap(),
                        validCids,
                        newListOfMetrikaCountersByCid
                );
            }

            campMetrikaCountersRepository.deleteMetrikaCounters(shard, validCidsWithMetrikaCounters);
            // DIRECT-23646: чтобы счетчики передавались в БК меняем statusBsSynced у баннеров
            bannerCommonRepository.updateStatusBsSyncedByCampaignIds(shard, validCidsWithMetrikaCounters,
                    StatusBsSynced.NO);
        }
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        logger.debug("metrika counters were deleted successfully");
        return Result.successful(null, validationResult);
    }

    public Map<Long, List<Long>> getCounterByCampaignIds(ClientId clientId, Collection<Long> cids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campMetrikaCountersRepository.getMetrikaCountersByCids(shard, cids);
    }

    /**
     * Получить количество кликов по номерам телефонов для заданного домена
     * В YT таблице, из которой достается информация, все домены в ASCII.
     * Для поиска в таблице переданные домены конвертируются в ASCII,
     */
    public Map<Long, Map<String, Integer>> getClicksOnPhonesByCounters(String domain, Collection<Long> counterIds) {
        var asciiDomain = IDN.toASCII(domain);
        return calltrackingNumberClicksRepository.getClicksOnPhonesByCounters(asciiDomain, counterIds);
    }

    /**
     * Получить количество кликов по номерам телефонов для переданных пар счетчик+домен
     * В YT таблице, из которой достается информация, все домены в ASCII.
     * Для поиска в таблице переданные домены конвертируются в ASCII,
     * но в ответе домен будет в том виде, в котором он передан
     */
    public Map<CounterIdWithDomain, Map<String, Integer>> getClicksOnPhonesByDomainWithCounterIds(
            Collection<CounterIdWithDomain> counterIdWithDomains) {
        if (counterIdWithDomains.isEmpty()) {
            return Map.of();
        }
        Map<String, String> domainToAsciiDomain = listToMap(
                counterIdWithDomains,
                CounterIdWithDomain::getDomain,
                c -> IDN.toASCII(c.getDomain())
        );
        // Отображение, чтобы достать исходный домен из ASCII обратно, чтобы вернуть так, как подавали на вход
        // При обратном отображении может получиться несколько доменов для одного ASCII домена,
        // если, например, в исходном списке были и кириллица, и punycode,
        // которые в ASCII переводятся как одинаковые punycode
        Map<String, Set<String>> asciiDomainToDomains = new HashMap<>();
        domainToAsciiDomain.forEach((d, ad) -> asciiDomainToDomains.computeIfAbsent(ad, v -> new HashSet<>()).add(d));

        List<CounterIdWithDomain> counterIdWithAsciiDomains = mapList(
                counterIdWithDomains,
                c -> c.copyWithDomain(domainToAsciiDomain.get(c.getDomain()))
        );

        var clicksOnPhonesByAsciiDomainWithCounterIds =
                calltrackingNumberClicksRepository.getClicksOnPhonesByDomainWithCounterIds(counterIdWithAsciiDomains);

        Map<CounterIdWithDomain, Map<String, Integer>> clicksOnPhonesByDomainWithCounterIds = new HashMap<>();
        clicksOnPhonesByAsciiDomainWithCounterIds.forEach((c, phoneClicks) -> {
            var asciiDomain = c.getDomain();
            asciiDomainToDomains.get(asciiDomain)
                    .forEach(d -> clicksOnPhonesByDomainWithCounterIds.put(c.copyWithDomain(d), phoneClicks));
        });
        return clicksOnPhonesByDomainWithCounterIds;
    }

    ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validateUpdateCountersRequest(
            ClientId clientId, UpdateCampMetrikaCountersRequest request, boolean needCheckCountersCount) {
        ModelItemValidationBuilder<UpdateCampMetrikaCountersRequest> ivb =
                ModelItemValidationBuilder.of(request);
        ivb.list(UpdateCampMetrikaCountersRequest.CIDS)
                .check(notNull())
                .check(eachNotNull())
                .check(minListSize(1));
        ivb.list(UpdateCampMetrikaCountersRequest.METRIKA_COUNTERS)
                .check(notNull())
                .check(eachNotNull())
                .check(minListSize(1))
                .check(maxListSize(MAX_METRIKA_COUNTERS_COUNT), When.isTrue(needCheckCountersCount))
                .checkEach(validId())
                .checkEach(notGreaterThan(CampMetrikaCountersService.MAX_METRIKA_COUNTER_ID), When.isValid());
        return ivb.getResult();
    }


    private ValidationResult<DeleteCampMetrikaCountersRequest, Defect> validateDeleteCountersRequest(
            DeleteCampMetrikaCountersRequest request) {
        ModelItemValidationBuilder<DeleteCampMetrikaCountersRequest> ivb =
                ModelItemValidationBuilder.of(request);
        ivb.list(DeleteCampMetrikaCountersRequest.CIDS)
                .check(notNull())
                .check(eachNotNull())
                .check(minListSize(1));
        return ivb.getResult();
    }


    ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validateAdd(
            int shard,
            ClientId clientId,
            UpdateCampMetrikaCountersRequest request,
            Map<Long, List<Long>> campMetrikaCounters) {
        List<Long> cids = request.getCids();
        Set<Long> performanceIds = getPerformanceIds(clientId, cids);

        ModelItemValidationBuilder<UpdateCampMetrikaCountersRequest> ivb =
                ModelItemValidationBuilder.of(request);
        ivb.list(UpdateCampMetrikaCountersRequest.CIDS)
                .checkBy(elements -> validateElements(shard, clientId, elements, campMetrikaCounters))
                .checkEach(notInSet(performanceIds), cantAddOrDeleteMetrikaCountersToPerformanceCampaign(),
                        When.isValid());
        return ivb.getResult();
    }


    ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validateRemove(
            int shard,
            ClientId clientId,
            UpdateCampMetrikaCountersRequest request,
            Map<Long, List<Long>> newMetrikaCountersByCid,
            Map<Long, List<Long>> oldMetrikaCountersByCid,
            Map<Long, Set<Long>> oldSpravCountersByCid) {
        Set<Long> performanceCampaignIds = getPerformanceIds(clientId, request.getCids());
        Set<Long> affectedPerformanceCampaignIds = filterToSet(performanceCampaignIds,
                cid -> !ListUtils.isEqualList(newMetrikaCountersByCid.get(cid), oldMetrikaCountersByCid.get(cid)));

        var ivb = ModelItemValidationBuilder.of(request);
        ivb.list(UpdateCampMetrikaCountersRequest.CIDS)
                .checkBy(elements -> validateElements(shard, clientId, elements, newMetrikaCountersByCid))
                .checkEach(notInSet(affectedPerformanceCampaignIds),
                        cantAddOrDeleteMetrikaCountersToPerformanceCampaign(), When.isValid())
                .checkEach(spravCountersAreNotDeleted(oldSpravCountersByCid, newMetrikaCountersByCid));
        return ivb.getResult();
    }


    ValidationResult<DeleteCampMetrikaCountersRequest, Defect> validateDelete(
            int shard,
            ClientId clientId,
            DeleteCampMetrikaCountersRequest request) {
        List<Long> cids = request.getCids();
        Set<Long> performanceIds = getPerformanceIds(clientId, cids);

        ModelItemValidationBuilder<DeleteCampMetrikaCountersRequest> ivb =
                ModelItemValidationBuilder.of(request);
        ivb.list(DeleteCampMetrikaCountersRequest.CIDS)
                .checkBy(elements -> validateElements(shard, clientId, elements, new HashMap<>()))
                .checkEach(notInSet(performanceIds), cantAddOrDeleteMetrikaCountersToPerformanceCampaign(),
                        When.isValid());
        return ivb.getResult();
    }

    ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validateReplace(
            int shard,
            ClientId clientId,
            UpdateCampMetrikaCountersRequest request,
            Set<Long> performanceCids,
            Map<Long, Set<Long>> oldSpravCountersByCid,
            Map<Long, List<Long>> newMetrikaCountersByCid) {
        var counterIds = EntryStream.of(newMetrikaCountersByCid)
                .values()
                .flatMap(Collection::stream)
                .toSet();
        Set<Long> availableCounters = getAvailableCounterIdsByClientId(clientId, counterIds);
        Set<Long> performanceIdsWithOneCounter = StreamEx.of(performanceCids)
                .filter(cid -> newMetrikaCountersByCid.get(cid) != null && newMetrikaCountersByCid.get(cid).size() == 1)
                .toSet();
        Set<Long> performanceIdsWithTooManyCounters = Sets.difference(performanceCids, performanceIdsWithOneCounter);
        Set<Long> performanceIdsWithUnavailableCounter = StreamEx.of(performanceIdsWithOneCounter)
                .filter(cid -> !availableCounters.containsAll(newMetrikaCountersByCid.get(cid))).toSet();

        ModelItemValidationBuilder<UpdateCampMetrikaCountersRequest> ivb =
                ModelItemValidationBuilder.of(request);
        ivb.list(UpdateCampMetrikaCountersRequest.CIDS)
                .checkBy(elements -> validateElements(shard, clientId, elements, newMetrikaCountersByCid))
                .checkEach(notInSet(performanceIdsWithTooManyCounters),
                        CollectionDefects.maxCollectionSize(1), When.isValid())
                .checkEach(notInSet(performanceIdsWithUnavailableCounter),
                        metrikaCounterIsUnavailable(), When.isValid())
                .checkEach(spravCountersAreNotDeleted(oldSpravCountersByCid, newMetrikaCountersByCid));
        return ivb.getResult();
    }

    private Constraint<Long, Defect> spravCountersAreNotDeleted(Map<Long, Set<Long>> oldSpravCountersByCid,
                                                                Map<Long, List<Long>> newCountersByCid) {
        return campaignId -> {
            Set<Long> oldSpravCounters = oldSpravCountersByCid.get(campaignId);
            if (oldSpravCounters == null) {
                return null;
            }
            List<Long> newCounters = newCountersByCid.get(campaignId);
            Set<Long> missingCounters = oldSpravCounters.stream()
                    .filter(not(newCounters::contains))
                    .collect(Collectors.toSet());
            return missingCounters.size() > 0 ?
                    mustContainMetrikaCounters(missingCounters) : null;
        };
    }

    private ValidationResult<List<Long>, Defect> validateElements(
            int shard,
            ClientId clientId,
            List<Long> cids,
            Map<Long, List<Long>> campMetrikaCounters) {
        Collection<CampaignSimple> campaigns = campaignRepository.getCampaignsSimple(shard, cids).values();

        Set<Long> clientCampaignIds =
                StreamEx.of(campaigns).filter(c -> c.getClientId().equals(clientId.asLong())).map(CampaignSimple::getId)
                        .toSet();
        Set<Long> archivedIds =
                StreamEx.of(campaigns).filter(CampaignSimple::getStatusArchived).map(CampaignSimple::getId).toSet();
        Set<CampaignType> unsupportedTypes = new HashSet<>(Set.of(CampaignType.MOBILE_CONTENT, CampaignType.MCB));
        if (featureService.isEnabledForClientId(clientId, FeatureName.IS_CPM_BANNER_CAMPAIGN_DISABLED)) {
            unsupportedTypes.addAll(CPM_TYPES);
        }
        Set<Long> unsupportedTypesIds = StreamEx.of(campaigns).filter(c -> unsupportedTypes.contains(c.getType()))
                .map(CampaignSimple::getId).toSet();

        Set<Long> tooManyCountersIds = EntryStream.of(campMetrikaCounters)
                .filterValues(counters -> counters.size() > MAX_METRIKA_COUNTERS_COUNT)
                .keys().toSet();

        return ListValidationBuilder.<Long, Defect>of(cids)
                .check(notNull())
                .check(eachNotNull())
                .check(minListSize(1))
                .checkEach(validId())
                .checkEach(inSet(clientCampaignIds), CommonDefects.objectNotFound(), When.isValid())
                .checkEach(notInSet(archivedIds), archivedCampaignModification(), When.isValid())
                .checkEach(notInSet(unsupportedTypesIds), metrikaCountersUnsupportedCampType(), When.isValid())
                .checkEach(notInSet(tooManyCountersIds), maxMetrikaCountersListSize(MAX_METRIKA_COUNTERS_COUNT),
                        When.isValid())
                .getResult();
    }

    /**
     * Возвращает id кампаний для которых обновляли счетчики
     * Сложность получения id заключается, в том что в результате операции они не возвращаются и нужно уметь отличать,
     * что результат валидации получен на превалидации или нет.
     * Ожидается, что превалидация может сработать только для счетчиков, а id кампаний валидируются в вызывающем коде,
     * до вызова операции обновления.
     */
    public static List<Long> getUpdatedCampaignIdsFromResult(Result<UpdateCampMetrikaCountersRequest> updateResult) {
        //noinspection unchecked,rawtypes
        var validationResult =
                (ValidationResult<UpdateCampMetrikaCountersRequest, Defect>) updateResult.getValidationResult();
        UpdateCampMetrikaCountersRequest request = validationResult.getValue();

        ValidationResult<List<Long>, Defect> metrikaCountersValidationResult =
                validationResult.getOrCreateSubValidationResult(
                        field(UpdateCampMetrikaCountersRequest.METRIKA_COUNTERS.name()), request.getMetrikaCounters());
        // проверяем, что сработала превалидация или нет
        // если сработала, то возвращаем пустой список, т.к. ничего не обновлено в рамках операции
        if (metrikaCountersValidationResult.hasAnyErrors()) {
            return Collections.emptyList();
        }

        return getValidCidsFromUpdateValidationResult(request.getCids(), validationResult);
    }

    private static List<Long> getValidCidsFromUpdateValidationResult(
            List<Long> cids,
            ValidationResult<UpdateCampMetrikaCountersRequest, Defect> validationResult) {
        ValidationResult<List<Long>, Defect> cidsValidationResult =
                validationResult.getOrCreateSubValidationResult(
                        field(UpdateCampMetrikaCountersRequest.CIDS.name()), cids);
        return getValidItems(cidsValidationResult);
    }


    private List<Long> getValidCidsFromDeleteValidationResult(List<Long> cids,
                                                              ValidationResult<DeleteCampMetrikaCountersRequest,
                                                                      Defect> validationResult) {
        ValidationResult<List<Long>, Defect> cidsValidationResult =
                validationResult.getOrCreateSubValidationResult(
                        field(DeleteCampMetrikaCountersRequest.CIDS.name()), cids);
        return getValidItems(cidsValidationResult);
    }

    //Потенциально может вернуть до 300к счетчиков
    //используейте getAvailableCounterIdsByClientId с фильтром по counterId
    @Deprecated
    public Set<Long> getAvailableCounterIdsByClientId(ClientId clientId) {
        List<Long> clientRepresentativesUids = rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        List<UserCounters> userCounters = metrikaClient.getUsersCountersNum(clientRepresentativesUids);
        return StreamEx.of(userCounters).flatCollection(UserCounters::getCounterIds).map(Integer::longValue).toSet();
    }

    public Set<Long> getAvailableCounterIdsByClientId(ClientId clientId, Collection<Long> counterIds) {
        if (counterIds.isEmpty()) {
            return emptySet();
        }
        List<Long> clientRepresentativesUids = rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        List<UserCounters> userCounters = metrikaClient.getUsersCountersNum2(clientRepresentativesUids, counterIds)
                .getUsers();
        return StreamEx.of(userCounters).flatCollection(UserCounters::getCounterIds).map(Integer::longValue).toSet();
    }


    //Потенциально может вернуть 100к счетчиков.
    // использовать метод getAvailableCountersByClientId с фильтрацией по счетчикам
    @Deprecated
    public Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersByClientId(ClientId clientId) {
        return getAvailableCountersByClientId(clientId, CounterWithAdditionalInformationFilter.defaultFilter());
    }

    public Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersByClientId(
            ClientId clientId,
            Collection<Long> counterIds) {
        return getAvailableCountersByClientId(clientId, CounterWithAdditionalInformationFilter.defaultFilter()
                .withCounterIds(counterIds));
    }

    public Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersByClientId(
            ClientId clientId,
            CounterWithAdditionalInformationFilter filter) {
        List<Long> clientRepresentativesUids = rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        return getAvailableCountersByUids(clientRepresentativesUids, filter);
    }

    public Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersByUids(
            List<Long> uids,
            CounterWithAdditionalInformationFilter filter) {
        if (filter.counterIds != null) {
            if (filter.counterIds.isEmpty()) {
                return emptySet();
            }
            return getAvailableExtendedCountersWithFilter(uids, filter)
                    .getCounters();
        }
        List<UserCountersExtended> usersCountersNumExtended = metrikaClient.getUsersCountersNumExtended(uids);

        return toMetrikaCountersWithAdditionalInformation(usersCountersNumExtended, filter);
    }

    public MetrikaCounterWithAdditionalInformationContainer getAvailableExtendedCountersWithFilter(
            List<Long> uids,
            CounterWithAdditionalInformationFilter filter) {
        var counterFilter = new UserCountersExtendedFilter()
                .withPrefix(filter.counterNamePrefix);
        if (filter.counterIds != null) {
            counterFilter.withCounterIds(new ArrayList<>(filter.counterIds));
        }
        if (filter.countersLimit != null) {
            counterFilter.withLimit(filter.countersLimit);
        }
        var response = metrikaClient
                .getUsersCountersNumExtended2(uids, counterFilter);

        return new MetrikaCounterWithAdditionalInformationContainer()
                .withCounters(toMetrikaCountersWithAdditionalInformation(response.getUsers(), filter))
                .withHasMoreCounters(response.isHasMoreCounters());
    }

    public Set<MetrikaCounterWithAdditionalInformation> getAvailableForEditingCounters(Long uid,
                                                                                       Collection<Long> counterIds) {
        return filterToSet(getAvailableCountersByUids(List.of(uid),
                        CounterWithAdditionalInformationFilter.defaultFilter()
                                .withCounterIds(counterIds)),
                counter -> counter.getPermissionsByUid().containsKey(uid) &&
                        COUNTER_PERMISSIONS_TREATED_AS_EDITABLE.contains(counter.getPermissionsByUid().get(uid)));
    }

    /**
     * Доступные клиенту счетчики.
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к доступным счетчикам идентификаторы счетчиков организаций этого клиента.
     * Счетчики организаций выбираются из БД
     */
    public Set<Long> getAvailableCounterIdsForGoals(ClientId clientId) {
        return getAvailableCounterIdsForGoals(clientId, () -> getSpravCounterIds(clientId, null));
    }

    /**
     * Доступные клиенту счетчики.
     * если counterIds не пустой, то проверяет доступность только счетчиков из списка
     * если counterIds пустой, то возвращает все счетчики
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к доступным счетчикам идентификаторы счетчиков организаций этого клиента.
     * Счетчики организаций фильтруются из заданного списка {@code inputCounterIds}
     * Их типы узнаем из Метрики, т.к. в БД их еще может не быть
     */
    public Set<Long> getAvailableAndFilterInputCounterIdsInMetrikaForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds
    ) {
        return getAvailableCounterIdsForGoals(clientId, inputCounterIds, this::filterSpravCounterIdsInMetrika);
    }

    public Set<Long> getAvailableAndFilterInputCounterIdsInMetrikaForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds,
            boolean isStrictFilterByInputCounters
    ) {
        return getAvailableCounterIdsForGoals(clientId, inputCounterIds, this::filterSpravCounterIdsInMetrika,
                isStrictFilterByInputCounters);
    }

    /**
     * Доступные клиенту счетчики.
     * если counterIds не пустой, то проверяет доступность только счетчиков из списка
     * если counterIds пустой, то возвращает все счетчики
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к доступным счетчикам идентификаторы счетчиков организаций этого клиента.
     * Счетчики организаций фильтруются из заданного списка {@code inputCounterIds}
     */
    public Set<Long> getAvailableAndFilterInputCounterIdsForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds
    ) {
        return getAvailableCounterIdsForGoals(clientId, inputCounterIds, ids -> filterSpravCounterIds(clientId, ids));
    }

    /**
     * Доступные клиенту счетчики.
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к доступным счетчикам идентификаторы счетчиков организаций этого клиента.
     * Счетчики организаций фильтруются из списка организаций, установленных на {@code campaignIds}
     */
    public Set<Long> getAvailableCounterIdsFromCampaignIdsForGoals(ClientId clientId, Collection<Long> campaignIds) {
        return getAvailableCounterIdsForGoals(clientId, () -> getSpravCounterIds(clientId, campaignIds));
    }

    /**
     * Доступные клиенту счетчики.
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к заданному списку информации о счетчиках {@code infoCounters}
     * информацию о счетчиках организаций этого клиента (только идентификатор и тип).
     * Счетчики организаций выбираются из БД
     */
    public Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersForGoals(ClientId clientId) {
        return getAvailableCountersForGoals(clientId, () -> getSpravCounterIds(clientId, null));
    }

    /**
     * Доступные клиенту счетчики.
     * если inputCounterIds не пустой, то проверяет доступность только счетчиков из списка
     * если inputCounterIds пустой, то возвращает все счетчики
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к заданному списку информации о счетчиках {@code infoCounters}
     * информацию о счетчиках организаций этого клиента (только идентификатор и тип).
     * Счетчики организаций фильтруются из заданного списка {@code inputCounterIds}
     * Их типы узнаем из Метрики, т.к. в БД их еще может не быть
     */
    public Set<MetrikaCounterWithAdditionalInformation> getAvailableAndFilterInputCountersInMetrikaForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds
    ) {
        return getAvailableCountersForGoals(clientId, inputCounterIds, this::filterSpravCounterIdsInMetrika);
    }

    public Set<MetrikaCounterWithAdditionalInformation> getAvailableAndFilterInputCountersInMetrikaForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds,
            boolean isStrictFilterByInputCounters
    ) {
        return getAvailableCountersForGoals(clientId, inputCounterIds, this::filterSpravCounterIdsInMetrika,
                isStrictFilterByInputCounters);
    }

    /**
     * Доступные клиенту счетчики.
     * <p>
     * Если у заданного клиента включена фича {@link FeatureName#GOALS_FROM_ALL_ORGS_ALLOWED},
     * то добавить к заданному списку информации о счетчиках {@code infoCounters}
     * информацию о счетчиках организаций этого клиента (только идентификатор и тип).
     * Счетчики организаций фильтруются из списка организаций, установленных на {@code campaignIds}
     */
    public Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersFromCampaignsForGoals(
            ClientId clientId,
            Collection<Long> campaignIds
    ) {
        return getAvailableCountersForGoals(clientId, () -> getSpravCounterIds(clientId, campaignIds));
    }

    /**
     * Получить счетчики по домену
     *
     * @param domain           юникодный домен в нижнем регистре
     * @param weakRestrictions если {@code true}, то используются ослабленные ограничения на аудиторию домена
     */
    public List<MetrikaCounterByDomain> getMetrikaCountersByDomain(
            ClientId clientId, String domain, boolean weakRestrictions) {
        if (StringUtils.isEmpty(domain)) {
            return emptyList();
        }

        boolean useHitLogTable = featureService.isEnabledForClientId(clientId,
                FeatureName.USE_COUNTERS_BY_DOMAIN_TABLE_BY_HIT_LOG);
        List<MetrikaCounterByDomain> counters = weakRestrictions ?
                metrikaCounterByDomainRepository.getAllCountersByDomain(domain, useHitLogTable) :
                metrikaCounterByDomainRepository.getRestrictedCountersByDomain(domain, useHitLogTable);
        return filterOutdatedCounters(counters, weakRestrictions);
    }

    private List<MetrikaCounterByDomain> filterOutdatedCounters(List<MetrikaCounterByDomain> allCounters,
                                                                boolean weakRestrictions) {
        long nowEpochDay = LocalDate.now().toEpochDay();
        int counterLifetimeInDays = weakRestrictions ?
                metrikaCounterWithWeakRestrictionsLifetimeInDaysProperty.getOrDefault(Integer.MAX_VALUE) :
                metrikaCounterLifetimeInDaysProperty.getOrDefault(Integer.MAX_VALUE);

        return filterList(allCounters, counter -> {
            long counterLastOccurrenceEpochDay =
                    LocalDateTime.ofEpochSecond(counter.getTimestamp(), 0, ZoneOffset.UTC).toLocalDate().toEpochDay();

            return nowEpochDay - counterLastOccurrenceEpochDay <= counterLifetimeInDays;
        });
    }

    public Map<Long, Boolean> getGrantAccessRequestedByCounterId(Long uid, Set<Long> counterIds) {
        if (counterIds.isEmpty()) {
            return emptyMap();
        }

        GrantAccessRequestStatusesRequest request = new GrantAccessRequestStatusesRequest()
                .withCounterIds(mapList(counterIds, identity()));

        GrantAccessRequestStatusesResponse response = metrikaClient.getGrantAccessRequestStatuses(uid, request);

        return StreamEx.of(response.getGrantAccessRequestStatuses())
                .toMap(GrantAccessRequestStatus::getCounterId, GrantAccessRequestStatus::getAccessRequested);
    }

    public RequestGrantsResponse requestMetrikaCountersAccess(String userLogin,
                                                              Collection<Long> counterIds) {
        List<RequestGrantsRequestItem> requestItems = mapList(counterIds,
                counterId -> new RequestGrantsRequestItem()
                        .withObjectId(counterId.toString())
                        .withObjectType(RequestGrantsObjectType.COUNTER)
                        .withPermission(RequestGrantsPermissionType.RO)
                        .withRequesterLogin(userLogin));
        RequestGrantsRequest request = new RequestGrantsRequest()
                .withRequestItems(requestItems);

        return metrikaClient.requestCountersGrants(request);
    }

    public List<GetExistentCountersResponseItem> getExistentCounters(Collection<Long> counterIds) {
        if (isEmpty(counterIds)) {
            return emptyList();
        }

        GetExistentCountersRequest request = new GetExistentCountersRequest()
                .withCounterIds(new ArrayList<>(counterIds));

        return metrikaClient.getExistentCounters(request).getResponseItems();
    }

    public Map<Long, Boolean> checkCountersEcommerce(Set<Long> counters) {
        try {
            Map<Long, Double> productImpressionsByCounterId =
                    metrikaClient.getProductImpressionsByCounterId(counters, 14);
            return listToMap(counters, Function.identity(),
                    counterId -> productImpressionsByCounterId.get(counterId) != null
                            && productImpressionsByCounterId.get(counterId) > 0);
        } catch (Exception exception) {
            logger.error("getProductImpressions error for counters {}", counters, exception);
            return emptyMap();
        }
    }

    private Set<Long> getPerformanceIds(ClientId clientId, List<Long> cids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Collection<CampaignSimple> campaigns = campaignRepository.getCampaignsSimple(shard, cids).values();
        return StreamEx.of(campaigns).filter(c -> c.getType().equals(CampaignType.PERFORMANCE))
                .map(CampaignSimple::getId).toSet();
    }

    private Map<Long, List<MetrikaCounter>> getCampMetrikaCountersForUpdate(
            Map<Long, List<Long>> countersByCid,
            Map<Long, MetrikaCounterWithAdditionalInformation> availableCountersById,
            Map<Long, Boolean> ecommerceFlagByCounter, Set<Long> performanceIds) {
        //поле has_ecommerce заполняется только для performance кампаний
        return EntryStream.of(countersByCid)
                .mapToValue((cid, counters) -> mapList(counters, c -> new MetrikaCounter()
                        .withId(c)
                        .withSource(getSource(availableCountersById, c))
                        .withHasEcommerce(performanceIds.contains(cid) ? ecommerceFlagByCounter.get(c) : null)))
                .toMap();
    }

    private MetrikaCounterSource getSource(Map<Long, MetrikaCounterWithAdditionalInformation> availableCountersById,
                                           Long counterId) {
        return Optional.ofNullable(availableCountersById.get(counterId))
                .map(MetrikaCounterWithAdditionalInformation::getSource)
                .orElse(MetrikaCounterSource.SPRAV);
    }

    public static Map<Long, List<MetrikaCounter>> calculateCountersByCampaignIdForNewMetrikaCounterIds(
            Map<Long, Set<Long>> newMetrikaCounterIdsByCampaignId,
            Map<Long, List<MetrikaCounter>> oldMetrikaCountersByCampaignId,
            Set<Long> organizationCounterIds) {

        Map<Long, MetrikaCounter> oldMetrikaCounterById =
                EntryStream.of(oldMetrikaCountersByCampaignId)
                        .values()
                        .flatMap(Collection::stream)
                        .distinct(MetrikaCounter::getId)
                        .mapToEntry(MetrikaCounter::getId,
                                identity())
                        .toMap();

        Map<Long, Boolean> hasEcommerceByCounterId = EntryStream.of(newMetrikaCounterIdsByCampaignId)
                .values()
                .flatMap(Collection::stream)
                .distinct()
                .mapToEntry(identity(), oldMetrikaCounterById::get)
                .mapValues(counter -> ifNotNull(counter,
                        MetrikaCounter::getHasEcommerce))
                .collect(nullFriendlyMapCollector());

        Map<Long, MetrikaCounterSource> sourceByCounterId = EntryStream.of(newMetrikaCounterIdsByCampaignId)
                .values()
                .flatMap(Collection::stream)
                .distinct()
                .mapToEntry(identity(), counterId -> getSource(counterId, oldMetrikaCounterById,
                        organizationCounterIds))
                .toMap();

        return EntryStream.of(newMetrikaCounterIdsByCampaignId)
                .removeValues(Set::isEmpty)
                .mapValues(counters -> mapList(counters, c -> new MetrikaCounter()
                        .withId(c)
                        .withSource(sourceByCounterId.get(c))
                        .withHasEcommerce(hasEcommerceByCounterId.get(c))))
                .toMap();
    }

    private static MetrikaCounterSource getSource(
            Long counterId,
            Map<Long, MetrikaCounter> oldMetrikaCounterById,
            Set<Long> organizationCounterIds) {
        if (organizationCounterIds.contains(counterId)) {
            return MetrikaCounterSource.SPRAV;
        }

        MetrikaCounter oldCounter =
                oldMetrikaCounterById.get(counterId);

        if (oldCounter == null || oldCounter.getSource() == null) {
            return MetrikaCounterSource.UNKNOWN;
        }

        return oldCounter.getSource();
    }

    private Set<Long> filterSpravCounterIds(ClientId clientId, Collection<Long> counterIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return campMetrikaCountersRepository.getActiveCounterIds(
                shard,
                clientId,
                null,
                counterIds,
                MetrikaCounterSource.SPRAV
        );
    }

    private Set<Long> filterSpravCounterIdsInMetrika(Collection<Long> counterIds) {
        return filterSpravCounterIdsInMetrika(getExistentCounters(counterIds));
    }

    public Set<Long> filterSpravCounterIdsInMetrika(List<GetExistentCountersResponseItem> existentCounters) {
        return filterAndMapToSet(existentCounters,
                it -> toMetrikaCounterSource(it.getCounterSource()) == MetrikaCounterSource.SPRAV,
                GetExistentCountersResponseItem::getCounterId);
    }

    private Set<Long> getSpravCounterIds(ClientId clientId, @Nullable Collection<Long> cids) {
        int shard = shardHelper.getShardByClientId(clientId);
        return campMetrikaCountersRepository.getActiveCounterIds(shard, clientId, cids, MetrikaCounterSource.SPRAV);
    }

    private Set<Long> getAvailableCounterIdsForGoals(ClientId clientId, Supplier<Set<Long>> counterFilter) {
        return getAvailableCounterIdsForGoals(clientId, emptyList(), unused -> counterFilter.get());
    }

    private Set<Long> getAvailableCounterIdsForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds,
            Function<Set<Long>, Set<Long>> counterFilter
    ) {
        return getAvailableCounterIdsForGoals(clientId, inputCounterIds, counterFilter, false);
    }

    private Set<Long> getAvailableCounterIdsForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds,
            Function<Set<Long>, Set<Long>> counterFilter,
            boolean isStrictFilterByInputCounters
    ) {
        Set<Long> availableCounterIds;
        if (!isStrictFilterByInputCounters && inputCounterIds.isEmpty()) {
            availableCounterIds = getAvailableCounterIdsByClientId(clientId);
        } else {
            availableCounterIds = getAvailableCounterIdsByClientId(clientId, inputCounterIds);
        }
        if (!featureService.isEnabledForClientId(clientId, FeatureName.GOALS_FROM_ALL_ORGS_ALLOWED)) {
            return availableCounterIds;
        }
        // Чтобы не обрабатывать счетчики, для которых мы уже получили информацию
        Set<Long> unavailableInputCounters = new HashSet<>(inputCounterIds);
        unavailableInputCounters.removeAll(availableCounterIds);
        Set<Long> spravCounterIds = counterFilter.apply(unavailableInputCounters);
        availableCounterIds.addAll(spravCounterIds);
        return availableCounterIds;
    }

    private Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersForGoals(
            ClientId clientId,
            Supplier<Set<Long>> counterFilter) {
        return getAvailableCountersForGoals(clientId, emptyList(), unused -> counterFilter.get());
    }

    private Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds,
            Function<Set<Long>, Set<Long>> counterFilter) {
        return getAvailableCountersForGoals(clientId, inputCounterIds, counterFilter, false);
    }

    private Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersForGoals(
            ClientId clientId,
            Collection<Long> inputCounterIds,
            Function<Set<Long>, Set<Long>> counterFilter,
            boolean isStrictFilterByInputCounters
    ) {
        Set<MetrikaCounterWithAdditionalInformation> availableCounters;
        if (!isStrictFilterByInputCounters && inputCounterIds.isEmpty()) {
            availableCounters = getAvailableCountersByClientId(clientId);
        } else {
            availableCounters = getAvailableCountersByClientId(clientId, inputCounterIds);
        }
        if (!featureService.isEnabledForClientId(clientId, FeatureName.GOALS_FROM_ALL_ORGS_ALLOWED)) {
            return availableCounters;
        }
        var availableCounterIds = mapSet(availableCounters, MetrikaCounterWithAdditionalInformation::getId);
        // Чтобы не обрабатывать счетчики, для которых мы уже получили информацию
        Set<Long> unavailableInputCounters = new HashSet<>(inputCounterIds);
        unavailableInputCounters.removeAll(availableCounterIds);
        Set<Long> spravCounterIds = counterFilter.apply(unavailableInputCounters);
        for (long id : spravCounterIds) {
            if (availableCounterIds.contains(id)) {
                continue;
            }
            var info = new MetrikaCounterWithAdditionalInformation()
                    .withId(id)
                    .withHasEcommerce(false)
                    .withSource(MetrikaCounterSource.SPRAV);
            availableCounters.add(info);
        }
        return availableCounters;
    }

    public static class CounterWithAdditionalInformationFilter {
        Set<MetrikaCounterSource> allowedSources;
        //отправляется в метрику
        private Collection<Long> counterIds;
        //отправляется в метрику, counterIds соединяется по or с counterName
        private String counterNamePrefix;
        private Set<Long> campaignIds;
        private Integer countersLimit;

        public static CounterWithAdditionalInformationFilter defaultFilter() {
            return new CounterWithAdditionalInformationFilter();
        }

        public CounterWithAdditionalInformationFilter withAllowedSources(Set<MetrikaCounterSource> allowedSources) {
            this.allowedSources = allowedSources;
            return this;
        }

        public CounterWithAdditionalInformationFilter withCounterIds(Collection<Long> counterIds) {
            this.counterIds = counterIds;
            return this;
        }

        public CounterWithAdditionalInformationFilter withCounterNamePrefix(String counterNamePrefix) {
            this.counterNamePrefix = counterNamePrefix;
            return this;
        }

        public CounterWithAdditionalInformationFilter withCampaignIds(Set<Long> campaignIds) {
            this.campaignIds = campaignIds;
            return this;
        }

        public CounterWithAdditionalInformationFilter withLimit(Integer countersLimit) {
            this.countersLimit = countersLimit;
            return this;
        }

        public boolean isSuitable(MetrikaCounterWithAdditionalInformation t) {
            if (allowedSources == null) {
                return true;
            }
            return t.getSource() == null && allowedSources.contains(MetrikaCounterSource.UNKNOWN)
                    ||
                    allowedSources.contains(t.getSource());
        }
    }

    public CountersContainer getAvailableCountersByClientAndCampaignId(
            ClientId clientId,
            Long operatorUid,
            Set<Long> campaignIds) {
        Map<Long, List<Long>> campaignsWithCounterIds = getCounterByCampaignIds(clientId, campaignIds);
        List<Long> counterIds = EntryStream.of(campaignsWithCounterIds).values().flatMap(Collection::stream).toList();
        return getAvailableCountersByClientAndOperatorUid(clientId, operatorUid, counterIds);
    }

    public CountersContainer getAvailableCountersByClientAndOperatorUid(
            ClientId clientId,
            Long operatorUid,
            CounterWithAdditionalInformationFilter counterFilter) {
        Set<Long> clientRepresentativesUids =
                new HashSet<>(rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId));
        List<Long> allUids =
                StreamEx.of(clientRepresentativesUids).append(operatorUid).nonNull().distinct().toList();
        Set<MetrikaCounterWithAdditionalInformation> availableCounters =
                getAvailableCountersByUids(allUids, counterFilter);
        return getCountersContainer(clientRepresentativesUids, operatorUid, availableCounters);
    }

    public CountersContainer getAvailableExtendedCounters(
            ClientId clientId,
            Long operatorUid,
            CounterWithAdditionalInformationFilter counterFilter) {
        Set<Long> clientRepresentativesUids =
                new HashSet<>(rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId));
        List<Long> allUids =
                StreamEx.of(clientRepresentativesUids).append(operatorUid).nonNull().distinct().toList();
        if (counterFilter.campaignIds != null) {
            return getAvailableCountersByClientAndCampaignId(clientId, operatorUid, counterFilter.campaignIds);
        } else {
            var availableCounters =
                    getAvailableExtendedCountersWithFilter(allUids, counterFilter);
            return getCountersContainer(clientRepresentativesUids, operatorUid, availableCounters);
        }
    }

    public CountersContainer getAvailableCountersByClientAndOperatorUid(
            ClientId clientId,
            Long operatorUid) {
        return getAvailableCountersByClientAndOperatorUid(
                clientId,
                operatorUid,
                CampMetrikaCountersService.CounterWithAdditionalInformationFilter.defaultFilter());
    }

    public CountersContainer getAvailableCountersByClientAndOperatorUid(
            ClientId clientId,
            Long operatorUid,
            Collection<Long> counterIds) {
        return getAvailableCountersByClientAndOperatorUid(
                clientId,
                operatorUid,
                CampMetrikaCountersService.CounterWithAdditionalInformationFilter.defaultFilter()
                        .withCounterIds(counterIds));
    }

    /*
     * Не возвращаются удаленные счетчики и
     * счетчики у которых в Метрике не разрешено использование без доступа
     */
    public Set<GetExistentCountersResponseItem> getAllowedInaccessibleCounters(Set<Long> inaccessibleCounterIds) {
        if (isEmpty(inaccessibleCounterIds)) {
            return emptySet();
        }
        List<GetExistentCountersResponseItem> existentCounters = getExistentCounters(inaccessibleCounterIds);
        Set<Long> existentCounterIds = listToSet(existentCounters, GetExistentCountersResponseItem::getCounterId);
        Set<Long> notExistentCounterIds = Sets.difference(inaccessibleCounterIds, existentCounterIds);
        if (!notExistentCounterIds.isEmpty()) {
            logger.info("There are not existent inaccessible counters: {}", notExistentCounterIds);
        }

        Set<GetExistentCountersResponseItem> allowedInaccessibleCounters =
                getAllowedByMetrikaInaccessibleCounters(existentCounters);
        Set<Long> allowedInaccessibleCounterIds =
                mapSet(allowedInaccessibleCounters, GetExistentCountersResponseItem::getCounterId);
        Set<Long> notAllowedByMetrikaInaccessibleCounterIds =
                Sets.difference(existentCounterIds, allowedInaccessibleCounterIds);
        if (!notAllowedByMetrikaInaccessibleCounterIds.isEmpty()) {
            logger.info("There are not allowed by metrika settings inaccessible counters: {}",
                    notAllowedByMetrikaInaccessibleCounterIds);
        }
        return allowedInaccessibleCounters;
    }

    public Set<Long> getAllowedInaccessibleCounterIds(Set<Long> inaccessibleCounterIds) {
        return mapSet(getAllowedInaccessibleCounters(inaccessibleCounterIds),
                GetExistentCountersResponseItem::getCounterId);
    }

    private Set<GetExistentCountersResponseItem> getAllowedByMetrikaInaccessibleCounters(
            List<GetExistentCountersResponseItem> existentInaccessibleCounters) {
        return StreamEx.of(existentInaccessibleCounters)
                .filter(c -> nvl(c.getAllowUseGoalsWithoutAccess(), true))
                .toSet();
    }

    public boolean isDomainNotAllowedForUnavailableCounters(@Nullable String domain) {
        return domain != null && domainsNotAllowedForUnavailableCounters.getOrDefault(Set.of()).contains(domain);
    }

    private CountersContainer getCountersContainer(Set<Long> clientRepresentativesUids,
                                                   Long operatorUid,
                                                   Set<MetrikaCounterWithAdditionalInformation> availableCounters) {
        return getCountersContainer(clientRepresentativesUids, operatorUid,
                new MetrikaCounterWithAdditionalInformationContainer()
                        .withCounters(availableCounters)
                        .withHasMoreCounters(false));
    }

    private CountersContainer getCountersContainer(Set<Long> clientRepresentativesUids,
                                                   Long operatorUid,
                                                   MetrikaCounterWithAdditionalInformationContainer container) {
        Set<MetrikaCounterWithAdditionalInformation> clientAvailableCounters = new HashSet<>();
        Map<Long, MetrikaCounterPermission> operatorPermissionByCounterId = new HashMap<>();
        Set<Long> operatorEditableCounterIds = new HashSet<>();
        for (MetrikaCounterWithAdditionalInformation counter : container.getCounters()) {
            Set<Long> uids = counter.getPermissionsByUid().keySet();
            if (operatorUid != null && uids.contains(operatorUid)) {
                var permission = counter.getPermissionsByUid().get(operatorUid);
                operatorPermissionByCounterId.put(counter.getId(), permission);
                if (COUNTER_PERMISSIONS_TREATED_AS_EDITABLE.contains(permission)) {
                    operatorEditableCounterIds.add(counter.getId());
                }
            }
            if (CollectionUtils.containsAny(clientRepresentativesUids, uids)) {
                clientAvailableCounters.add(counter);
            }
        }
        return new CountersContainer(clientAvailableCounters,
                operatorPermissionByCounterId,
                operatorEditableCounterIds, container.getHasMoreCounters());
    }

    public static class CountersContainer {
        private final Set<MetrikaCounterWithAdditionalInformation> clientAvailableCounters;
        private final Map<Long, MetrikaCounterPermission> operatorPermissionByCounterId;
        private final Set<Long> operatorEditableCounterIds;
        private final Boolean hasMoreCounters;

        public CountersContainer(Set<MetrikaCounterWithAdditionalInformation> clientAvailableCounters,
                                 Map<Long, MetrikaCounterPermission> operatorPermissionByCounterId,
                                 Set<Long> operatorEditableCounterIds) {
            this(clientAvailableCounters, operatorPermissionByCounterId, operatorEditableCounterIds, false);
        }

        public CountersContainer(Set<MetrikaCounterWithAdditionalInformation> clientAvailableCounters,
                                 Map<Long, MetrikaCounterPermission> operatorPermissionByCounterId,
                                 Set<Long> operatorEditableCounterIds,
                                 Boolean hasMoreCounters) {
            this.clientAvailableCounters = clientAvailableCounters;
            this.operatorPermissionByCounterId = operatorPermissionByCounterId;
            this.operatorEditableCounterIds = operatorEditableCounterIds;
            this.hasMoreCounters = hasMoreCounters;
        }

        public Set<MetrikaCounterWithAdditionalInformation> getClientAvailableCounters() {
            return clientAvailableCounters;
        }

        public Map<Long, MetrikaCounterPermission> getOperatorPermissionByCounterId() {
            return operatorPermissionByCounterId;
        }

        public Set<Long> getOperatorEditableCounterIds() {
            return operatorEditableCounterIds;
        }

        public Boolean getHasMoreCounters() {
            return hasMoreCounters;
        }
    }

    public boolean hasAvailableCounters(String url, ClientId clientId) {
        var availableCounters = getAvailableCountersForGoals(clientId);
        return availableCounters.stream().anyMatch(c -> c.getDomain().equals(url));
    }

    private List<AppliedChanges<StrategyWithMetrikaCounters>> getAllChangesInStrategiesMetrikaCounters(
            int shard,
            ClientId clientId,
            List<Long> cids,
            Map<Long, List<Long>> newListsOfMetrikaCountersByCids
    ) {
        var strategyIdByCampaignId = campaignRepository.getStrategyIdsByCampaignIds(shard, clientId, cids);
        var campaignIdsByStrategyId = EntryStream.of(strategyIdByCampaignId)
                .invert()
                .grouping();

        var strategiesWithMetrikaCountersByStrategyId = StreamEx
                .of(strategyTypedRepository.getTyped(shard, strategyIdByCampaignId.values()))
                .select(StrategyWithMetrikaCounters.class)
                .toMap(StrategyWithMetrikaCounters::getId,
                        Function.identity());

        return EntryStream.of(strategiesWithMetrikaCountersByStrategyId)
                .mapKeyValue((sid, actualStrategy) ->
                        getStrategyWithMetrikaCountersAppliedChanges(
                                newListsOfMetrikaCountersByCids,
                                campaignIdsByStrategyId,
                                sid,
                                actualStrategy)
                )
                .toList();
    }

    private AppliedChanges<StrategyWithMetrikaCounters> getStrategyWithMetrikaCountersAppliedChanges(
            Map<Long, List<Long>> newListsOfMetrikaCountersByCids,
            Map<Long, List<Long>> campaignIdsByStrategyId,
            Long sid,
            StrategyWithMetrikaCounters actualStrategy) {
        var metrikaCounters = new HashSet<>(actualStrategy.getMetrikaCounters());

        var cidsForStrategy = campaignIdsByStrategyId.getOrDefault(sid, List.of());

        var actualCounters = Optional.of(cidsForStrategy)
                .filter(not(List::isEmpty))
                .map(cids -> cids.get(0))
                .map(cid -> newListsOfMetrikaCountersByCids.getOrDefault(cid, List.of()))
                .orElse(List.copyOf(metrikaCounters));

        return ModelChanges.build(
                        actualStrategy,
                        StrategyWithMetrikaCounters.METRIKA_COUNTERS,
                        actualCounters)
                .applyTo(actualStrategy);
    }

    private StrategyRepositoryContainer collectStrategiesInContainer(
            int shard,
            ClientId clientId,
            Map<Long, ? extends Collection<MetrikaCounter>> metrikaCountersByCid) {
        var userCountersExtendedById = StreamEx.of(metrikaCountersByCid.values())
                .map(counters -> mapList(
                        counters,
                        counter -> new CounterInfoDirect()
                                .withCounterSource(nvl(counter.getSource(), MetrikaCounterSource.UNKNOWN).toString())
                                .withId(counter.getId().intValue())
                                .withEcommerce(nvl(counter.getHasEcommerce(), false))))
                .flatMap(Collection::stream)
                .distinct()
                .mapToEntry(
                        CounterInfoDirect::getId,
                        Function.identity()
                )
                .mapKeys(Integer::longValue)
                .toMap();
        return new StrategyRepositoryContainer(
                shard,
                clientId,
                userCountersExtendedById,
                false);
    }

    // TODO: ожидаются улучшения в тикете DIRECT-164754
    public void updateMetrikaCounterForStrategies(int shard,
                                                  ClientId clientId,
                                                  Map<Long, List<MetrikaCounter>> countersForUpdate,
                                                  List<Long> validCids,
                                                  Map<Long, List<Long>> newMetrikaCountersByValidCid) {
        StrategyRepositoryContainer strategiesContainer =
                collectStrategiesInContainer(shard, clientId, countersForUpdate);
        Collection<AppliedChanges<StrategyWithMetrikaCounters>> changesToStrategiesWithCounters =
                getAllChangesInStrategiesMetrikaCounters(
                        shard,
                        clientId,
                        validCids,
                        newMetrikaCountersByValidCid);
        campMetrikaCountersRepository.updateMetrikaCountersToCampaignsAndStrategies(
                shard,
                countersForUpdate,
                strategiesContainer,
                changesToStrategiesWithCounters);
    }

}
