package ru.yandex.direct.core.entity.campaign.service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.SetUtils;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetCpaAlertRepository;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetHourlyAlertRepository;
import ru.yandex.direct.core.entity.balance.model.BalanceNotificationInfo;
import ru.yandex.direct.core.entity.balance.service.BalanceInfoQueueService;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.mailnotification.model.GenericEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;

@Service
@ParametersAreNonnullByDefault
public class CampaignAdditionalActionsService {

    private final BannerRelationsRepository bannerRelationsRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final BalanceInfoQueueService balanceInfoQueueService;
    private final MailNotificationEventService mailNotificationEventService;
    private final AdGroupRepository adGroupRepository;
    private final AutobudgetHourlyAlertRepository autobudgetHourlyAlertRepository;
    private final AutobudgetCpaAlertRepository autobudgetCpaAlertRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;

    @Autowired
    public CampaignAdditionalActionsService(BannerRelationsRepository bannerRelationsRepository,
                                            BannerCommonRepository bannerCommonRepository,
                                            BsResyncQueueRepository bsResyncQueueRepository,
                                            BalanceInfoQueueService balanceInfoQueueService,
                                            MailNotificationEventService mailNotificationEventService,
                                            AdGroupRepository adGroupRepository,
                                            AutobudgetHourlyAlertRepository autobudgetHourlyAlertRepository,
                                            AutobudgetCpaAlertRepository autobudgetCpaAlertRepository,
                                            AggregatedStatusesRepository aggregatedStatusesRepository) {
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.balanceInfoQueueService = balanceInfoQueueService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.adGroupRepository = adGroupRepository;
        this.autobudgetHourlyAlertRepository = autobudgetHourlyAlertRepository;
        this.autobudgetCpaAlertRepository = autobudgetCpaAlertRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
    }

    public void processAdditionalActionsContainer(DSLContext dslContext, Long operatorUid,
                                                  CampaignAdditionalActionsContainer container) {
        resetAdGroupStatusBsSynced(dslContext, container);
        Set<Long> campaignIdsWhichBannersBsSyncedStatusWereReset = resetBannersBsStatusSynced(dslContext, container);

        var campaignIdsWithPriority = getCampaignIdsWithPriorityForAddBannersToBsResyncQueue(container,
                campaignIdsWhichBannersBsSyncedStatusWereReset);
        if (!campaignIdsWithPriority.isEmpty()) {
            addBannersToBsResyncQueue(dslContext, campaignIdsWithPriority);
        }

        Set<Long> idsForUpdateBannersLastChange = container.getCampaignIdsForUpdateBannersLastChange()
                .stream()
                // Нет смысла обновлять LastChange по некоторым идентификаторам дважды:
                .filter(id -> !container.getCampaignIdsForResetBannersStatusBsSyncedAndUpdateLastChange().contains(id))
                .collect(Collectors.toSet());
        if (!idsForUpdateBannersLastChange.isEmpty()) {
            bannerCommonRepository.updateBannersLastChangeByCampaignIds(dslContext, idsForUpdateBannersLastChange);
        }

        List<BalanceNotificationInfo> balanceNotifications = container.getBalanceNotifications();
        if (!balanceNotifications.isEmpty()) {
            balanceInfoQueueService.addToBalanceInfoQueue(dslContext, balanceNotifications);
        }

        List<GenericEvent> mailEvents = container.getMailEvents();
        if (!mailEvents.isEmpty()) {
            mailNotificationEventService.queueEvents(dslContext, operatorUid, mailEvents);
        }

        Set<Long> idsForUpdateStatusShowsForecast = container.getCampaignIdsForUpdateAdGroupStatusShowsForecast();
        if (!idsForUpdateStatusShowsForecast.isEmpty()) {
            adGroupRepository.updateStatusShowsForecastByCampaignIds(dslContext.configuration(),
                    idsForUpdateStatusShowsForecast, StatusShowsForecast.NEW);
        }

        Set<Long> idsForMarkAggregatedStatusesAsObsolete =
                container.getCampaignIdsForMarkAggregatedStatusesAsObsolete();
        if (!idsForMarkAggregatedStatusesAsObsolete.isEmpty()) {
            aggregatedStatusesRepository.markCampaignStatusesAsObsolete(dslContext, null,
                    idsForMarkAggregatedStatusesAsObsolete);
        }

        autobudgetCpaAlertRepository.freezeAlerts(dslContext, container.getCampaignsToFreezeCpaAlerts());
        autobudgetHourlyAlertRepository.freezeAlerts(dslContext, container.getCampaignsToFreezeAlerts());
    }

    public Set<Long> resetBannersBsStatusSynced(DSLContext dslContext, CampaignAdditionalActionsContainer container) {
        Set<Long> campaignIdsForResetBannersStatusBsSynced = container.getCampaignIdsForResetBannersStatusBsSynced();
        bannerCommonRepository.updateStatusBsSyncedByCampaignIds(dslContext.configuration(),
                campaignIdsForResetBannersStatusBsSynced, false, StatusBsSynced.NO);

        Set<Long> campaignIdsForResetBannersStatusBsSyncedAndUpdateLastChange =
                new HashSet<>(container.getCampaignIdsForResetBannersStatusBsSyncedAndUpdateLastChange());
        //удаляем баннеры для которых выше сбросили статус
        campaignIdsForResetBannersStatusBsSyncedAndUpdateLastChange.removeAll(campaignIdsForResetBannersStatusBsSynced);
        bannerCommonRepository.updateStatusBsSyncedByCampaignIds(dslContext.configuration(),
                campaignIdsForResetBannersStatusBsSyncedAndUpdateLastChange, true, StatusBsSynced.NO);

        return SetUtils.union(campaignIdsForResetBannersStatusBsSynced,
                campaignIdsForResetBannersStatusBsSyncedAndUpdateLastChange);
    }

    public void resetAdGroupStatusBsSynced(DSLContext dslContext, CampaignAdditionalActionsContainer container) {
        var campaignIdsForResetAdGroupStatusBsSynced = container.getCampaignIdsForResetAdGroupStatusBsSynced();
        adGroupRepository.updateStatusBsSyncedByCampaignIds(dslContext.configuration(),
                campaignIdsForResetAdGroupStatusBsSynced, StatusBsSynced.NO);
    }

    public void addBannersToBsResyncQueue(DSLContext dslContext, Map<Long, BsResyncPriority> campaignIdsWithPriority) {
        Map<Long, List<Long>> bannerIdsByCampaignIds =
                bannerRelationsRepository.getBannerIdsByCampaignIdsMap(dslContext, campaignIdsWithPriority.keySet());

        List<BsResyncItem> bsResyncItems = EntryStream.of(bannerIdsByCampaignIds)
                .flatMapValues(StreamEx::of)
                .mapKeyValue((campaignId, bannerId) ->
                        new BsResyncItem(campaignIdsWithPriority.get(campaignId), campaignId, bannerId, null))
                .toList();

        bsResyncQueueRepository.addToResync(dslContext, bsResyncItems);
    }

    private static Map<Long, BsResyncPriority> getCampaignIdsWithPriorityForAddBannersToBsResyncQueue(
            CampaignAdditionalActionsContainer container,
            Set<Long> campaignIdsWhichBannersBsSyncedStatusWereReset) {
        return EntryStream.of(container.getCampaignIdsWithPriorityForAddBannersToBsResyncQueue())
                //удаляем кампании, чьим баннерам выше сбросили статус
                .removeKeys(campaignIdsWhichBannersBsSyncedStatusWereReset::contains)
                .toMap();
    }
}
