package ru.yandex.direct.core.entity.campaign.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartData
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartResultSuccess
import ru.yandex.direct.autobudget.restart.service.AutobudgetRestartService
import ru.yandex.direct.autobudget.restart.service.CampaignAutobudgetRestartContainer
import ru.yandex.direct.autobudget.restart.service.StrategyState
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.strategy.service.PrivatePackageStrategyAutobudgetRestartService
import ru.yandex.direct.core.entity.strategy.service.PublicPackageStrategyAutobudgetRestartService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.feature.FeatureName

/**
 * Сервис для вычисления рестарта автобюджета кампаний с учетом привязанности к пакетной стратегии.
 *
 * Для кампании привязанной к приватной стратегии время рестарта вычисляется через [AutobudgetRestartService].
 *
 * Для кампании привязанной к публичной стратегии время рестарта вычисляется через [PublicPackageStrategyAutobudgetRestartService].
 * При это выдерживается инвариант, что для всех кампаний под этим публичным пакетом время рестарта будет одинаковым.
 *
 * Больше деталей тут https://st.yandex-team.ru/DIRECT-164058#6254394d9aaac1660687876e
 * */
@Service
class CampaignAutobudgetRestartService(
    private val shardHelper: ShardHelper,
    private val campAutobudgetRestartService: AutobudgetRestartService,
    private val packageStrategyAutobudgetRestartService: PublicPackageStrategyAutobudgetRestartService,
    private val privateStrategyAutobudgetRestartService: PrivatePackageStrategyAutobudgetRestartService,
    private val strategyTypedRepository: StrategyTypedRepository,
    private val featureService: FeatureService,
    private val dslProvider: DslContextProvider
) {

    fun calculateRestartsAndSave(data: List<CampStrategyRestartData>): List<CampStrategyRestartContainer> {
        return shard(data, { it.cid }) { shard, shardedData ->
            calculateRestartsAndSave(shard, shardedData)
        }
    }

    fun calculateRestartsAndSave(
        shard: Int,
        shardedData: List<CampStrategyRestartData>
    ): List<CampStrategyRestartContainer> {
        val featureValueByCampaignId = getFeatureValueByCampaignId(shardedData.map { it.cid }.toSet())
        val partitioner = partitioner(shard, shardedData, featureValueByCampaignId)
        val campaignWithPrivateStrategyRestarts =
            calculateRestartsAndSaveForCampaignsWithPrivateStrategies(shard, partitioner)
        val campaignWithPublicStrategyRestarts = calculateRestartsAndSaveForPublicStrategies(shard, partitioner)
        return campaignWithPublicStrategyRestarts + campaignWithPrivateStrategyRestarts
    }

    private fun calculateRestartsAndSaveForCampaignsWithPrivateStrategies(
        shard: Int,
        partitioner: Partitioner
    ): List<CampStrategyRestartContainer> {
        val campaignRestartContainers = mutableListOf<CampaignAutobudgetRestartContainer>()
        logger.info("Start calculate campaigns restarts")
        dslProvider.ppcTransaction(shard) { configuration ->
            val dsl = configuration.dsl()
            campaignRestartContainers += campAutobudgetRestartService.calculateRestartsAndSave(
                dsl,
                partitioner.withPrivateOrUndefinedStrategy
            )
            privateStrategyAutobudgetRestartService.spreadRestartsToStrategies(
                dsl,
                partitioner.privateStrategyByCampaignId,
                campaignRestartContainers
            )
        }

        return campaignRestartContainers.map {
            CampStrategyRestartContainer(
                it.restartResult,
                it.restartDbData.state
            )
        }
    }

    private fun calculateRestartsAndSaveForPublicStrategies(
        shard: Int,
        partitioner: Partitioner
    ): List<CampStrategyRestartContainer> {
        logger.info("Start calculate strategies restarts")
        val strategyRestarts = packageStrategyAutobudgetRestartService.recalculateAndSaveRestarts(
            shard,
            partitioner.withPublicStrategy.values.distinctBy { it.id }
        )
        return strategyRestarts.mapNotNull { strategyRestart ->
            partitioner.campaignIdsByStrategyId[strategyRestart.strategyId]?.let { cids ->
                cids.map { cid ->
                    CampStrategyRestartContainer(
                        CampStrategyRestartResultSuccess(
                            cid,
                            strategyRestart.restartTime,
                            strategyRestart.softRestartTime,
                            strategyRestart.restartReason
                        ),
                        strategyRestart.state
                    )
                }
            }
        }.flatten()
    }

    private fun <Data, Result> shard(
        data: List<Data>,
        shard: (Data) -> Long,
        f: (Int, List<Data>) -> List<Result>
    ): List<Result> =
        shardHelper
            .groupByShard(data, ShardKey.CID) { shard(it) }
            .shardedDataMap
            .map { (shard, shardedData) ->
                f(shard, shardedData)
            }.flatten()

    internal class Partitioner(
        dataList: List<PartitionObject>,
        strategies: List<BaseStrategy>
    ) {
        private val strategiesById = strategies.mapNotNull { it as? CommonStrategy }.associateBy { it.id }

        val withPublicStrategy = dataList.mapNotNull { data ->
            val strategy = data.strategyId?.let(strategiesById::get)
            strategy
                ?.takeIf { it.isPublic }
                ?.let { data.cid to it }
        }.toMap()

        val campaignIdsByStrategyId = withPublicStrategy.map { (cid, strategy) ->
            strategy.id to cid
        }.groupBy({ it.first }) { it.second }

        val withPrivateOrUndefinedStrategy: List<CampStrategyRestartData> = dataList.filter { data ->
            val strategy = data.strategyId?.let(strategiesById::get)
            strategy == null || !strategy.isPublic
        }.map { it.data }

        val privateStrategyByCampaignId: Map<Long, CommonStrategy> = dataList.mapNotNull { data ->
            val strategy = data.strategyId?.let(strategiesById::get)
            strategy
                ?.takeIf { !it.isPublic }
                ?.let { data.cid to it }
        }.toMap()
    }

    private fun partitioner(
        shard: Int,
        data: List<CampStrategyRestartData>,
        featureValueByCampaignId: Map<Long, Boolean>
    ): Partitioner {
        val objects = data.map { campRestart(it, featureValueByCampaignId[it.cid] ?: false) }
        val strategyIds = objects.mapNotNull { it.strategyId }
        val strategies = if (strategyIds.isNotEmpty()) {
            strategyTypedRepository.getTyped(shard, strategyIds)
        } else listOf()

        return Partitioner(objects, strategies)
    }

    private fun getFeatureValueByCampaignId(campaignIds: Set<Long>): Map<Long, Boolean> {
        val campaignToClientId =
            shardHelper.getClientIdsByCampaignIds(campaignIds).mapValues { ClientId.fromLong(it.value) }
        val featureMap = featureService.isEnabledForClientIdsOnlyFromDb(
            campaignToClientId.values.toSet(),
            FeatureName.AUTOBUDGET_RESTART_WITH_PACKAGE_STRATEGY_SUPPORT_ENABLED.getName()
        )
        return campaignToClientId.mapValues { (_, clientId) ->
            featureMap[clientId] ?: false
        }
    }

    companion object {
        data class PartitionObject(
            val cid: Long,
            val data: CampStrategyRestartData,
            val strategyId: Long?
        )

        fun campRestart(
            data: CampStrategyRestartData,
            isFeatureEnabled: Boolean
        ): PartitionObject {
            val strategyId = data.strategyDto.strategyId?.takeIf { it != 0L && isFeatureEnabled }
            return PartitionObject(data.cid, data, strategyId)
        }

        data class CampStrategyRestartContainer(
            val restartResult: CampStrategyRestartResultSuccess,
            val state: StrategyState
        )

        private val logger = LoggerFactory.getLogger(CampaignAutobudgetRestartService::class.java)
    }
}
