package ru.yandex.direct.core.entity.campaign.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.model.request.CreateOrUpdateOrdersBatchItem;
import ru.yandex.direct.balance.client.model.request.CreateOrUpdateOrdersBatchRequest;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForBlockedMoneyCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.type.add.UidWithCampaignIds;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.product.model.ProductSimple;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.balance.client.model.request.CreateOrUpdateOrdersBatchItem.BALANCE_MAX_ORDER_NAME_LENGTH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DISALLOW_CAMPAIGN_NAME_LETTERS_RE;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CampaignBalanceService {
    private static final String DEFAULT_ORDER_NAME = "Новая";

    private final CampaignService campaignService;
    private final ProductService productService;
    private final BalanceService balanceService;
    private final RbacService rbacService;
    private final ClientService clientService;

    @Autowired
    public CampaignBalanceService(CampaignService campaignService, ProductService productService,
                                  BalanceService balanceService, RbacService rbacService, ClientService clientService) {
        this.campaignService = campaignService;
        this.productService = productService;
        this.balanceService = balanceService;
        this.rbacService = rbacService;
        this.clientService = clientService;
    }

    /**
     * Перенесена часть логики из protected/Campaign.pm:3432, касающаяся создания первой кампании
     */
    public boolean createOrUpdateOrdersOnFirstCampaignCreation(
            Long operatorUid,
            @Nullable UidAndClientId agencyUidAndClientId,
            @Nullable UidWithCampaignIds managerUidWithServicedCampaignIds,
            List<? extends CommonCampaign> campaigns) {
        Long agencyManager = null;
        if (agencyUidAndClientId != null) {
            agencyManager = Optional.ofNullable(getAgencyManager(agencyUidAndClientId))
                    .orElseThrow(() -> new IllegalStateException("There is no manager for agency with uid: " +
                            agencyUidAndClientId.getUid()));
        }

        CreateOrUpdateOrdersBatchRequest request = new CreateOrUpdateOrdersBatchRequest();
        request.setItems(getCreateOrUpdateOrdersBatchItems(agencyUidAndClientId, managerUidWithServicedCampaignIds,
                agencyManager, campaigns));
        request.setOperatorUid(operatorUid);
        return balanceService.createOrUpdateOrders(request);
    }

    @Nullable
    private Long getAgencyManager(UidAndClientId agencyUidAndClientId) {
        Client client = clientService.getClient(agencyUidAndClientId.getClientId());
        Long primaryDirectOrBayanManagerUid = nullableNvl(client.getPrimaryManagerUid(),
                client.getPrimaryBayanManagerUid());
        List<Long> managers = rbacService.getManagersOfUser(agencyUidAndClientId.getUid());

        if (managers.contains(primaryDirectOrBayanManagerUid) || client.getIsIdmPrimaryManager()) {
            return primaryDirectOrBayanManagerUid;
        }

        return null;
    }

    private List<CreateOrUpdateOrdersBatchItem> getCreateOrUpdateOrdersBatchItems(
            @Nullable UidAndClientId agencyUidAndClientId,
            @Nullable UidWithCampaignIds managerUidWithServicedCampaignIds,
            @Nullable Long agencyManager,
            List<? extends CommonCampaign> campaigns) {
        Map<Long, Boolean> isBlockedByCampaignId = campaignService.moneyOnCampaignsIsBlocked(
                mapList(campaigns, CampaignBalanceService::toCampaignForBlockedMoneyCheck),
                false, true, true);
        return mapList(campaigns, c -> getCreateOrUpdateOrdersBatchItem(agencyUidAndClientId,
                managerUidWithServicedCampaignIds, agencyManager, c, isBlockedByCampaignId.get(c.getId())));
    }

    private static CampaignForBlockedMoneyCheck toCampaignForBlockedMoneyCheck(CommonCampaign campaign) {
        return new Campaign()
                .withId(campaign.getId())
                .withType(campaign.getType())
                .withUserId(campaign.getUid())
                .withWalletId(campaign.getWalletId())
                .withAgencyId(campaign.getAgencyId())
                .withManagerUserId(campaign.getManagerUid())
                .withStatusModerate(campaign.getStatusModerate())
                .withStatusPostModerate(campaign.getStatusPostModerate())
                .withSum(campaign.getSum())
                .withSumToPay(campaign.getSumToPay());
    }

    private CreateOrUpdateOrdersBatchItem getCreateOrUpdateOrdersBatchItem(
            @Nullable UidAndClientId agencyUidAndClientId,
            @Nullable UidWithCampaignIds managerUidWithServicedCampaigns,
            @Nullable Long agencyManager,
            CommonCampaign campaign,
            Boolean isMoneyBlocked) {
        checkArgument(!CampaignTypeKinds.NO_SEND_TO_BILLING.contains(campaign.getType()),
                "Campaign payment is prohibited");

        ProductSimple product = productService.getProductById(campaign.getProductId());

        Long managerUid = null;
        if (agencyManager != null) {
            managerUid = agencyManager;
        } else if (managerUidWithServicedCampaigns != null) {
            Set<Long> servicedCampaignIds = listToSet(managerUidWithServicedCampaigns.getCampaignIds());
            if (servicedCampaignIds.contains(campaign.getId())) {
                managerUid = managerUidWithServicedCampaigns.getUid();
            }
        }

        CreateOrUpdateOrdersBatchItem item = new CreateOrUpdateOrdersBatchItem();
        item.setServiceOrderId(campaign.getId());
        item.setServiceId(product.getEngineId());
        item.setProductId(product.getId());
        item.setClientId(campaign.getClientId());
        item.setText(calculateName(campaign));
        item.setUnmoderated(calculatePayMethodFlag(campaign, isMoneyBlocked));
        item.setGroupServiceOrderId(campaign.getWalletId());
        item.setManagerUid(managerUid);

        if (agencyUidAndClientId != null) {
            item.setAgencyId(agencyUidAndClientId.getClientId().asLong());
        } else if (CampaignType.WALLET == campaign.getType() && campaign.getCurrency() != CurrencyCode.YND_FIXED) {
            item.setIsUaOptimize("1");
        }

        return item;
    }

    private static String calculateName(CommonCampaign campaign) {
        String name = campaign.getName()
                .substring(0, Math.min(BALANCE_MAX_ORDER_NAME_LENGTH, campaign.getName().length()))
                .replaceAll(DISALLOW_CAMPAIGN_NAME_LETTERS_RE, "");

        return name.isEmpty() ? DEFAULT_ORDER_NAME : name;
    }

    /**
     * Флаг обозначающий возможность оплачивать кампанию
     * 0 - можно, 1 - нельзя
     */
    private static Integer calculatePayMethodFlag(CommonCampaign campaign, Boolean isMoneyBlocked) {
        if (CampaignTypeKinds.BILLING_AGGREGATE.contains(campaign.getType())
                || campaign.getType() == CampaignType.INTERNAL_AUTOBUDGET) {
            return 0;
        }

        return isMoneyBlocked ? 1 : 0;
    }
}
