package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.BasicUplift;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStopReason;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBrandSurveyYtRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignBudgetReachDailyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.dbqueue.LimitOffset;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.RECALC_BRAND_LIFT_CAMPAIGNS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

@ParametersAreNonnullByDefault
@Component
public class CampaignBudgetReachService {
    protected static final LimitOffset JOBS_LIMIT = LimitOffset.maxLimited();

    private final CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository;
    private final CampaignBrandSurveyYtRepository campaignBrandSurveyYtRepository;
    private final CampaignRepository campaignRepository;
    private final DbQueueRepository dbQueueRepository;
    private final PricePackageService pricePackageService;
    private final BrandSurveyConditionsService brandSurveyConditionsService;

    public CampaignBudgetReachService(
            CampaignBudgetReachDailyRepository campaignBudgetReachDailyRepository,
            CampaignBrandSurveyYtRepository campaignBrandSurveyYtRepository,
            CampaignRepository campaignRepository,
            DbQueueRepository dbQueueRepository,
            PricePackageService pricePackageService,
            BrandSurveyConditionsService brandSurveyConditionsService
    ) {
        this.campaignBudgetReachDailyRepository = campaignBudgetReachDailyRepository;
        this.campaignBrandSurveyYtRepository = campaignBrandSurveyYtRepository;
        this.campaignRepository = campaignRepository;
        this.dbQueueRepository = dbQueueRepository;
        this.pricePackageService = pricePackageService;
        this.brandSurveyConditionsService = brandSurveyConditionsService;
    }

    public Map<Long, BrandSurveyStatus> getBrandStatusForCampaigns(int shard, ClientId clientId,
                                                                   List<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        var campaignIdToBrandSurveyIdMap =
                campaignRepository.getBrandSurveyIdsForCampaigns(shard, campaignIds);
        return getBrandStatusForCampaigns(shard, clientId, campaignIdToBrandSurveyIdMap);
    }

    public Map<Long, BrandSurveyStatus> getBrandStatusForCampaigns(
            int shard,
            ClientId clientId,
            Map<Long, String> campaignIdToBrandSurveyIdMap
    ) {
        if (campaignIdToBrandSurveyIdMap.isEmpty()) {
            return emptyMap();
        }

        var campaignIds = new ArrayList<>(campaignIdToBrandSurveyIdMap.keySet());
        var brandSurveyStatusForCampaigns =
                campaignBrandSurveyYtRepository.getBrandSurveyStatusForCampaigns(campaignIdToBrandSurveyIdMap);
        var campaignBudgetReachDaily = campaignBudgetReachDailyRepository.getCampaignBudgetReachForCampaigns(shard,
                campaignIds);
        List<Long> campaignIdsInDBQueue = getDBQueueStatusForCampaigns(shard, clientId, campaignIds);
        Map<Long, Campaign> campaigns = listToMap(campaignRepository.getCampaigns(shard, campaignIds), Campaign::getId);
        Map<Long, PricePackage> pricePackageByCampaigns = pricePackageService
                .getPricePackageByCampaigns(shard, campaigns.values());
        Map<String, Boolean> isBrandSurveyCalculatedMap =
                EntryStream.of(campaignIdToBrandSurveyIdMap)
                        .distinctValues()
                        .mapKeys(cid -> Boolean.FALSE)
                        .invert()
                        .toMap();

        Map<Long, BrandSurveyStatus> brandSurveyStatusByCampaignId = listToMap(campaignIds,
                campaignId -> campaignId,
                campaignId -> {
                    var brandSurveyStatus = nvl(
                            brandSurveyStatusForCampaigns.get(campaignId),
                            new BrandSurveyStatus()
                                    .withSurveyStatusDaily(SurveyStatus.DRAFT)
                                    .withReasonIds(emptyList())
                                    .withBrandSurveyStopReasonsDaily(EnumSet.noneOf(BrandSurveyStopReason.class))
                                    .withSumSpentByTotalPeriod(BigDecimal.ZERO)
                                    .withSumSpentByDay(BigDecimal.ZERO))
                                    .withBasicUplift(new BasicUplift());

                    var campaignBudgetReachForCampaign = campaignBudgetReachDaily.get(campaignId);
                    var budget = (campaignBudgetReachForCampaign != null) ?
                            campaignBudgetReachForCampaign.getBudgetSpent() : BigDecimal.ZERO;
                    brandSurveyStatus.setSumSpentByDay(budget);
                    var campaign = campaigns.get(campaignId);
                    Long period;
                    if (pricePackageByCampaigns.containsKey(campaignId)) {
                        period = brandSurveyConditionsService.getEstimatedCpmPricePeriod(
                                campaign.getStartTime(), campaign.getFinishTime(),
                                pricePackageByCampaigns.get(campaignId),
                                campaign.getStrategy().getStrategyData().getAutoProlongation() != 0L);
                    } else {
                        period = brandSurveyConditionsService.getEstimatedPeriod(
                                campaign.getStartTime(), campaign.getFinishTime(), campaign.getStrategy());
                    }
                    brandSurveyStatus.withSumSpentByTotalPeriod(budget.multiply(BigDecimal.valueOf(period)));
                    var brandSurveyId = campaignIdToBrandSurveyIdMap.get(campaignId);
                    if (campaignIdsInDBQueue.contains(campaignId)) {
                        brandSurveyStatus.setSurveyStatusDaily(SurveyStatus.CALCULATION);
                        isBrandSurveyCalculatedMap.put(brandSurveyId, Boolean.TRUE);
                    }
                    return brandSurveyStatus;
                });
        return EntryStream.of(brandSurveyStatusByCampaignId).mapToValue((cid, brandSurveyStatus) -> {
            var brandSurveyId = campaignIdToBrandSurveyIdMap.get(cid);
            if (isBrandSurveyCalculatedMap.get(brandSurveyId)) {
                brandSurveyStatus.setSurveyStatusDaily(SurveyStatus.CALCULATION);}
            return brandSurveyStatus;
            }).toMap();
    }

    private List<Long> getDBQueueStatusForCampaigns(int shard, ClientId clientId, ArrayList<Long> campaignIds) {

        var jobs = dbQueueRepository.getJobsByJobTypeAndClientIds(
                shard, RECALC_BRAND_LIFT_CAMPAIGNS, singletonList(clientId.asLong()), JOBS_LIMIT);

        return mapAndFilterList(jobs, job -> job.getArgs().getCampaignId(), campaignIds::contains);
    }
}
