package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Functions;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.bs.common.repository.BsCommonRepository;
import ru.yandex.direct.core.entity.campaign.StrategyChangeLogRecord;
import ru.yandex.direct.core.entity.campaign.container.CampaignNewMinusKeywords;
import ru.yandex.direct.core.entity.campaign.container.CampaignsSelectionCriteria;
import ru.yandex.direct.core.entity.campaign.container.WalletsWithCampaigns;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForBlockedMoneyCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignForLauncher;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyFinishedByDate;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyUrlMonitoring;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.campaign.model.WhenMoneyOnCampaignWas;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.repository.OptimizingCampaignRequestRepository;
import ru.yandex.direct.core.entity.campaign.repository.WhenMoneyOnCampWasRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.ArchiveCampaignValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.DeleteCampaignValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.DisableDomainValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.SuspendResumeCampaignValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.UnarchiveCampaignValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.UpdateCampaignValidationService;
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository;
import ru.yandex.direct.core.entity.client.repository.ClientManagersRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.KeywordStopwordsFixer;
import ru.yandex.direct.core.entity.keyword.repository.KeywordCacheRepository;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.usercampaignsfavorite.repository.UserCampaignsFavoriteRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.moderation.client.ModerationClient;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.creator.IgnoreDuplicatesOperationCreator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignMoreThanOnceInRequest;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNoRights;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.wrongDisabledDomainLimit;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.utils.CollectionUtils.flatToSet;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.result.PathHelper.field;

/**
 * Сервис кампаний.
 * Реализована логика обновления минус-фраз на кампанию и изменения связанных бизнес-логикой полей.
 * <p>
 * ! Обновление других полей с учётом бизнес-логики необходимо дополнительно реализовывать.
 */

@Service
@ParametersAreNonnullByDefault
public class CampaignService {
    private static final Logger logger = LoggerFactory.getLogger(CampaignService.class);
    private static final String LOGGER_NAME = "PPCLOG_CMD.log";
    private static final Logger changeLogger = LoggerFactory.getLogger(LOGGER_NAME);
    private static final Set<CampaignStatusModerate> MODERATE_STATUSES_TO_BLOCK = ImmutableSet.of(
            CampaignStatusModerate.NEW, CampaignStatusModerate.READY, CampaignStatusModerate.SENT,
            CampaignStatusModerate.NO
    );

    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;
    private final BsCommonRepository bsCommonRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final KeywordCacheRepository keywordCacheRepository;
    private final WhenMoneyOnCampWasRepository whenMoneyOnCampWasRepository;
    private final UpdateCampaignValidationService updateCampaignValidationService;
    private final DeleteCampaignValidationService deleteCampaignValidationService;
    private final SuspendResumeCampaignValidationService suspendResumeCampaignValidationService;
    private final ArchiveCampaignValidationService archiveCampaignValidationService;
    private final UnarchiveCampaignValidationService unarchiveCampaignValidationService;
    private final ClientManagersRepository clientManagersRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final KeywordStopwordsFixer keywordStopwordsFixer;
    private final KeywordNormalizer keywordNormalizer;
    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final IntApiClient intApiClient;
    private final DslContextProvider dslContextProvider;
    private final DisableDomainValidationService disableDomainValidationService;
    private final CampOperationQueueRepository campOperationQueueRepository;
    private final StrategyTranslationService strategyTranslationService;
    private final UserCampaignsFavoriteRepository userCampaignsFavoriteRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final BidRepository bidRepository;
    private final OptimizingCampaignRequestRepository optimizingCampaignRequestRepository;
    private final KeywordRepository keywordRepository;
    private final RequestBasedMetrikaClientFactory metrikaClientFactory;
    private final ClientService clientService;
    private final ModerationClient moderationClient;
    private final ModerationRepository moderationRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final BannerModerateService bannerModerateService;
    private final BannerRelationsRepository bannerRelationsRepository;

    @Autowired
    public CampaignService(CampaignRepository campaignRepository,
                           CampaignTypedRepository campaignTypedRepository,
                           CampaignModifyRepository campaignModifyRepository,
                           BsCommonRepository bsCommonRepository,
                           AdGroupRepository adGroupRepository,
                           BannerCommonRepository bannerCommonRepository,
                           KeywordCacheRepository keywordCacheRepository,
                           WhenMoneyOnCampWasRepository whenMoneyOnCampWasRepository,
                           UpdateCampaignValidationService updateCampaignValidationService,
                           DeleteCampaignValidationService deleteCampaignValidationService,
                           SuspendResumeCampaignValidationService suspendResumeCampaignValidationService,
                           ArchiveCampaignValidationService archiveCampaignValidationService,
                           UnarchiveCampaignValidationService unarchiveCampaignValidationService,
                           ClientManagersRepository clientManagersRepository,
                           MinusKeywordPreparingTool minusKeywordPreparingTool,
                           KeywordStopwordsFixer keywordStopwordsFixer,
                           KeywordNormalizer keywordNormalizer,
                           ShardHelper shardHelper,
                           RbacService rbacService,
                           IntApiClient intApiClient,
                           DslContextProvider dslContextProvider,
                           DisableDomainValidationService disableDomainValidationService,
                           CampOperationQueueRepository campOperationQueueRepository,
                           StrategyTranslationService strategyTranslationService,
                           UserCampaignsFavoriteRepository userCampaignsFavoriteRepository,
                           AggregatedStatusesRepository aggregatedStatusesRepository,
                           BidRepository bidRepository,
                           OptimizingCampaignRequestRepository optimizingCampaignRequestRepository,
                           KeywordRepository keywordRepository,
                           RequestBasedMetrikaClientFactory metrikaClientFactory,
                           ClientService clientService,
                           ModerationClient moderationClient,
                           ModerationRepository moderationRepository,
                           PpcPropertiesSupport ppcPropertiesSupport,
                           BannerModerateService bannerModerateService,
                           BannerRelationsRepository bannerRelationsRepository
                           ) {
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
        this.bsCommonRepository = bsCommonRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.keywordCacheRepository = keywordCacheRepository;
        this.whenMoneyOnCampWasRepository = whenMoneyOnCampWasRepository;
        this.updateCampaignValidationService = updateCampaignValidationService;
        this.deleteCampaignValidationService = deleteCampaignValidationService;
        this.suspendResumeCampaignValidationService = suspendResumeCampaignValidationService;
        this.archiveCampaignValidationService = archiveCampaignValidationService;
        this.unarchiveCampaignValidationService = unarchiveCampaignValidationService;
        this.clientManagersRepository = clientManagersRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.keywordStopwordsFixer = keywordStopwordsFixer;
        this.keywordNormalizer = keywordNormalizer;
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.intApiClient = intApiClient;
        this.dslContextProvider = dslContextProvider;
        this.disableDomainValidationService = disableDomainValidationService;
        this.campOperationQueueRepository = campOperationQueueRepository;
        this.strategyTranslationService = strategyTranslationService;
        this.userCampaignsFavoriteRepository = userCampaignsFavoriteRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.bidRepository = bidRepository;
        this.optimizingCampaignRequestRepository = optimizingCampaignRequestRepository;
        this.keywordRepository = keywordRepository;
        this.metrikaClientFactory = metrikaClientFactory;
        this.clientService = clientService;
        this.moderationClient = moderationClient;
        this.moderationRepository = moderationRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.bannerModerateService = bannerModerateService;
        this.bannerRelationsRepository = bannerRelationsRepository;
    }

    /**
     * Получение остатка средств на кошельках для списка кампаний.
     *
     * @return Мапа с ключом исходных cid на {@link WalletRestMoney}.
     * Если для кампании нет кошелька, значение будет не {@code null},
     * но будет содержать нулевой остаток и {@code id=null}.
     */
    public Map<Long, WalletRestMoney> getWalletsRestMoney(ClientId clientId, Collection<Campaign> campaigns) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        WalletsWithCampaigns walletsWithCampaigns = campaignRepository.getWalletAllCampaigns(shard, clientId,
                campaigns);
        return new WalletRestMoneyCalculator(walletsWithCampaigns).getWalletRestMoneyByCampaignIds(campaigns);
    }

    /**
     * Получение остатка средств на кошельках по по id кампаний-кошельков
     * Если кампании-кошелька не существует, то в результате её не будет.
     */
    public Map<Long, WalletRestMoney> getWalletsRestMoneyByWalletCampaignIds(int shard,
                                                                             Collection<Long> walletCampaignIds) {
        WalletsWithCampaigns walletsWithCampaigns =
                campaignRepository.getWalletsWithCampaignsByWalletCampaignIds(shard, walletCampaignIds, false);
        return getWalletsRestMoneyByWalletsWithCampaigns(walletsWithCampaigns);
    }

    /**
     * Получение остатка средств на кошельках по id кампаний-кошельков и кампаниям, к ним привязанным
     */
    public Map<Long, WalletRestMoney> getWalletsRestMoneyByWalletsWithCampaigns(WalletsWithCampaigns walletsWithCampaigns) {
        return new WalletRestMoneyCalculator(walletsWithCampaigns).getWalletRestMoneyByWalletCampaignIds();
    }

    /**
     * По заданному набору {@code campaignIds} возвращает набор {@link Campaign}
     * с заполненным {@link ru.yandex.direct.core.entity.campaign.model.DbStrategy}
     */
    public List<Campaign> getCampaignsWithStrategies(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaignsWithStrategy(shard, campaignIds);
    }

    /**
     * По заданному набору {@code campaignIds} возвращает набор {@link Campaign}
     */
    public List<Campaign> getCampaigns(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaigns(shard, campaignIds);
    }

    /**
     * По заданному набору {@code campaignIds} возвращает набор {@link Campaign}
     */
    public Map<Long, Campaign> getCampaignsMap(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaignsMap(shard, campaignIds);
    }

    public Map<Long, Long> getCampaignIdByPerfFilterId(ClientId clientId, List<PerformanceFilter> filters) {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, Long> adGroupIdByFilterId =
                listToMap(filters, PerformanceFilter::getId, PerformanceFilter::getPid);
        Set<Long> adGroupIds = new HashSet<>(adGroupIdByFilterId.values());
        Map<Long, Long> campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);
        return EntryStream.of(adGroupIdByFilterId)
                .mapValues(campaignIdsByAdGroupIds::get)
                .toMap();
    }

    public Map<Long, Set<Long>> getCampaignIdsByFeedId(ClientId clientId, Collection<Long> feedIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, List<Long>> adGroupIdsByFeedId = adGroupRepository.getAdGroupIdsByFeedId(shard, feedIds);
        Set<Long> adGroupIds = flatToSet(adGroupIdsByFeedId.values());
        Map<Long, Long> campaignIdByAdGroupId = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);
        return EntryStream.of(adGroupIdsByFeedId)
                .mapValues(groupIds -> listToSet(groupIds, campaignIdByAdGroupId::get))
                .toMap();
    }

    public CampaignsUpdateOperation createPartialUpdateOperation(
            List<ModelChanges<Campaign>> modelChangesList,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            long operatorUid, ClientId clientId) {
        return createUpdateOperation(modelChangesList, minusPhraseValidationMode, operatorUid, clientId,
                Applicability.PARTIAL);
    }

    public CampaignsUpdateOperation createFullUpdateOperation(
            List<ModelChanges<Campaign>> modelChangesList,
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
            long operatorUid, ClientId clientId) {
        return createUpdateOperation(modelChangesList, minusPhraseValidationMode, operatorUid, clientId,
                Applicability.FULL);
    }

    public MassResult<Long> updateCampaignsFull(List<ModelChanges<Campaign>> modelChanges,
                                                MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                                long operatorUid, ClientId clientId) {
        return updateCampaigns(modelChanges, minusPhraseValidationMode,
                operatorUid, clientId, Applicability.FULL);
    }

    public MassResult<Long> updateCampaignsPartial(List<ModelChanges<Campaign>> modelChanges,
                                                   MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                                   long operatorUid, ClientId clientId) {
        return updateCampaigns(modelChanges, minusPhraseValidationMode,
                operatorUid, clientId, Applicability.PARTIAL);
    }

    public AppendCampaignMinusKeywordsOperation createAppendMinusKeywordsOperation(
            List<CampaignNewMinusKeywords> newMinusKeywords,
            long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new AppendCampaignMinusKeywordsOperation(newMinusKeywords,
                campaignRepository, adGroupRepository, bannerCommonRepository,
                keywordCacheRepository, updateCampaignValidationService,
                minusKeywordPreparingTool, keywordStopwordsFixer, keywordNormalizer,
                operatorUid, clientId, shard);
    }

    public Set<String> getDisabledDomainsByCampaignId(int shard, Long campaignId, GroupType groupType) {
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, singletonList(campaignId));
        if (campaigns.isEmpty()) {
            return emptySet();
        }
        Campaign campaign = campaigns.get(0);
        if (groupType == GroupType.VIDEO || groupType == GroupType.VIDEO_NON_SKIPPABLE) {
            if (campaign.getDisabledVideoPlacements() == null) {
                return emptySet();
            }
            return new HashSet<>(campaign.getDisabledVideoPlacements());
        } else {
            if (campaign.getDisabledDomains() == null) {
                return emptySet();
            }
            return new HashSet<>(campaign.getDisabledDomains());
        }
    }

    public Map<Long, Set<Integer>> getGeoByCampaignIds(ClientId clientId, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getGeoByCampaignIds(shard, ids);
    }

    private CampaignsUpdateOperation createUpdateOperation(List<ModelChanges<Campaign>> modelChangesList,
                                                           MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                                           long operatorUid, ClientId clientId,
                                                           Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new CampaignsUpdateOperation(applicability, modelChangesList, campaignRepository, adGroupRepository,
                bannerCommonRepository, updateCampaignValidationService, minusKeywordPreparingTool,
                minusPhraseValidationMode, operatorUid,
                clientId, shard);
    }

    private MassResult<Long> updateCampaigns(List<ModelChanges<Campaign>> modelChanges,
                                             MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                             long operatorUid, ClientId clientId, Applicability applicability) {
        return createUpdateOperation(modelChanges, minusPhraseValidationMode, operatorUid, clientId, applicability)
                .prepareAndApply();
    }

    public MassResult<Long> deleteCampaigns(List<Long> campaignIds,
                                            long operatorUid, ClientId clientId, Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new CampaignsDeleteOperation(applicability, campaignIds, campaignRepository,
                campOperationQueueRepository, deleteCampaignValidationService, clientManagersRepository, rbacService,
                dslContextProvider, operatorUid, clientId, shard)
                .prepareAndApply();
    }

    public Map<Long, Long> getCampaignIdByOrderId(ClientId clientId, Collection<Long> orderIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<CampaignSimple> campaigns = campaignRepository.getCampaignsSimpleBySelectionCriteria(shard,
                new CampaignsSelectionCriteria()
                        .withClientId(clientId)
                        .withOrderIds(orderIds));
        return listToMap(campaigns, CampaignSimple::getOrderId, CampaignSimple::getId);
    }

    public Map<Long, Long> getOrderIdByCampaignId(Collection<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .stream()
                .mapKeyValue(bsCommonRepository::getOrderIdForCampaigns)
                .flatMapToEntry(Functions.identity())
                .toMap();
    }

    public Map<Long, Long> getMasterIdBySubCampaignId(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getMasterIdBySubCampaignId(shard, campaignIds);
    }

    public Map<Long, Long> getSubCampaignIdsWithMasterIds(Collection<Long> masterIds) {
        return shardHelper.groupByShard(masterIds, ShardKey.CID)
                .stream()
                .mapKeyValue(campaignRepository::getSubCampaignIdsWithMasterIds)
                .flatMapToEntry(Functions.identity())
                .toMap();
    }

    /**
     * Получает подлежащие кампании для всех переданных мастер-кампаний.
     *
     * @return мапу (подлежащая кампания -> мастер-кампания)
     */
    public Map<Long, Long> getSubCampaignIdsWithMasterIds(Collection<Long> masterIds, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getSubCampaignIdsWithMasterIds(shard, masterIds);
    }

    /**
     * Определяет какие на кампаниях есть либо будут деньги - заблокированные, либо нет.
     * Условие: не была в БК, не сервисируемая, не агентсткая, и (предварительно принята, либо отклонена на модерации)
     * <p>
     * Такие условия выставлены из-за того, что нет возможности получить от Биллинга тип денег на кампании
     * <p>
     * https://st.yandex-team.ru/DIRECT-22284
     * <p>
     * В перле известна как CampaignOperations::mass_check_block_money_camps
     *
     * @param campaigns                   список кампаний для проверки. Каждая кампания должна польностью реализовывать
     *                                    интерфейс так, чтобы все поля были заполнены. Валидации этого внутри метода
     *                                    не производится
     * @param noCheckCampaignsUnderWallet не проверять другие кампании под общим счетом кампании (или под самой
     *                                    кампанией, если она - общий счет). Используется при переносе для включения
     *                                    общего счета для определения промодерированности кампании общего счета.
     *                                    В перле - is_enable_wallet
     * @param statusOnly                  считать блокировку денег на кампании без учета сумм на ней, а только с учетом
     *                                    конкретных статусов.
     *                                    not delete condition for sum_to_pay (needed for moderateClientNew.pl)
     */
    public Map<Long, Boolean> moneyOnCampaignsIsBlocked(Collection<CampaignForBlockedMoneyCheck> campaigns,
                                                        boolean noCheckCampaignsUnderWallet, boolean statusOnly) {
        return moneyOnCampaignsIsBlocked(campaigns, noCheckCampaignsUnderWallet, statusOnly, false);
    }

    /**
     * Определяет какие на кампаниях есть либо будут деньги - заблокированные, либо нет.
     * Условие: не была в БК, не сервисируемая, не агентсткая, и (предварительно принята, либо отклонена на модерации)
     * <p>
     * Такие условия выставлены из-за того, что нет возможности получить от Биллинга тип денег на кампании
     * <p>
     * https://st.yandex-team.ru/DIRECT-22284
     * <p>
     * В перле известна как CampaignOperations::mass_check_block_money_camps
     *
     * @param campaigns                   список кампаний для проверки. Каждая кампания должна польностью реализовывать
     *                                    интерфейс так, чтобы все поля были заполнены. Валидации этого внутри метода
     *                                    не производится
     * @param noCheckCampaignsUnderWallet не проверять другие кампании под общим счетом кампании (или под самой
     *                                    кампанией, если она - общий счет). Используется при переносе для включения
     *                                    общего счета для определения промодерированности кампании общего счета.
     *                                    В перле - is_enable_wallet
     * @param statusOnly                  считать блокировку денег на кампании без учета сумм на ней, а только с учетом
     *                                    конкретных статусов.
     *                                    not delete condition for sum_to_pay (needed for moderateClientNew.pl)
     * @param noSubSelects                сообщить методу, что переданный список достаточен для определения
     *                                    заблокированности денег и не нужно довыбирать из базы данные о кампаниях под
     *                                    кошельками
     */
    public Map<Long, Boolean> moneyOnCampaignsIsBlocked(Collection<CampaignForBlockedMoneyCheck> campaigns,
                                                        boolean noCheckCampaignsUnderWallet, boolean statusOnly,
                                                        boolean noSubSelects) {
        // Определяем все кошельки, под которым находятся переданные кампании
        Map<Long, CampaignForBlockedMoneyCheck> idToCampaign = new HashMap<>();
        Map<Long, Long> idToWalletCampaignId = new HashMap<>();
        Set<Long> userIds = new HashSet<>();
        Set<Long> walletIds = new HashSet<>();
        for (CampaignForBlockedMoneyCheck campaign : campaigns) {
            // В перле это выглядит так: my $type = $camp->{type} || $camp->{mediaType};
            CampaignType type = checkNotNull(campaign.getType(), "Campaign type not defined for campaign with id %s",
                    campaign.getId());
            if ((isValidId(campaign.getWalletId()) || type == CampaignType.WALLET) &&
                    isValidId(campaign.getUserId()) &&
                    !noCheckCampaignsUnderWallet) {
                // есть общий счет на кампании или запросили сам общий счет, считаем с учетом всех кампаний в группе
                // с одним общим счетом
                // если вычисляем статус кампании при включении счета (noCheckCampaignsUnderWallet), то не нужно
                // смотреть на кампании под счетом
                Long walletCampaignId = type == CampaignType.WALLET ? campaign.getId() : campaign.getWalletId();
                idToWalletCampaignId.put(campaign.getId(), walletCampaignId);

                userIds.add(campaign.getUserId());
                walletIds.add(walletCampaignId);
            }

            idToCampaign.put(campaign.getId(), campaign); // если под кошельком ничего нет, смотрим на сам кошелёк
        }

        // Если мы нашли используемые кошельки, то получаем все кампании, которые находятся под этими кошельками
        Map<Long, List<CampaignForBlockedMoneyCheck>> walletCampaignIdToCampaigns;
        if (noSubSelects) {
            walletCampaignIdToCampaigns = StreamEx.of(campaigns)
                    .filter(c -> c.getType() != CampaignType.WALLET)
                    .filter(c -> isValidId(c.getWalletId()))
                    .groupingBy(CampaignForBlockedMoneyCheck::getWalletId);
        } else {
            walletCampaignIdToCampaigns =
                    getWalletIdToCampaigns(userIds, walletIds);
        }

        Map<Long, Boolean> walletCampaignIdToBlockedByStatusAndSum = new HashMap<>();
        Map<Long, Boolean> campaignIdToBlockedStatus = new HashMap<>();
        for (CampaignForBlockedMoneyCheck campaign : campaigns) {
            Long walletCampaignId = idToWalletCampaignId.get(campaign.getId());
            List<CampaignForBlockedMoneyCheck> walletCampaigns;
            boolean blockedByStatusAndSum;
            if (walletCampaignId != null && walletCampaignIdToCampaigns.containsKey(walletCampaignId)) {
                walletCampaigns = walletCampaignIdToCampaigns.get(walletCampaignId);
                blockedByStatusAndSum = walletCampaignIdToBlockedByStatusAndSum.computeIfAbsent(walletCampaignId,
                        id -> moneyIsBlockedByStatusAndSum(statusOnly, walletCampaigns));
            } else {
                walletCampaigns = singletonList(idToCampaign.get(campaign.getId()));
                blockedByStatusAndSum = moneyIsBlockedByStatusAndSum(statusOnly, walletCampaigns);
            }

            boolean blockedMoney = blockedByStatusAndSum &&
                    !isValidId(campaign.getManagerUserId()) &&
                    !isValidId(campaign.getAgencyUserId());
            campaignIdToBlockedStatus.put(campaign.getId(), blockedMoney);
        }

        return campaignIdToBlockedStatus;
    }

    private boolean moneyIsBlockedByStatusAndSum(boolean statusOnly,
                                                 List<CampaignForBlockedMoneyCheck> walletCampaigns) {
        boolean blockedMoney = true;
        BigDecimal sum = BigDecimal.ZERO;
        BigDecimal sumToPay = BigDecimal.ZERO;
        for (CampaignForBlockedMoneyCheck walletCampaign : walletCampaigns) {
            if (!blockedMoney) {
                break;
            }
            blockedMoney = walletCampaign.getStatusPostModerate() != CampaignStatusPostmoderate.ACCEPTED &&
                    (walletCampaign.getStatusPostModerate() == CampaignStatusPostmoderate.YES ||
                            MODERATE_STATUSES_TO_BLOCK.contains(walletCampaign.getStatusModerate()));

            if (!statusOnly) {
                if (walletCampaign.getSum() != null) {
                    sum = sum.add(walletCampaign.getSum());
                }
                if (walletCampaign.getSumToPay() != null) {
                    sumToPay = sumToPay.add(walletCampaign.getSumToPay());
                }
            }
        }

        if (!statusOnly) {
            // not delete condition for sum_to_pay !!!
            // needed for moderateClientNew.pl
            blockedMoney = blockedMoney &&
                    (sum.compareTo(BigDecimal.ZERO) > 0 || sumToPay.compareTo(BigDecimal.ZERO) > 0);
        }
        return blockedMoney;
    }

    /**
     * Получение кампаний-кошельков по id кампаний-кошельков
     */
    public Collection<WalletCampaign> getWalletsByWalletCampaignIds(int shard, Collection<Long> walletCampaignIds) {
        return campaignRepository.getWalletsByWalletCampaignIds(shard, walletCampaignIds);
    }

    private Map<Long, List<CampaignForBlockedMoneyCheck>> getWalletIdToCampaigns(Set<Long> userIds,
                                                                                 Set<Long> walletIds) {
        if (walletIds.isEmpty() || userIds.isEmpty()) {
            return emptyMap();
        }

        return shardHelper.groupByShard(userIds, ShardKey.UID)
                .stream()
                .mapKeyValue((shard, uid) -> campaignRepository
                        .getCampaignsUnderWalletsForBlockedMoneyCheck(shard, uid, walletIds))
                .flatMap(List::stream)
                .groupingBy(CampaignForBlockedMoneyCheck::getWalletId);
    }

    /**
     * Определяет какие на кампании есть либо будут деньги - заблокированные, либо нет.
     * Условие: не была в БК, не сервисируемая, не агентская, и (предварительно принята, либо отклонена на модерации)
     * <p>
     * Такие условия выставлены из-за того, что нет возможности получить от Биллинга тип денег на кампании
     * <p>
     * https://st.yandex-team.ru/DIRECT-22284
     *
     * @param campaign                    кампания, которую проверяем
     * @param noCheckCampaignsUnderWallet перенос для включения общего счета (используется для определения
     *                                    промодерированности кампании общего счета)
     * @param statusOnly                  not delete condition for sum_to_pay (needed for moderateClientNew.pl)
     */
    public boolean moneyOnCampaignIsBlocked(CampaignForBlockedMoneyCheck campaign, boolean noCheckCampaignsUnderWallet,
                                            boolean statusOnly) {
        return moneyOnCampaignsIsBlocked(singletonList(campaign), noCheckCampaignsUnderWallet, statusOnly)
                .get(campaign.getId());
    }

    /**
     * Метод для логирования интервалов времени когда на кампании {@param cid} были деньги.
     * Заполняет таблицу ppc.when_money_on_camp_was, где запись (cid, interval_start, interval_end) означает,
     * что на кампании cid все время между interval_start, interval_end был ненулевой остаток.
     * <p>
     * В зависимости от указанного события {@param event} открывает новый интервал или закрывает текущий:
     * - WhenMoneyOnCampWasEvents.MONEY_IN (пополнение) открываем новый интервал, где interval_end = "2038-01-19
     * 00:00:00"
     * - WhenMoneyOnCampWasEvents.MONEY_OUT (пополнение) закрываем текущий (открытый) интервал  interval_end = NOW()
     * <p>
     * Если шард для идетификатора кампании не найден, метод возвращает управление, ничего не сделав.
     *
     * @param campaignId - id кампании для которой делаем логирование
     * @param event      - событие пополнения или окончания средств на кампании.
     * @throws IllegalArgumentException если cid или event null
     */
    public void whenMoneyOnCampWas(Long campaignId, WhenMoneyOnCampWasEvents event) {
        checkNotNull(campaignId, "cid cannot be null");
        checkNotNull(event, "event cannot be null");

        int shard = shardHelper.getShardByCampaignId(campaignId);
        if (shard == NO_SHARD) {
            return;
        }

        List<Long> campaignList = new ArrayList<>();
        campaignList.add(campaignId);

        //если cid - это кампания-кошелек, то выполняем операцию для всех дочерних кампаний для данного кошелька
        //в перловом коде Notify_Order метод вызывается только для для кампаний не являющихся кошельком
        //(Campaign::when_money_on_camp_was(cid => $cid, event => 'money_in') if $x->{wallet_cid} == 0;)
        //но логику  метода решено сохранить в перловозданном виде
        List<Long> campaignUnderWalletList = campaignRepository.getCampaignIdsUnderWallet(shard, campaignId);
        campaignList.addAll(campaignUnderWalletList);

        switch (event) {
            case MONEY_IN:
                Set<Long> campsWithOpenInetrval =
                        whenMoneyOnCampWasRepository.getCidsWithOpenedInterval(shard, campaignList);
                List<Long> campsWithClosedInterval = campaignList.stream().filter(
                        r -> !campsWithOpenInetrval.contains(r)).collect(toList());
                if (!campsWithClosedInterval.isEmpty()) {
                    whenMoneyOnCampWasRepository.openInterval(shard, campsWithClosedInterval);
                }
                break;
            case MONEY_OUT:
                whenMoneyOnCampWasRepository.closeInterval(shard, campaignList);
                break;
            default:
                throw new IllegalStateException("Unsupported event: " + event);
        }
    }

    /**
     * Разархивация кампании
     */
    public void unarchiveCampaign(Long uid, Long cid) {
        intApiClient.unarcCampaign(uid, cid, false);
    }

    /**
     * Получить когда на кампании были деньги для указанного списка кампаний.
     *
     * @return список с данными когда на кампании были деньги
     */
    public List<WhenMoneyOnCampaignWas> getWhenMoneyOnCampaignsWas(Collection<Long> campaignIds) {
        List<WhenMoneyOnCampaignWas> result = new ArrayList<>();

        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .forEach((shard, campaignIdsGroupByShard) ->
                        result.addAll(
                                whenMoneyOnCampWasRepository.getWhenMoneyOnCampaignsWas(shard, campaignIdsGroupByShard)
                        )
                );

        return result;
    }

    public List<CampaignForNotifyFinishedByDate> getCampaignsForNotifyFinishedByDate(int shard,
                                                                                     List<Integer> finishedDaysAgo) {
        return campaignRepository.getActiveCampaignsFinishedByDate(shard, finishedDaysAgo);
    }

    /**
     * Кампании для нотификаций URL мониторинга доменов
     *
     * @param domains URL-ы доменов, по которым сработал мониторинг
     * @return отображение ID пользователей на отображение URL-ов доменов на список кампаний для нотификации
     */
    public Map<Long, Map<String, List<CampaignForNotifyUrlMonitoring>>> getCampaignsForNotifyUrlMonitoring(
            List<Pair<String, String>> domains) {
        return shardHelper.dbShards().stream()
                .map(shard -> campaignRepository.getCampaignsForNotifyUrlMonitoring(shard, domains))
                .map(Map::entrySet)
                .flatMap(Collection::stream)
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    /**
     * Поиск кампаний заданных типов для одного ClientID
     *
     * @param clientId      ClientID, для которого ищем кампании
     * @param campaignTypes типы кампаний
     */
    public List<CampaignSimple> searchCampaignsByClientIdAndTypes(ClientId clientId,
                                                                  Collection<CampaignType> campaignTypes) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.searchNotEmptyNotArchivedCampaigns(shard, clientId, campaignTypes);
    }

    /**
     * Поиск кампаний заданных типов для одного ClientID
     *
     * @param clientId      ClientID, для которого ищем кампании
     * @param campaignTypes типы кампаний
     */
    public List<Campaign> searchNotEmptyCampaignsByClientIdAndTypes(ClientId clientId,
                                                                    Collection<CampaignType> campaignTypes) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.searchNotEmptyCampaigns(shard, clientId, campaignTypes);
    }

    /**
     * Получить идентификаторы мобильных приложений, привязанных к РМП кампании
     *
     * @return отображение campaignId -> mobileAppId
     */
    public Map<Long, Long> getCampaignMobileAppId(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaignMobileAppIds(shard, campaignIds);
    }

    /**
     * Получить идентификаторы мобильных приложений клиента, привязанных к РМП кампании, но не UAC
     */
    public List<Long> getNonArchivedMobileAppCampaignIds(ClientId clientId, Long mobileAppId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getNonArchivedMobileAppCampaignIds(shard, clientId, mobileAppId);
    }

    public Boolean getStatusShowByCampaignId(int shard, Long campaignId) {
        return campaignRepository.getStatusShowByCampaignId(shard, campaignId);
    }

    /**
     * Поиск кампаний заданных типов для списка ClientID
     *
     * @param clientIds     список ClientID, для которых ищем кампании
     * @param campaignTypes типы кампаний
     */
    public Map<Long, List<CampaignSimple>> searchCampaignsByClientIdsAndTypes(Collection<Long> clientIds,
                                                                              Collection<CampaignType> campaignTypes) {
        Map<Long, List<CampaignSimple>> result = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, clientIdsGroupByShard) ->
                        result.putAll(campaignRepository
                                .searchNotEmptyNotArchivedCampaigns(shard,
                                        mapList(clientIdsGroupByShard, ClientId::fromLong), campaignTypes))
                );
        return result;
    }


    /**
     * Поиск всех кампаний (включая пустые и архивные)
     * заданных типов для списка ClientID
     *
     * @param clientIds     список ClientID, для которых ищем кампании
     * @param campaignTypes типы кампаний
     * @return список кампаний по ClientID
     */
    public Map<Long, List<CampaignSimple>> searchAllCampaignsByClientIdsAndTypes(
            Collection<Long> clientIds, Collection<CampaignType> campaignTypes) {
        Map<Long, List<CampaignSimple>> result = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, clientIdsGroupByShard) ->
                        result.putAll(campaignRepository
                                .searchAllCampaigns(shard,
                                        mapList(clientIdsGroupByShard, ClientId::fromLong), campaignTypes))
                );
        return result;
    }

    public List<CampaignForLauncher> getCampaignsForLauncher() {
        return StreamEx.of(shardHelper.dbShards())
                .map(campaignRepository::getCampaignsForLauncher)
                .flatMap(Collection::stream)
                .collect(toList());
    }

    public List<Campaign> searchCampaignsByCriteria(ClientId clientId, CampaignsSelectionCriteria criteria) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaigns(dslContextProvider.ppc(shard), criteria, false);
    }

    public Set<Long> searchFavoriteCampaignIds(long uid) {
        int shard = shardHelper.getShardByClientUid(uid);
        return userCampaignsFavoriteRepository.getCampaignIdsByUid(dslContextProvider.ppc(shard), uid);
    }

    public Map<Long, List<MeaningfulGoal>> getMeaningfulGoalsByCampaignId(
            ClientId clientId,
            Collection<Long> campaignIds
    ) {
        int shard = shardHelper.getShardByClientId(clientId);
        return campaignRepository.getMeaningfulGoalsByCampaignId(shard, campaignIds);
    }

    /**
     * Удаление домена из черного списка по списку кампаний
     *
     * @param shard       шард
     * @param campaignIds список кампаний
     * @param domain      удаляемый домен
     */
    public void removeDomainFromDisabled(int shard, List<Long> campaignIds, String domain) {
        for (Long campaignId : campaignIds) {
            dslContextProvider.ppc(shard).transaction(configuration -> {
                DSLContext txContext = configuration.dsl();

                Set<String> domains = campaignRepository
                        .getDisabledDomainsByCampaignIds(txContext, singletonList(campaignId), true).get(campaignId);

                if (domains != null && domains.remove(domain)) {
                    ModelChanges<Campaign> changes = new ModelChanges<>(campaignId, Campaign.class);
                    changes.process(domains, Campaign.DISABLED_DOMAINS);
                    changes.process(StatusBsSynced.NO, Campaign.STATUS_BS_SYNCED);
                    changes.process(LocalDateTime.now(), Campaign.LAST_CHANGE);
                    AppliedChanges<Campaign> appliedChanges = changes.applyTo(new Campaign().withId(campaignId));
                    campaignRepository.updateCampaigns(txContext, singleton(appliedChanges));

                    logger.info("Domain {} deleted from DontShow for cid {}", domain, campaignId);
                }
            });
        }
    }

    /**
     * Проверить что кампания существует и не помечена удаленной (statusEmpty=Yes)
     *
     * @param campaignId номер кампании
     * @return {@code true} при существовании кампании со statusEmpty=No в шарде, хранящемся в метабазе
     */
    public boolean campaignExists(long campaignId) {
        int shard = shardHelper.getShardByCampaignId(campaignId);
        return shard != NO_SHARD && campaignRepository.campaignExists(shard, campaignId);
    }

    /**
     * Получить словарь соответствий "номер кампании" - тип/архивность
     *
     * @param campaignIds идентификаторы кампаний
     * @return словарь, ключами в которого являются идентификаторы кампаний, а значениями - их тип
     */
    public Map<Long, CampaignType> getCampaignsTypes(Collection<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedByDefault()
                .stream()
                .mapKeyValue(campaignRepository::getCampaignsTypeMap)
                .flatMap(EntryStream::of)
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    /**
     * Получить словарь соответствий "номер кампании" - тип/архивность
     *
     * @param campaignIds       идентификаторы кампаний
     * @param filterStatusEmpty оставить только кампании с statusEmpty=No
     * @return словарь, ключами в которого являются идентификаторы кампаний, а значениями - их тип
     */
    public Map<Long, CampaignTypeSource> getCampaignsTypeSourceMap(Collection<Long> campaignIds,
                                                                   boolean filterStatusEmpty) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedByDefault()
                .stream()
                .mapKeyValue((shard, ids) -> campaignRepository
                        .getCampaignsTypeSourceMap(shard, ids, filterStatusEmpty))
                .flatMap(EntryStream::of)
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    /**
     * Поиск существующих кампаний для одного ClientID
     *
     * @param clientId    ClientID, для которого ищем кампании
     * @param campaignIds идентификаторы кампаний
     */
    public Set<Long> getExistingCampaignIds(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getExistingCampaignIds(shard, clientId, campaignIds);
    }

    /**
     * Поиск существующих не-подлежащих кампаний для одного ClientID
     *
     * @param clientId    ClientID, для которого ищем кампании
     * @param campaignIds идентификаторы кампаний
     */
    public Set<Long> getExistingNonSubCampaignIds(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getExistingNonSubCampaignIds(shard, clientId, campaignIds);
    }

    /**
     * Получить плейсы внутренней рекламы по идентификаторам кампаний.
     * <p>
     * Метод не проверяет видимость кампаний клиентом. Об этом должен позаботится вызывающий код.
     *
     * @param campaignIds Идентификаторы кампаний
     */
    public Map<Long, Long> getCampaignInternalPlaces(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaignInternalPlaces(shard, campaignIds);
    }


    private ValidationResult<Campaign, Defect> updateCampaign(int shard, long operatorUid, ClientId clientId,
                                                              ModelChanges<Campaign> changes,
                                                              Campaign campaign) {
        if (!updateCampaignValidationService.isCampaignWritable(operatorUid, campaign.getId())) {
            return ValidationResult.failed(campaign, campaignNoRights());
        }
        ModelItemValidationBuilder<Campaign> vb = ModelItemValidationBuilder.of(changes.toModel());
        if (changes.isPropChanged(Campaign.DISABLED_VIDEO_PLACEMENTS)) {

            List<String> domains = changes.getChangedProp(Campaign.DISABLED_VIDEO_PLACEMENTS);
            vb.item(Campaign.DISABLED_VIDEO_PLACEMENTS).check(
                            Constraint.fromPredicate(d -> disableDomainValidationService.isValidSize(clientId,
                                            domains.size()),
                                    wrongDisabledDomainLimit()))
                    .check(Constraint.fromPredicate(d -> disableDomainValidationService.isValidDomains(domains,
                                    clientId),
                            invalidValue()), When.isValid());

            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }
            changes.process(StatusBsSynced.NO, Campaign.STATUS_BS_SYNCED);
            changes.process(LocalDateTime.now(), Campaign.LAST_CHANGE);
        }
        AppliedChanges<Campaign> appliedChanges = changes.applyTo(campaign);
        campaignRepository.updateCampaigns(dslContextProvider.ppc(shard), singleton(appliedChanges));
        return vb.getResult();
    }

    public ValidationResult<Campaign, Defect> addOrRemoveDisabledVideoDomains(long operatorUid, ClientId clientId,
                                                                              long campaignId,
                                                                              List<String> disabledVideoDomains,
                                                                              boolean disable) {
        if (isEmpty(disabledVideoDomains)) {
            return ValidationResult.failed(null, wrongDisabledDomainLimit());
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, singletonList(campaignId));
        if (campaigns.isEmpty()) {
            ValidationResult<Campaign, Defect> result = new ValidationResult<>((Campaign) null);
            ValidationResult<Long, Defect> campaignIdResult =
                    result.getOrCreateSubValidationResult(field(Campaign.ID.name()), campaignId);
            campaignIdResult.addError(campaignNotFound());
            return result;
        }
        Campaign campaign = campaigns.get(0);

        Set<String> disabledVideoDomainsNew =
                new LinkedHashSet<>(Optional.ofNullable(campaign.getDisabledVideoPlacements())
                        .orElse(Collections.emptyList()));
        final boolean isChanged;
        if (disable) {
            isChanged = disabledVideoDomainsNew.addAll(disabledVideoDomains);
        } else {
            isChanged = disabledVideoDomainsNew.removeAll(disabledVideoDomains);
        }
        if (!isChanged) {
            return ValidationResult.success(campaign);
        }
        ModelChanges<Campaign> changes = new ModelChanges<>(campaignId, Campaign.class);
        changes.process(new ArrayList<>(disabledVideoDomainsNew), Campaign.DISABLED_VIDEO_PLACEMENTS);
        return updateCampaign(shard, operatorUid, clientId, changes, campaign);
    }

    /**
     * @param shard      номер шарда
     * @param campaignId идентификатор кампании
     * @return минус-слова кампании или {@link Collections#emptyList()}, если минус-слов нет
     */
    public List<String> getMinusKeywordsByCampaignId(int shard, Long campaignId) {
        Map<Long, List<String>> minusKeywordsByCampaignIds =
                campaignRepository.getMinusKeywordsByCampaignIds(shard, Collections.singletonList(campaignId));
        return minusKeywordsByCampaignIds.getOrDefault(campaignId, emptyList());
    }

    public <C extends CampaignWithStrategy> void logStrategyChange(
            List<AppliedChanges<C>> changes,
            Long operatorUid) {
        for (var campaignAppliedChanges : changes) {
            StrategyChangeLogRecord strategyChangeLogRecord = getStrategyLogRecord(campaignAppliedChanges, operatorUid);
            changeLogger.info("{} {}", strategyChangeLogRecord.getPrefixLogTime(),
                    JsonUtils.toJson(strategyChangeLogRecord));
        }
    }

    <C extends CampaignWithStrategy> StrategyChangeLogRecord getStrategyLogRecord(
            AppliedChanges<C> changes,
            Long operatorUid) {
        Long campaignId = changes.getModel().getId();
        StrategyChangeLogRecord strategyChangeLogRecord = new StrategyChangeLogRecord(Trace.current().getTraceId());
        strategyChangeLogRecord.setCids(List.of(campaignId));
        strategyChangeLogRecord.setPath(StrategyChangeLogRecord.LOG_STRATEGY_CHANGE_CMD_NAME);
        strategyChangeLogRecord.setOperatorId(operatorUid);
        Map<String, Object> params = strategyChangeLogRecord.getParam();
        DbStrategy oldStrategy = changes.getOldValue(CampaignWithStrategy.STRATEGY);

        params.put("cid", campaignId);
        params.put("old", JsonUtils.toJson(new DbStrategyHuman(oldStrategy,
                strategyTranslationService.getRussianTranslation(oldStrategy.getStrategyName()))));
        DbStrategy newStrategy = changes.getNewValue(CampaignWithStrategy.STRATEGY);
        params.put("new", JsonUtils.toJson(new DbStrategyHuman(newStrategy,
                strategyTranslationService.getRussianTranslation(newStrategy.getStrategyName()))));
        return strategyChangeLogRecord;
    }

    public Set<Long> getClientCampaignIds(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return campaignRepository.getCampaignIdsByClientIds(shard, Set.of(clientId));
    }

    public Set<Long> getClientNonSubCampaignIds(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return campaignRepository.getNonSubCampaignIdsByClientIds(shard, Set.of(clientId));
    }

    // вернуть cid последней неархивной кампании клиента (с агентством при наличии) с переданным CampaignSource
    public Optional<Long> getLastNotArchivedCampaignOfSource(ClientId clientId,
                                                             @Nullable ClientId agencyClientId,
                                                             CampaignSource source) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        var dbSource = CampaignSource.toSource(source);
        if (dbSource != null) { // иначе не может быть, раз source не null, но инспекцию триггерит
            return campaignRepository.getLastNotArchivedCampaignOfSource(shard, clientId, agencyClientId, dbSource);
        }
        return Optional.empty();
    }

    public MassResult<Long> suspendCampaigns(List<Long> campaignIds, long operatorUid, ClientId clientId) {
        return suspendResumeCampaigns(campaignIds, false, operatorUid, clientId);
    }

    public MassResult<Long> resumeCampaigns(List<Long> campaignIds, long operatorUid, ClientId clientId) {
        return suspendResumeCampaigns(campaignIds, true, operatorUid, clientId);
    }

    protected MassResult<Long> suspendResumeCampaigns(
            List<Long> campaignIds, boolean resume,
            long operatorUid, ClientId clientId
    ) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        OperationCreator<Long, Operation<Long>> operationCreator =
                cids -> new CampaignsSuspendResumeOperation(
                        cids, resume, suspendResumeCampaignValidationService,
                        campaignRepository, operatorUid, clientId, shard
                );
        Comparator<Long> comparator = Long::compare;
        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(campaignMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, comparator, resultVisitor)
                .create(campaignIds)
                .prepareAndApply();
    }

    public MassResult<Long> archiveCampaigns(
            List<Long> campaignIds,
            long operatorUid, ClientId clientId
    ) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        OperationCreator<Long, Operation<Long>> operationCreator =
                cids -> new CampaignsArchiveOperation(
                        cids, archiveCampaignValidationService, ppcPropertiesSupport,
                        campaignRepository, bannerCommonRepository, aggregatedStatusesRepository,
                        bidRepository, optimizingCampaignRequestRepository, campOperationQueueRepository,
                        keywordRepository, dslContextProvider, operatorUid, clientId, shard
                );
        Comparator<Long> comparator = Long::compare;
        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(campaignMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, comparator, resultVisitor)
                .create(campaignIds)
                .prepareAndApply();
    }

    public MassResult<Long> unarchiveCampaigns(List<Long> campaignIds, long operatorUid,
                                               UidAndClientId uidAndclientId) {
        int shard = shardHelper.getShardByClientIdStrictly(uidAndclientId.getClientId());
        OperationCreator<Long, Operation<Long>> operationCreator =
                cids -> new CampaignsUnarchiveOperation(
                        cids, unarchiveCampaignValidationService, campaignRepository,
                        campaignTypedRepository, campaignModifyRepository, bidRepository, keywordRepository,
                        bannerCommonRepository, aggregatedStatusesRepository, campOperationQueueRepository,
                        rbacService, metrikaClientFactory, clientService, moderationClient, moderationRepository,
                        bannerModerateService, bannerRelationsRepository, dslContextProvider, operatorUid,
                        uidAndclientId, shard
                );
        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(campaignMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, Long::compare, resultVisitor)
                .create(campaignIds)
                .prepareAndApply();
    }

    public boolean needChangeStatusModeratePayCondition(ClientId clientId, Long cid) {
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.needChangeStatusModeratePayCondition(shard, cid);
    }
}
