package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.campaign.container.CampaignStrategyChangingSettings;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithRestartOfConversionStrategyTimeSaving;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.model.TextCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.pricerecalculation.CommonCampaignPriceRecalculationService;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainerImpl;
import ru.yandex.direct.core.entity.campaign.service.validation.type.disabled.DisabledFieldsDataContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.type.update.CampaignWithCustomStrategyUpdateValidationTypeSupport;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.core.entity.strategy.type.withcustomperiodbudgetandcustombid.CpmCampaignAndCpmNotRestartingStrategyWithCustomPeriodInfo;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static java.time.LocalDateTime.now;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.getTextCampaignStrategyChangingSettings;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isStrategyAvailableForSimpleView;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isStrategyGoalIdChanged;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.modifyLastBidderRestartTime;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.shouldFetchUnavailableGoals;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isStrategyModelChanged;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.isStrategyWithSupportOfLearningStatus;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для работы со стратегиями кампании.
 */
@Service
public class CampaignStrategyService {
    private final CampaignModifyRepository campaignModifyRepository;
    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignWithCustomStrategyUpdateValidationTypeSupport campaignWithStrategyUpdateValidationTypeSupport;
    private final OrderStatService orderStatService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService;
    private final FeatureService featureService;
    private final MetrikaClient metrikaClient;

    public CampaignStrategyService(CampaignModifyRepository campaignModifyRepository, ShardHelper shardHelper,
                                   RbacService rbacService, CampaignRepository campaignRepository,
                                   CampaignTypedRepository campaignTypedRepository,
                                   CampaignWithCustomStrategyUpdateValidationTypeSupport campaignWithStrategyUpdateValidationTypeSupport,
                                   OrderStatService orderStatService,
                                   AutobudgetAlertService autobudgetAlertService,
                                   CommonCampaignPriceRecalculationService commonCampaignPriceRecalculationService,
                                   FeatureService featureService,
                                   MetrikaClient metrikaClient) {

        this.campaignModifyRepository = campaignModifyRepository;
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignWithStrategyUpdateValidationTypeSupport = campaignWithStrategyUpdateValidationTypeSupport;
        this.orderStatService = orderStatService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.commonCampaignPriceRecalculationService = commonCampaignPriceRecalculationService;
        this.featureService = featureService;
        this.metrikaClient = metrikaClient;
    }

    /**
     * Установка стратегии на кампанию.
     * Аналог функции camp_set_strategy в старом директе.
     * Используется для обновления только стратегии через интапи в старом интерфейсе.
     * Отличается от {@link RestrictedCampaignsUpdateOperation}, что при обновлении стратегии валидируем только
     * стратегию.
     * И сохраняем, даже если остальная кампания невалидна.
     * Потеряет актуальность после перехода со старого интерфейса в DNA.
     */
    public MassResult<Long> updateTextCampaignStrategy(Long cid, DbStrategy strategy,
                                                       Long operatorUid,
                                                       UidAndClientId uidAndClientId,
                                                       boolean isAttributionModelChanged) {
        int shard = shardHelper.getShardByCampaignId(cid);
        TextCampaign typedCampaign = getCampaignWithStrategy(shard, cid);

        ModelChanges<TextCampaignWithCustomStrategy> campModelChanges = ModelChanges.build(typedCampaign,
                TextCampaignWithCustomStrategy.STRATEGY, strategy);

        //валидация. Валидировать нужно кампанию с новой стратегией
        var campaign = getCampaignWithStrategy(shard, cid).withStrategy(strategy);
        var representativeUids = rbacService.getClientRepresentativesUids(uidAndClientId.getClientId());
        var enabledFeatures = featureService.getEnabledForClientId(uidAndClientId.getClientId());
        List<Long> cidWithAutoRecs = enabledFeatures
                .contains(FeatureName.ENABLE_AUTO_APPLY_RECOMMENDATION.getName()) &&
                campaign.getIsRecommendationsManagementEnabled() ?
                List.of(cid) : emptyList();
        var container = new CampaignValidationContainerImpl(
                shard, operatorUid, uidAndClientId.getClientId(), null, new CampaignOptions(),
                new RequestBasedMetrikaClientAdapter(
                        metrikaClient,
                        representativeUids,
                        enabledFeatures,
                        Collections.singletonList(campaign),
                        shouldFetchUnavailableGoals(campaign, enabledFeatures)
                ), emptyMap(), new DisabledFieldsDataContainer(cidWithAutoRecs)
        );

        if (!isStrategyAvailableForSimpleView(strategy)) {
            campModelChanges.process(false, TextCampaignWithCustomStrategy.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED);
        }

        boolean strategyHasChangesFromClient = !strategy.equals(typedCampaign.getStrategy());

        var appliedChanges =
                campModelChanges.applyTo((TextCampaignWithCustomStrategy) typedCampaign);
        var appliedChangesList = List.of(appliedChanges);

        ValidationResult<List<CampaignWithCustomStrategy>, Defect> vr =
                // это нужно переделать на валидатор и не вызывать саппорт напрямую
                campaignWithStrategyUpdateValidationTypeSupport
                        .validate(container,
                                ListValidationBuilder.<CampaignWithCustomStrategy, Defect>of(List.of(campaign))
                                        .getResult(),
                                Map.of(0, appliedChanges.castModelUp(CampaignWithCustomStrategy.class)));
        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(List.of(cid), vr);
        }

        if (!strategyHasChangesFromClient && !isAttributionModelChanged) {
            return MassResult.successfulMassAction(List.of(cid), vr);
        }

        boolean strategyLearningStatusEnabled = featureService.isEnabledForClientId(uidAndClientId.getClientId(),
                FeatureName.CONVERSION_STRATEGY_LEARNING_STATUS_ENABLED);

        if (strategyLearningStatusEnabled) {
            boolean strategySupportsLearningStatus = isStrategyWithSupportOfLearningStatus(
                    appliedChanges.getModel().getStrategy().getStrategyData());
            modifyLastBidderRestartTimeIfNeed(isAttributionModelChanged, strategySupportsLearningStatus,
                    appliedChanges);
        }

        RestrictedCampaignsUpdateOperationContainer updateParameters =
                RestrictedCampaignsUpdateOperationContainer.create(
                        shard,
                        operatorUid,
                        uidAndClientId.getClientId(),
                        uidAndClientId.getUid(),
                        rbacService.getChiefByClientId(uidAndClientId.getClientId()),
                        new DisabledFieldsDataContainer(cidWithAutoRecs));
        campaignModifyRepository.updateCampaigns(updateParameters, appliedChangesList);

        //для "сервисных изменений"(статус обучения) не хотим выполнять дополнительных действий
        if (!strategyHasChangesFromClient) {
            return MassResult.successfulMassAction(List.of(cid), vr);
        }

        CampaignStrategyChangingSettings textCampaignStrategyChangingSettings =
                getTextCampaignStrategyChangingSettings(appliedChangesList);
        commonCampaignPriceRecalculationService
                .afterTextCampaignsStrategyChanged(appliedChangesList, textCampaignStrategyChangingSettings,
                        operatorUid, uidAndClientId);

        autobudgetAlertService.freezeAlertsOnStrategyChange(uidAndClientId.getClientId(),
                filterList(appliedChangesList, CampaignStrategyUtils::filterAutoBudgetOldOrNew));
        checkBroadMatchFlag(shard, typedCampaign, strategy);
        return MassResult.successfulMassAction(List.of(cid), vr);
    }

    private void modifyLastBidderRestartTimeIfNeed(boolean isAttributionModelChanged,
                                                   boolean isConversionStrategy,
                                                   AppliedChanges<TextCampaignWithCustomStrategy> appliedChanges) {
        if (!isConversionStrategy) {
            return;
        }
        LocalDateTime now = now();
        boolean isStrategyGoalIdChanged =
                appliedChanges.changed(CampaignWithRestartOfConversionStrategyTimeSaving.STRATEGY) &&
                        isStrategyGoalIdChanged(appliedChanges);

        boolean isChangedStrategyModel =
                isStrategyModelChanged(appliedChanges.getOldValue(CampaignWithRestartOfConversionStrategyTimeSaving.STRATEGY).getStrategyData(),
                        appliedChanges.getModel().getStrategy().getStrategyData());

        boolean isRestart = isAttributionModelChanged || isStrategyGoalIdChanged || isChangedStrategyModel;

        modifyLastBidderRestartTime(isRestart, now, appliedChanges);
    }

    /**
     * При смене статегии на стратегию в сетях при флаге ДРФ мы выключаем флаг
     */
    private void checkBroadMatchFlag(int shard, TextCampaign campaign, DbStrategy strategy) {
        if (strategy.isSearchStop() && campaign.getBroadMatch().getBroadMatchFlag()) {
            Campaign camp = campaignRepository.getCampaigns(shard, List.of(campaign.getId())).get(0);
            var modelChanges = new ModelChanges<>(campaign.getId(), Campaign.class)
                    .process(false, Campaign.BROAD_MATCH_FLAG);
            campaignRepository.updateCampaigns(shard, List.of(modelChanges.applyTo(camp)));
        }
    }

    private TextCampaign getCampaignWithStrategy(int shard, Long cid) {
        return (TextCampaign) campaignTypedRepository
                .getTypedCampaigns(shard, List.of(cid))
                .get(0);
    }

    /**
     * Корректируем флаг "оплата за показы" в стратегии кампании при необходимости
     */
    public static DbStrategy correctCampaignStrategyUsePayForConversionValue(DbStrategy strategy) {
        if (strategy != null) {
            //В DIRECT-152359 делаем проставление true для pay_for_conversion для CPV стратегий
            var strategyName = strategy.getStrategyName();
            if (strategyName == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD ||
                    strategyName == StrategyName.AUTOBUDGET_AVG_CPV) {
                DbStrategy copyStrategy = strategy.copy();
                StrategyData copyStrategyData = copyStrategy.getStrategyData().copy();
                copyStrategy.setStrategyData(copyStrategyData.withPayForConversion(true));
                return copyStrategy;
            }
        }
        return strategy;
    }

    /**
     * Вычисляем минимальный бюджет для cpm кампаниий
     */
    public Map<Long, BigDecimal> calculateMinimalAvailableBudgetForCpmNotRestartingStrategyWithCustomPeriod(
            List<CpmCampaignWithCustomStrategy> currentCampaigns,
            List<LocalDate> newFinishDateList,
            CurrencyCode currencyCode) {
        if (currentCampaigns.isEmpty()) {
            return emptyMap();
        }
        checkArgument(currentCampaigns.size() == newFinishDateList.size(),
                "Campaigns count should have same size with new strategies");
        LocalDate now = LocalDate.now();

        List<Long> orderIds = mapList(currentCampaigns, CommonCampaign::getOrderId);
        List<LocalDate> oldDatesOfStart = StreamEx.of(currentCampaigns)
                .map(CpmCampaignWithCustomStrategy::getStrategy)
                .map(DbStrategyBase::getStrategyData)
                .map(StrategyData::getStart)
                .toList();
        List<LocalDate> currentDates = StreamEx.generate(() -> now)
                .limit(orderIds.size())
                .toList();

        Map<Long, Money> ordersSumSpent = orderStatService.getSpentSumForOrderDuringPeriod(orderIds, oldDatesOfStart,
                currentDates,
                currencyCode, true);

        return EntryStream.zip(currentCampaigns, newFinishDateList)
                .mapToValue((currentCampaign, newFinishDate) ->
                        CampaignStrategyUtils.calculateMinimumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(currentCampaign,
                                newFinishDate,
                                ordersSumSpent.get(currentCampaign.getOrderId()).bigDecimalValue(), now, currencyCode))
                .mapKeys(CpmCampaignWithCustomStrategy::getId)
                .toMap();
    }

    /**
     * Вычисляем минимальный бюджет для cpm кампаний пакетных стратегий
     */
    public Map<Long, BigDecimal> calculateMinimalAvailableBudgetForCpmNotRestartingStrategyWithCustomPeriod(
            List<CpmCampaignAndCpmNotRestartingStrategyWithCustomPeriodInfo> campaignsAndStrategiesInfo,
            CurrencyCode currencyCode) {
        if (campaignsAndStrategiesInfo.isEmpty()) {
            return emptyMap();
        }
        LocalDate now = LocalDate.now();

        List<Long> orderIds = StreamEx.of(campaignsAndStrategiesInfo)
                .map(CpmCampaignAndCpmNotRestartingStrategyWithCustomPeriodInfo::getCampaign)
                .map(CpmCampaignWithCustomStrategy::getOrderId)
                .toList();
        List<LocalDate> currentDates = StreamEx.generate(() -> now)
                .limit(orderIds.size())
                .toList();

        Map<Long, Money> ordersSumSpent = orderStatService.getSpentSumForOrderDuringPeriod(orderIds,
                mapList(campaignsAndStrategiesInfo,
                        CpmCampaignAndCpmNotRestartingStrategyWithCustomPeriodInfo::getOldStrategyStart),
                currentDates,
                currencyCode, true);

        return StreamEx.of(campaignsAndStrategiesInfo)
                .mapToEntry(
                        campaignAndStrategyInfo -> campaignAndStrategyInfo.getCampaign().getId(),
                        campaignAndStrategyInfo -> CampaignStrategyUtils.calculateMinimumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                                campaignAndStrategyInfo.getOldStrategyStart(),
                                campaignAndStrategyInfo.getNewStrategyFinish(),
                                campaignAndStrategyInfo.getOldStrategyBudget(),
                                ordersSumSpent.get(campaignAndStrategyInfo.getCampaign().getOrderId()).bigDecimalValue(),
                                now,
                                currencyCode))
                .toMap();
    }
}
