package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.CampaignUtils;
import ru.yandex.direct.core.entity.campaign.container.CampaignStrategyChangingSettings;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDefaultPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithRestartOfConversionStrategyTimeSaving;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CpmCampaignWithPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaignWithPriceRecalculation;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.WithStrategy;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static java.time.temporal.ChronoUnit.DAYS;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.extractCounterIds;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.AVAILABLE_PROPERTIES_FOR_SIMPLE_STRATEGY;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.AVAILABLE_STRATEGY_NAMES_FOR_SIMPLE_STRATEGY;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MANUAL_PRICE_STRATEGIES;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.utils.CommonUtils.max;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class CampaignStrategyUtils {

    private CampaignStrategyUtils() {
    }

    public static <C extends CampaignWithDefaultPriceRecalculation> CampaignStrategyChangingSettings getTextCampaignStrategyChangingSettings(
            List<AppliedChanges<C>> appliedChanges) {
        checkArgument(appliedChanges.get(0).getModel().getCurrency() != null,
                "Model should have currency");
        CurrencyCode commonCurrency = appliedChanges.get(0).getModel().getCurrency();

        checkCampaignsHaveSameCurrency(appliedChanges, commonCurrency);
        return CampaignStrategyChangingSettings.create(
                commonCurrency.getCurrency().getMinPrice(),
                commonCurrency.getCurrency().getMaxPrice(),
                true);
    }

    public static <C extends CampaignWithDefaultPriceRecalculation> CampaignStrategyChangingSettings getDynamicCampaignStrategyChangingSettings(
            List<AppliedChanges<DynamicCampaignWithPriceRecalculation>> appliedChanges) {
        checkArgument(appliedChanges.get(0).getModel().getCurrency() != null,
                "Model should have currency");
        CurrencyCode commonCurrency = appliedChanges.get(0).getModel().getCurrency();

        checkCampaignsHaveSameCurrency(appliedChanges, commonCurrency);
        return CampaignStrategyChangingSettings.create(
                commonCurrency.getCurrency().getMinPrice(),
                commonCurrency.getCurrency().getMaxPrice(),
                true);
    }

    public static CampaignStrategyChangingSettings getCpmCampaignStrategyChangingSettings(
            List<AppliedChanges<CpmCampaignWithPriceRecalculation>> appliedChanges) {
        CurrencyCode commonCurrency = appliedChanges.get(0).getModel().getCurrency();

        checkCampaignsHaveSameCurrency(appliedChanges, commonCurrency);
        return CampaignStrategyChangingSettings.create(
                commonCurrency.getCurrency().getMinCpmPrice(),
                commonCurrency.getCurrency().getMaxCpmPrice(),
                true);
    }

    private static <C extends CampaignWithStrategy> void checkCampaignsHaveSameCurrency(
            List<AppliedChanges<C>> appliedChanges, CurrencyCode commonCurrency) {
        checkArgument(StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .map(CampaignWithStrategy::getCurrency)
                .allMatch(currency -> currency == commonCurrency), "campaigns should be with same currency");
    }

    public static BigDecimal calculateMinimalAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
            LocalDate start,
            LocalDate finish,
            CurrencyCode currencyCode) {
        return BigDecimal.valueOf(DAYS.between(start, finish) + 1)
                .multiply(currencyCode.getCurrency().getMinDailyBudgetForPeriod());
    }

    public static BigDecimal calculateMaximumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
            LocalDate start,
            LocalDate finish,
            CurrencyCode currencyCode) {
        return BigDecimal.valueOf(DAYS.between(start, finish) + 1)
                .multiply(currencyCode.getCurrency().getMaxDailyBudgetForPeriod());
    }

    public static BigDecimal calculateMinimumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
            CpmCampaignWithCustomStrategy campaign, LocalDate newFinishDate,
            BigDecimal spentFromStatistic, LocalDate now, CurrencyCode currencyCode) {
        return calculateMinimumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
                campaign.getStrategy().getStrategyData().getStart(),
                newFinishDate, campaign.getStrategy().getStrategyData().getBudget(),
                spentFromStatistic, now, currencyCode);
    }

    public static BigDecimal calculateMinimumAvailableBudgetForCpmRestartingStrategyWithCustomPeriod(
            LocalDate oldStartDate, LocalDate newFinishDate, BigDecimal oldBudget,
            BigDecimal spentFromStatistic, LocalDate now, CurrencyCode currencyCode) {
        var minimalBudget = BigDecimal.valueOf(getDaysCountInPeriod(oldStartDate, newFinishDate))
                .multiply(currencyCode.getCurrency().getMinDailyBudgetForPeriod());

        BigDecimal minimalAdditionalBudget = currencyCode.getCurrency().getMinDailyBudgetForPeriod()
                .multiply(BigDecimal.valueOf(DAYS.between(now, newFinishDate)));
        return spentFromStatistic.multiply(BigDecimal.valueOf(1.1)).setScale(0, RoundingMode.HALF_UP).add(minimalAdditionalBudget)
                .min(oldBudget)
                .max(minimalBudget);
    }

    private static Long getDaysCountInPeriod(LocalDate oldStartDate, LocalDate newFinish) {
        return DAYS.between(oldStartDate, newFinish) + 1;
    }

    /**
     * переходим на автобюджет с неавтобюджетной стратегии
     * ИЛИ если переходим с поискового автобюджета на контекстный или наоборот
     */
    public static <C extends CampaignWithStrategy> boolean filterEnableAutoBudget(
            AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
        DbStrategy oldStrategy = ac.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
        if (oldStrategy == null) {
            return false;
        }
        if (!newStrategy.isAutoBudget()) {
            return false;
        }
        return !oldStrategy.isAutoBudget()
                || ((newStrategy.isDifferentPlaces() || oldStrategy.isDifferentPlaces())
                && !newStrategy.getStrategyName().equals(oldStrategy.getStrategyName())
        );
    }

    /**
     * уходим с автобюджета на ручное управление ставками
     */
    public static <C extends CampaignWithStrategy> boolean filterAutoBudgetToManual(
            AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithStrategy.STRATEGY);
        DbStrategy oldStrategy = ac.getOldValue(CampaignWithStrategy.STRATEGY);
        return oldStrategy != null && !newStrategy.isAutoBudget() && oldStrategy.isAutoBudget();
    }

    /**
     * Новая стратегия ручная
     */
    public static <C extends CampaignWithStrategy> boolean filterToManual(
            AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithStrategy.STRATEGY);
        return !newStrategy.isAutoBudget();
    }

    public static <C extends CampaignWithStrategy> boolean filterToAutobudget(AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithStrategy.STRATEGY);
        return newStrategy.isAutoBudget();
    }

    public static <C extends CampaignWithStrategy> boolean filterToAutobudgetRoi(AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithStrategy.STRATEGY);
        return newStrategy != null && nvl(newStrategy.getStrategyName(), "").equals(StrategyName.AUTOBUDGET_ROI);
    }

    public static <C extends CampaignWithDefaultPriceRecalculation> boolean filterAutoBudgetOldOrNew(
            AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
        DbStrategy oldStrategy = ac.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
        return newStrategy.isAutoBudget() || oldStrategy != null && oldStrategy.isAutoBudget();
    }

    /**
     * переход на поисковую стратегию со статегии "независимое управление"
     */
    public static <C extends CampaignWithDefaultPriceRecalculation> boolean filterDifferentToSearch(
            AppliedChanges<C> ac) {
        DbStrategy newStrategy = ac.getNewValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
        if (newStrategy.isDifferentPlaces()) {
            return false;
        }
        DbStrategy oldStrategy = ac.getOldValue(CampaignWithDefaultPriceRecalculation.STRATEGY);
        if (oldStrategy == null) {
            return false;
        }
        return oldStrategy.isDifferentPlaces() || oldStrategy.isAutoBudget();
    }

    public static void modifyLastBidderRestartTime(boolean isRestart, LocalDateTime now,
                                                   AppliedChanges<? extends WithStrategy> appliedChanges) {
        LocalDateTime lastBidderRestartTime = isRestart ?
                now :
                getLastRestartTimeForConversionStrategyWithoutRestart(appliedChanges);

        DbStrategy newStrategy = getStrategyCopyWithLasBidderRestartTime(appliedChanges.getModel().getStrategy(),
                lastBidderRestartTime);
        appliedChanges.modify(CampaignWithRestartOfConversionStrategyTimeSaving.STRATEGY, newStrategy);
    }

    private static LocalDateTime getLastRestartTimeForConversionStrategyWithoutRestart(
            AppliedChanges<? extends WithStrategy> appliedChanges) {
        return appliedChanges
                .getOldValue(CampaignWithRestartOfConversionStrategyTimeSaving.STRATEGY)
                .getStrategyData()
                .getLastBidderRestartTime();
    }

    public static DbStrategy getStrategyCopyWithLasBidderRestartTime(DbStrategy strategy,
                                                                     LocalDateTime lastBidderRestartTime) {
        var newStrategy = strategy.copy();
        StrategyData newStrategyData = newStrategy.getStrategyData().copy();
        newStrategyData.setLastBidderRestartTime(lastBidderRestartTime);
        newStrategy.setStrategyData(newStrategyData);
        return newStrategy;
    }

    public static boolean isStrategyGoalIdChanged(
            AppliedChanges<? extends WithStrategy> appliedChanges) {
        Long oldGoalId = getStrategyGoalId(appliedChanges.getOldValue(
                CampaignWithRestartOfConversionStrategyTimeSaving.STRATEGY));
        Long newGoalId = getStrategyGoalId(appliedChanges.getNewValue(
                CampaignWithRestartOfConversionStrategyTimeSaving.STRATEGY));
        return !Objects.equals(oldGoalId, newGoalId);
    }

    private static Long getStrategyGoalId(DbStrategy strategy) {
        return strategy.getStrategyData().getGoalId();
    }

    public static boolean isPayForConversionStrategyData(@Nullable StrategyData strategyData) {
        return strategyData != null && (strategyData.getAvgCpa() != null || strategyData.getAvgCpi() != null) && strategyData.getPayForConversion() != null && strategyData.getPayForConversion();
    }

    /**
     * Проверка достаточно ли средств в кампании с  Оплатой за конверсии для оплаты будущих конверсий.
     * Сравнивает остаток на кампании с пороговыми значениями max(N * priceOfConversion , M),
     * где N - кол-во конверсий и M - предполагаемая сумма конверсий на ближайшие 21 день
     * (биллинг для разных типов атрибуции происходит в течение 21го дня)
     * Подразумевается, что сюда передаётся стратегий, где включена оплата за конверсии.
     * Для стратегий отличных от CPA, CPI всегда возвращает false
     *
     * @param sumTotal           - остаток средств на кампании с учетом общего счета
     * @param strategyData       - параметры стратегии
     * @param avgCpaWarningRatio - количество конверсий
     * @param minReservedBudget  - минимально допустимый остаток для оплаты будущих конверсий
     */
    @Nullable
    public static boolean hasLackOfFundsOnCampaignWithPayForConversion(BigDecimal sumTotal,
                                                                       StrategyData strategyData,
                                                                       BigDecimal avgCpaWarningRatio,
                                                                       BigDecimal minReservedBudget) {
        var priceOfConversion = strategyData.getAvgCpa() != null ? strategyData.getAvgCpa() : strategyData.getAvgCpi();
        if (priceOfConversion == null) {
            return false;
        }
        return sumTotal.compareTo(max(priceOfConversion.multiply(avgCpaWarningRatio), minReservedBudget)) < 0;
    }

    public static boolean isAttributionModelAvailableForSimpleView(CampaignAttributionModel attributionModel,
                                                                   CampaignAttributionModel defaultAttributionModel) {
        return attributionModel == defaultAttributionModel;
    }

    public static boolean isStrategyAvailableForSimpleView(DbStrategy strategy) {
        StrategyData strategyData = strategy.getStrategyData();

        boolean strategyHasProProperties = StreamEx.of(StrategyData.allModelProperties())
                .remove(AVAILABLE_PROPERTIES_FOR_SIMPLE_STRATEGY::contains)
                .map(property -> property.get(strategyData))
                .nonNull()
                .findAny()
                .isPresent();

        boolean strategyNameAvailableForSimpleView = strategyData.getName() != null &&
                AVAILABLE_STRATEGY_NAMES_FOR_SIMPLE_STRATEGY.contains(strategyData.getName());

        return strategyNameAvailableForSimpleView &&
                !strategyHasProProperties;
    }

    public static boolean isAggregatorGoalId(Long goalId) {
        return goalId == MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
    }

    public static boolean shouldFetchUnavailableGoals(BaseCampaign campaign,
                                                      Set<String> enabledFeatures) {
        if (shouldFetchUnavailableAutoGoals(campaign, enabledFeatures)) {
            return true;
        }
        return isUnavailableGoalsAllowed(campaign, enabledFeatures);
    }

    /**
     * На кампании можно использовать автоцели без доступа,
     * если автоцель используется в стратегии оптимизации конверсий с оплатой за клики или
     * если автоцели используются в ключевых целях.
     */
    private static boolean shouldFetchUnavailableAutoGoals(BaseCampaign campaign, Set<String> enabledFeatures) {
        if (campaign instanceof CampaignWithCustomStrategy) {
            if (isUnavailableAutoGoalsAllowedForCampaignWithStrategy(
                    (CampaignWithCustomStrategy) campaign, enabledFeatures)) {
                return true;
            }
        } else if (campaign instanceof CampaignWithStrategy) {
            if (isUnavailableAutoGoalsAllowedForCampaignWithStrategy(
                    (CampaignWithStrategy) campaign, enabledFeatures)) {
                return true;
            }
        }
        if (campaign instanceof CampaignWithMeaningfulGoalsWithRequiredFields) {
            if (isUnavailableAutoGoalsAllowedForCampaignWithSource(
                    (CampaignWithMeaningfulGoalsWithRequiredFields) campaign, enabledFeatures)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isUnavailableGoalsAllowed(
            BaseCampaign campaign,
            Set<String> enabledFeatures) {
        if (campaign instanceof CampaignWithSource) {
            var source = ((CampaignWithSource) campaign).getSource();
            return isUnavailableGoalsAllowed(source, enabledFeatures);
        }
        return false;
    }

    public static boolean isUnavailableGoalsAllowed(
            CampaignSource campaignSource,
            Set<String> enabledFeatures) {
        boolean allowedForUac = AvailableCampaignSources.INSTANCE.isUC(campaignSource) &&
                enabledFeatures.contains(FeatureName.UAC_UNAVAILABLE_GOALS_ALLOWED.getName());
        boolean allowedForDirect = !AvailableCampaignSources.INSTANCE.isUC(campaignSource) &&
                enabledFeatures.contains(FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED.getName());
        return allowedForUac || allowedForDirect;
    }

    public static boolean isUnavailableAutoGoalsAllowedForCampaignWithStrategy(CampaignWithCustomStrategy campaign,
                                                                               Set<String> enabledFeatures) {
        if (campaign instanceof CampaignWithSource) {
            var dbStrategy = campaign.getStrategy();
            var source = ((CampaignWithSource) campaign).getSource();
            return isUnavailableAutoGoalsAllowedForCampaignWithSource(source, enabledFeatures)
                    && isConversionOptimizationStrategyWithPayForClicks(
                    dbStrategy.getStrategyName(),
                    dbStrategy.getStrategyData().getGoalId(),
                    dbStrategy.getStrategyData().getAvgCpa(),
                    dbStrategy.getStrategyData().getPayForConversion());
        }
        return false;
    }

    public static boolean isUnavailableAutoGoalsAllowedForCampaignWithStrategy(
            CampaignWithStrategy campaign, Set<String> enabledFeatures) {
        var dbStrategy = campaign.getStrategy();
        var source = ((CampaignWithSource) campaign).getSource();
        return isUnavailableAutoGoalsAllowedForCampaignWithSource(source, enabledFeatures)
                && isConversionOptimizationStrategyWithPayForClicks(
                dbStrategy.getStrategyName(),
                dbStrategy.getStrategyData().getGoalId(),
                dbStrategy.getStrategyData().getAvgCpa(),
                dbStrategy.getStrategyData().getPayForConversion());
    }

    public static boolean isUnavailableAutoGoalsAllowedForCampaignWithStrategy(CampaignSource campaignSource,
                                                                               @Nullable StrategyName strategyName,
                                                                               @Nullable Long goalId,
                                                                               @Nullable BigDecimal avgCpa,
                                                                               @Nullable Boolean isPayForConversion,
                                                                               Set<String> enabledFeatures) {
        // Для автоцелей без доступа даём настроить стратегию оптимизации конверсий с оплатой за клики
        return isUnavailableAutoGoalsAllowedForCampaignWithSource(campaignSource, enabledFeatures)
                && isConversionOptimizationStrategyWithPayForClicks(
                    strategyName,
                    goalId,
                    avgCpa,
                    isPayForConversion);
    }

    public static boolean isUnavailableAutoGoalsAllowedForCampaignWithSource(CampaignWithSource campaign,
                                                                             Set<String> enabledFeatures) {
        return isUnavailableAutoGoalsAllowedForCampaignWithSource(campaign.getSource(), enabledFeatures);
    }

    public static boolean isUnavailableAutoGoalsAllowedForCampaignWithSource(CampaignSource campaignSource,
                                                                             Set<String> enabledFeatures) {
        boolean allowedForUac = AvailableCampaignSources.INSTANCE.isUC(campaignSource) &&
                enabledFeatures.contains(FeatureName.UAC_UNAVAILABLE_AUTO_GOALS_ALLOWED.getName());
        boolean allowedForDirect = CampaignUtils.isDirectSource(campaignSource) &&
                enabledFeatures.contains(FeatureName.DIRECT_UNAVAILABLE_AUTO_GOALS_ALLOWED.getName());
        boolean allowedBySource = campaignSource == CampaignSource.GEO;
        return allowedForUac || allowedForDirect || allowedBySource;
    }

    public static boolean strategySupportsDayBudget(DbStrategy strategy) {
        return !strategy.isAutoBudget() && (MANUAL_PRICE_STRATEGIES.contains(strategy.getStrategyName())
                || CampOptionsStrategy.DIFFERENT_PLACES.equals(strategy.getStrategy()));
    }

    public static Map<Long, Long> getGoalIdToCounterIdForCampaignsWithoutCounterIds(
            Collection<? extends BaseCampaign> preValidItems,
            RequestBasedMetrikaClientAdapter metrikaClient) {
        var goalIds = StreamEx.of(preValidItems.stream())
                .select(CampaignWithCustomStrategy.class)
                // нет смысла загружать counterId из метрики, если он уже указан в запросе
                .filter(it -> extractCounterIds(it).size() == 0)
                .map(CampaignConverter::getCampaignGoalId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        return getGoalIdToCounterIdMap(goalIds, metrikaClient);
    }

    public static Map<Long, Long> getGoalIdToCounterIdMap(
            Set<Long> goalIds,
            RequestBasedMetrikaClientAdapter metrikaClient) {
        var metrikaGoals = metrikaClient.getGoalsByUid(goalIds);

        return EntryStream.of(metrikaGoals)
                .values()
                .flatMap(List::stream)
                .distinct(RetargetingCondition::getId)
                .toMap(RetargetingCondition::getId, it -> (long) it.getCounterId());
    }

    private static boolean isConversionOptimizationStrategyWithPayForClicks(
            @Nullable StrategyName strategyName,
            @Nullable Long goalId,
            @Nullable BigDecimal avgCpa,
            @Nullable Boolean isPayForConversion) {
        return strategyName == StrategyName.AUTOBUDGET
                && goalId != null
                && avgCpa == null
                && !Boolean.TRUE.equals(isPayForConversion);
    }

}
