package ru.yandex.direct.core.entity.campaign.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.audience.client.YaAudienceClient;
import ru.yandex.direct.audience.client.model.CreateExperimentRequest;
import ru.yandex.direct.audience.client.model.CreateExperimentResponse;
import ru.yandex.direct.audience.client.model.ExperimentSegmentRequest;
import ru.yandex.direct.audience.client.model.SetExperimentGrantRequest;
import ru.yandex.direct.core.entity.brandSurvey.BrandSurvey;
import ru.yandex.direct.core.entity.brandlift.repository.BrandSurveyRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.client.model.ClientExperiment;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BRAND_LIFT_EXPERIMENT_THRESHOLD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BRAND_LIFT_SURVEYS_LOGIN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SEGMENT_GOAL_ID_SHIFT;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Service
@ParametersAreNonnullByDefault
public class CampaignWithBrandLiftExperimentsService {
    private static final String EXPERIMENT_PERMISSION = "view";
    public static final long DEFAULT_METRIKA_COUNTER_FOR_AB = 0L;
    public static final String BRANDLIFT_BK_SETTINGS =
            "{\"is_yandex_experiment\": true, \"expression\": [[[\"page-id\", \"greater\", \"0\"]]]}";
    private static final String EXPERIMENT_NAME_FORMAT = "Client %d global segment";

    private final YaAudienceClient yaAudienceClient;
    private final BrandSurveyRepository brandSurveyRepository;
    private final ShardHelper shardHelper;
    private final ClientService clientService;

    @Autowired
    public CampaignWithBrandLiftExperimentsService(YaAudienceClient yaAudienceClient,
                                                   BrandSurveyRepository brandSurveyRepository, ShardHelper shardHelper,
                                                   ClientService clientService) {
        this.yaAudienceClient = yaAudienceClient;
        this.brandSurveyRepository = brandSurveyRepository;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
    }

    public Map<Long, Long> prepareExperiment(String clientLogin, CampaignWithBrandLift campaign) {

        ClientExperiment experiment = getExperimentOrCreate(ClientId.fromLong(campaign.getClientId()), clientLogin);

        Long experimentId = experiment.getExperimentId();
        Long segmentGoalId = experiment.getSegmentId();

        // не сохраняется в базе, необходим для метода renameExperiment, который вызывается в afterExecution
        campaign.setExperimentId(experimentId);

        campaign.setSectionIds(nvl(campaign.getSectionIds(), new ArrayList<>()));
        campaign.setAbSegmentGoalIds(nvl(campaign.getAbSegmentGoalIds(), new ArrayList<>()));

        campaign.getSectionIds().add(experimentId);
        campaign.getAbSegmentGoalIds().add(segmentGoalId);

        return Map.of(experimentId, segmentGoalId);
    }

    public ClientExperiment createExperiment(String clientLogin, String experimentName) {
        CreateExperimentRequest createExperimentRequest = createExperimentRequest(experimentName);

        CreateExperimentResponse experiment = yaAudienceClient.createExperiment(clientLogin,
                createExperimentRequest).getCreateExperimentResponse();

        Long experimentId = experiment.getExperimentId();

        //Сразу дёргаем апдейт, так как на создании игнорируются флаги bkSettings
        yaAudienceClient.updateExperiment(clientLogin, experimentId, createExperimentRequest);

        SetExperimentGrantRequest setExperimentGrantRequest = new SetExperimentGrantRequest()
                .withUserLogin(BRAND_LIFT_SURVEYS_LOGIN)
                .withPermission(EXPERIMENT_PERMISSION);

        yaAudienceClient.setExperimentGrant(experimentId, setExperimentGrantRequest);

        return new ClientExperiment()
                .withExperimentId(experiment.getExperimentId())
                .withSegmentId(SEGMENT_GOAL_ID_SHIFT + experiment.getExperimentSegments().get(0).getSegmentId());
    }

    public void renameExperiment(String clientLogin, CampaignWithBrandLift campaign) {
        long experimentId = campaign.getExperimentId();
        CreateExperimentRequest request = createExperimentRequest(String.format(EXPERIMENT_NAME_FORMAT, campaign.getClientId()));

        yaAudienceClient.updateExperiment(clientLogin, experimentId, request);
    }

    public void addBrandSurvey(String clientLogin, BrandSurvey brandSurvey) {
        brandSurveyRepository.addBrandSurvey(shardHelper.getShardByLoginStrictly(clientLogin), brandSurvey);
    }

    public void renameBrandSurvey(String clientLogin, String brandSurveyId, String newName) {
        brandSurveyRepository.renameBrandSurvey(shardHelper.getShardByLoginStrictly(clientLogin), brandSurveyId, newName);
    }

    public List<BrandSurvey> clientBrandLifts(ClientId clientId) {
        return brandSurveyRepository
                .getClientBrandSurveys(shardHelper.getShardByClientId(clientId), clientId.asLong());
    }

    public ClientExperiment getExperimentOrCreate(ClientId clientId, String clientLogin) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return clientService.getExperiment(shard, clientId.asLong())
                .orElseGet(() -> {

                    checkNotNull(clientLogin,
                            "login for creating brandlift is null for ClientId=" + clientId.asLong());

                    ClientExperiment experiment = createExperiment(clientLogin,
                            String.format(EXPERIMENT_NAME_FORMAT, clientId.asLong()));
                    clientService.saveClientExperiment(shard, clientId.asLong(), experiment);
                    return experiment;
                });
    }

    private static CreateExperimentRequest createExperimentRequest(String experimentName) {
        ExperimentSegmentRequest segmentA = new ExperimentSegmentRequest()
                .withName("A")
                .withStart(0)
                .withEnd(BRAND_LIFT_EXPERIMENT_THRESHOLD);

        ExperimentSegmentRequest segmentB = new ExperimentSegmentRequest()
                .withName("B")
                .withStart(BRAND_LIFT_EXPERIMENT_THRESHOLD)
                .withEnd(100);

        return new CreateExperimentRequest()
                .withExperimentName(experimentName)
                .withBkSettings(BRANDLIFT_BK_SETTINGS)
                .withExperimentSegmentRequests(List.of(segmentA, segmentB))
                .withCounterIds(singletonList(DEFAULT_METRIKA_COUNTER_FOR_AB));
    }
}
