package ru.yandex.direct.core.entity.campaign.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandSafety;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class CampaignWithBrandSafetyService {

    private static final String BRAND_SAFETY_CONDITION_NAME = "brandsafety condition";
    private static final String BRAND_SAFETY_CONDITION_DESCRIPTION = "brandsafety categories";

    private final RetargetingConditionRepository retargetingConditionRepository;
    private final ShardHelper shardHelper;

    public CampaignWithBrandSafetyService(RetargetingConditionRepository retargetingConditionRepository,
                                          ShardHelper shardHelper) {
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.shardHelper = shardHelper;
    }

    public Map<Long, List<Long>> getCategories(Collection<Long> campaignIds) {
        Map<Long, RetargetingCondition> conditions = getRetargetingConditions(campaignIds);
        return EntryStream.of(conditions).mapValues(CampaignWithBrandSafetyService::convertCondition).toMap();
    }

    @NotNull
    public Map<Long, RetargetingCondition> getRetargetingConditions(Collection<Long> campaignIds) {
        Map<Long, RetargetingCondition> conditions = new HashMap<>();
        shardHelper.groupByShard(campaignIds, ShardKey.CID).forEach((shard, campsForShard) ->
                conditions.putAll(retargetingConditionRepository
                        .getBrandSafetyRetConditionsByCampaignIds(shard, campsForShard)));
        return conditions;
    }

    public Map<Long, Long> saveCategoriesAndGetRetCondIds(int shard,
                                                          ClientId clientId,
                                                          Collection<CampaignWithBrandSafety> campaigns) {
        List<RetargetingCondition> existingBrandSafetyRetConditions = retargetingConditionRepository
                .getBrandSafetyRetConditionsByClient(shard, clientId);

        Map<Rule, Long> idByExistingRetConditionRule = existingBrandSafetyRetConditions
                .stream()
                .collect(Collectors.toMap(
                        retCondition -> retCondition.getRules().get(0),
                        RetargetingCondition::getId,
                        // из двух одинаковых retCondition нужно какой-то выбрать, выберем с наименьшим id
                        Math::min
                ));

        List<CampaignWithBrandSafety> campaignsWithNotEmptyCategories = filterList(campaigns,
                campaign -> !campaign.getBrandSafetyCategories().isEmpty());

        List<CampaignWithBrandSafety> campaignsForAdd = filterList(campaignsWithNotEmptyCategories,
                campaign -> !idByExistingRetConditionRule.containsKey(createRuleForCategories(campaign)));

        List<Long> addedRetargetingConditionIds = addBrandSafetyCategories(shard, clientId, campaignsForAdd);

        List<Rule> addedRules = mapList(campaignsForAdd, CampaignWithBrandSafetyService::createRuleForCategories);
        Map<Rule, Long> addedRulesbyId = EntryStream.zip(addedRules, addedRetargetingConditionIds).toMap();

        idByExistingRetConditionRule.putAll(addedRulesbyId);

        return listToMap(campaignsWithNotEmptyCategories, CampaignWithBrandSafety::getId,
                campaign -> idByExistingRetConditionRule.get(createRuleForCategories(campaign)));
    }

    private List<Long> addBrandSafetyCategories(int shard, ClientId clientId, Collection<CampaignWithBrandSafety> campaigns) {
        List<RetargetingCondition> conditions = mapList(campaigns,
                campaign -> createRetargetingConditionForCategories(clientId, campaign));
        return retargetingConditionRepository.add(shard, conditions);
    }

    private static RetargetingCondition createRetargetingConditionForCategories(ClientId clientId, CampaignWithBrandSafety campaignWithBrandSafety) {
        Rule rule = createRuleForCategories(campaignWithBrandSafety);
        RetargetingCondition condition = new RetargetingCondition();
        condition.withRules(singletonList(rule))
                .withType(ConditionType.brandsafety)
                .withClientId(clientId.asLong())
                .withName(BRAND_SAFETY_CONDITION_NAME)
                .withDescription(BRAND_SAFETY_CONDITION_DESCRIPTION)
                .withDeleted(false)
                .withInterest(false)
                .withLastChangeTime(LocalDateTime.now());
        return condition;
    }

    private static Rule createRuleForCategories(CampaignWithBrandSafety campaignWithBrandSafety) {
        List<Goal> goals = mapList(campaignWithBrandSafety.getBrandSafetyCategories(),
                categoryId -> (Goal) new Goal()
                        .withType(GoalType.BRANDSAFETY)
                        .withId(categoryId)
                        .withTime(0));
        return new Rule()
                .withGoals(goals)
                .withType(RuleType.NOT);
    }

    private static List<Long> convertCondition(RetargetingCondition condition) {
        return mapList(condition.collectGoals(), Goal::getId);
    }
}
