package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithPricePackage;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSize;
import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSizeWithExpand;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.InventoryType;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.pricepackage.model.PriceMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCampaignOptions;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageWithoutClients;
import ru.yandex.direct.core.entity.pricepackage.model.ShowsFrequencyLimit;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.FunctionalUtils;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.FULLSCREEN;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INAPP;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INBANNER;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INPAGE;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INROLL;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INROLL_OVERLAY;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INSTREAM_WEB;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INTERSTITIAL;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.MIDROLL;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.OVERLAY;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.PAUSEROLL;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.POSTROLL;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.POSTROLL_OVERLAY;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.POSTROLL_WRAPPER;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.PREROLL;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.REWARDED;
import static ru.yandex.direct.core.entity.campaign.service.pricerecalculation.PriceCalculator.getSeasonalPriceRatio;
import static ru.yandex.direct.core.entity.campaign.service.pricerecalculation.PriceCalculator.percentIntegerToBigDecimal;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.DateTimeUtils.inPastOrToday;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.FunctionalUtils.setUnion;

@ParametersAreNonnullByDefault
public class CampaignWithPricePackageUtils {

    public static final CreativeSize CPM_PRICE_PREMIUM_FORMAT_CREATIVE_SIZE = new CreativeSize(1836L, 572L);
    public static final Set<CreativeSize> CPM_PRICE_PREMIUM_FORMAT_MOBILE_CREATIVE_SIZE =
            Set.of(new CreativeSize(640L, 268L),
                    new CreativeSize(640L, 335L),
                    new CreativeSize(640L, 201L));
    public static final CreativeSize DESKTOP_DEFAULT_CREATIVE_SIZE = new CreativeSize(1456L, 180L);
    public static final CreativeSize MOBILE_DEFAULT_CREATIVE_SIZE = new CreativeSize(640L, 134L);

    //Each value contains a set of  equivalent formats, so only one of them is needed to make campaign full
    public static final ImmutableSetMultimap<ViewType, Set<CreativeSize>> VIEW_TYPE_TO_CREATIVE_SIZES =
            ImmutableSetMultimap.of(
                    ViewType.DESKTOP, Set.of(DESKTOP_DEFAULT_CREATIVE_SIZE, CPM_PRICE_PREMIUM_FORMAT_CREATIVE_SIZE),
                    ViewType.MOBILE, ImmutableSet.<CreativeSize>builder()
                            .add(MOBILE_DEFAULT_CREATIVE_SIZE)
                            .addAll(CPM_PRICE_PREMIUM_FORMAT_MOBILE_CREATIVE_SIZE)
                            .build(),
                    ViewType.NEW_TAB, Set.of(new CreativeSize(1456L, 180L))
            );

    public static final Map<InventoryType, List<InventoryType>> ACTUAL_INVENTORY_WIDE_TO_NARROW = Map.of(
            INSTREAM_WEB, List.of(
                    PREROLL,
                    MIDROLL,
                    POSTROLL,
                    PAUSEROLL,
                    OVERLAY,
                    POSTROLL_OVERLAY,
                    POSTROLL_WRAPPER,
                    INROLL_OVERLAY,
                    INROLL),
            INAPP, List.of(
                    INTERSTITIAL,
                    FULLSCREEN
            ),
            INPAGE, emptyList(),
            INBANNER, emptyList(),
            REWARDED, emptyList()
    );

    @Nullable
    public static DbStrategy getStrategy(CampaignWithPricePackage campaign, @Nullable PricePackage pricePackage,
                                         Boolean backendCpmPriceCampaignBudgetCalcEnabled) {
        if (pricePackage == null) {
            return null;
        }
        checkState(pricePackage.getId().equals(campaign.getPricePackageId()));

        return (DbStrategy) new DbStrategy()
                .withStrategy(CampOptionsStrategy.DIFFERENT_PLACES)
                .withStrategyName(StrategyName.PERIOD_FIX_BID)
                .withStrategyData(new StrategyData()
                        .withName(CampaignsStrategyName.period_fix_bid.getLiteral())
                        .withVersion(1L)
                        // автопродление включается настройкой в прайсовом видео. Для главной всегда выключено
                        .withAutoProlongation(
                                !pricePackage.isFrontpagePackage()
                                        && campaign.getAutoProlongation() != null && campaign.getAutoProlongation()
                                        ? 1L : 0L)
                        .withStart(campaign.getStartDate())
                        .withFinish(campaign.getEndDate())
                        .withBudget(calculateBudget(campaign, pricePackage, backendCpmPriceCampaignBudgetCalcEnabled)))
                .withAutobudget(CampaignsAutobudget.NO)
                .withPlatform(CampaignsPlatform.CONTEXT);
    }

    @Nullable
    public static BigDecimal calcPackagePrice(CampaignWithPricePackage campaign, PricePackage pricePackage,
                                                Boolean returnNullIfNoSeasonFound,
                                                Boolean backendCpmPriceCampaignBudgetCalcEnabled) {

        BigDecimal totalPrice = pricePackage.getPrice();
        if (backendCpmPriceCampaignBudgetCalcEnabled) {
            PriceMarkup seasonalPriceMarkup = getSeasonalPriceRatio(campaign, pricePackage);
            if (returnNullIfNoSeasonFound && seasonalPriceMarkup == null) {
                return null;
            }
            Integer percent = (seasonalPriceMarkup == null) ? 0 : seasonalPriceMarkup.getPercent();
            totalPrice = pricePackage.getPrice().multiply(percentIntegerToBigDecimal(percent));
        }
        return totalPrice;
    }

    @Nullable
    public static BigDecimal calculateBudget(CampaignWithPricePackage campaign, PricePackage pricePackage,
                                              Boolean backendCpmPriceCampaignBudgetCalcEnabled) {
        Long flightOrderVolume = campaign.getFlightOrderVolume();
        BigDecimal pricePackagePrice = pricePackage.getPrice();
        CurrencyCode pricePackageCurrency = pricePackage.getCurrency();
        if (nvl(pricePackage.getIsCpd(), false)) {
            return pricePackagePrice;
        }

        if (flightOrderVolume == null) {
            return null;
        }
        checkNotNull(pricePackagePrice);
        checkNotNull(pricePackageCurrency);

        BigDecimal totalPrice = calcPackagePrice(campaign, pricePackage, false,
                backendCpmPriceCampaignBudgetCalcEnabled);
        return calcBudgetByVolume(totalPrice, pricePackageCurrency, flightOrderVolume);
    }

    public static BigDecimal calcBudgetByVolume(BigDecimal price, CurrencyCode currency, Long volume) {
        // вычисляется по формуле: p * v / 1000, где
        // p: cpm из пакета
        // v: объём заказа
        // округляем в большую сторону, чтобы объём гарантировано открутился
        return Money.valueOf(price, currency)
                .multiply(volume)
                .divide(1000L)
                .roundToCentUp()
                .bigDecimalValue();
    }

    public static Long calcVolumeByCurrency(BigDecimal price, CurrencyCode currency, BigDecimal budget) {
        if (budget == null) {
            return 0L;
        }
        return budget
                .multiply(BigDecimal.valueOf(1000L))
                .divide(Money.valueOf(price, currency).bigDecimalValue(), 0, RoundingMode.DOWN)
                .longValue();
    }

    public static Set<Set<CreativeSizeWithExpand>> collectCampaignCreativeSizesWithExpand(CpmPriceCampaign campaign) {
        boolean allowExpandedDesktopCreative = campaign.getFlightTargetingsSnapshot().getAllowExpandedDesktopCreative();

        return campaign.getFlightTargetingsSnapshot().getViewTypes().stream()
                .map(viewType -> {
                    boolean allowExpanded = viewType == ViewType.DESKTOP && allowExpandedDesktopCreative;

                    return VIEW_TYPE_TO_CREATIVE_SIZES.get(viewType)
                            .stream()
                            .map(formatSet -> mapSet(formatSet, c -> new CreativeSizeWithExpand(c.getWidth(),
                                    c.getHeight(), allowExpanded)))
                            .collect(Collectors.toSet());
                })
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
    }

    @Deprecated
    public static boolean isBannerActive(OldBanner banner) {
        return banner.getStatusShow() && !banner.getStatusArchived()
                && banner.getStatusModerate() == OldBannerStatusModerate.YES
                && banner.getStatusPostModerate() == OldBannerStatusPostModerate.YES;
    }

    public static boolean isBannerActive(BannerWithPricePackage banner) {
        return isBannerActive(
                banner.getStatusShow(),
                banner.getStatusArchived(),
                banner.getStatusModerate(),
                banner.getStatusPostModerate());
    }

    public static boolean isBannerNonRejectedOrArchived(BannerWithPricePackage banner) {
        return !banner.getStatusArchived()
                && banner.getStatusPostModerate() != BannerStatusPostModerate.REJECTED
                && banner.getStatusShow()
                && banner.getStatusModerate() != BannerStatusModerate.NO;
    }
    public static boolean wasBannerActive(AppliedChanges<BannerWithPricePackage> changes) {
        return isBannerActive(
                changes.getOldValue(BannerWithPricePackage.STATUS_SHOW),
                changes.getOldValue(BannerWithPricePackage.STATUS_ARCHIVED),
                changes.getOldValue(BannerWithPricePackage.STATUS_MODERATE),
                changes.getOldValue(BannerWithPricePackage.STATUS_POST_MODERATE));
    }

    public static boolean isBannerActive(boolean statusShow,
                                         boolean statusArchived,
                                         BannerStatusModerate statusModerate,
                                         BannerStatusPostModerate statusPostModerate) {
        return statusShow
                && !statusArchived
                && statusModerate == BannerStatusModerate.YES
                && statusPostModerate == BannerStatusPostModerate.YES;
    }

    public static boolean isCampaignStarted(CpmPriceCampaign campaign) {
        return isCampaignReadyToStart(campaign) && inPastOrToday(campaign.getStartDate());
    }

    public static boolean isCampaignReadyToStart(CpmPriceCampaign campaign) {
        return campaign.getFlightStatusApprove() == PriceFlightStatusApprove.YES
                && campaign.getFlightStatusCorrect() == PriceFlightStatusCorrect.YES
                && campaign.getStatusShow();
    }

    /**
     * Проверяет что среди переданных баннеров под каждый требуемый формат морды кампании есть хотя бы один активный
     * баннер.
     */
    public static boolean isCampaignFullWithBanners(CpmPriceCampaign campaign, List<BannerWithPricePackage> banners,
                                                    Map<Long, Creative> bannerIdToCreatives) {
        Set<Set<CreativeSizeWithExpand>> requiredCreativeSizesWithExpand =
                collectCampaignCreativeSizesWithExpand(campaign);

        if (requiredCreativeSizesWithExpand.stream().mapToLong(Set::size).sum() == 0) {
            return false;
        }

        List<BannerWithPricePackage> activeBanners =
                filterList(banners, CampaignWithPricePackageUtils::isBannerActive);
        List<Creative> activeBannersCreatives = activeBanners.stream()
                .map(BannerWithPricePackage::getId)
                .map(bannerIdToCreatives::get)
                .collect(Collectors.toList());
        if (!activeBannersCreatives.isEmpty()
                && StreamEx.of(activeBannersCreatives).allMatch(it -> it.getType() == CreativeType.CPM_VIDEO_CREATIVE)) {
            //Для видео на главной достаточно чтобы был один активный баннер
            //Видео не имеет размеров и кодируется в разные.
            // Поэтому логика проверки нужных размеров креатива для неё не нужна
            // У видео есть соотношение сторон, но в данном случае оно одно и достаточно наличие одного активного баннера
            return true;
        }

        return hasCreativeForEachFormat(requiredCreativeSizesWithExpand, activeBannersCreatives);
    }

    private static boolean hasCreativeForEachFormat(Set<Set<CreativeSizeWithExpand>> requiredCreativeSizesWithExpand,
                                                    Collection<Creative> creatives) {

        Set<CreativeSizeWithExpand> creativeSizesWithExpand = listToSet(creatives, CreativeSizeWithExpand::new);

        for (var equivalentFormatsSet : requiredCreativeSizesWithExpand) {
            if (Sets.intersection(equivalentFormatsSet, creativeSizesWithExpand).isEmpty()) {
                return false;
            }
        }

        return true;
    }

    /**
     * @return true, если пакет не null и в пакете есть заданное фиксированное ограничение частоты показов
     */
    public static boolean pricePackageHasImpressionRate(@Nullable PricePackageWithoutClients pricePackage) {
        return Optional.ofNullable(pricePackage)
                .map(PricePackageWithoutClients::getCampaignOptions)
                .map(PricePackageCampaignOptions::getShowsFrequencyLimit)
                .filter(it -> it.getMinLimit() == null || !it.getMinLimit())
                .map(ShowsFrequencyLimit::getFrequencyLimit).isPresent();
    }

    /**
     * @return true, если пакет не null и разрешает чтобы в прайсовой кампании была частота показов
     */
    public static boolean pricePackageAllowsImpressionRate(@Nullable PricePackageWithoutClients pricePackage) {
        return Optional.ofNullable(pricePackage)
                .map(PricePackageWithoutClients::getCampaignOptions)
                .map(PricePackageCampaignOptions::getShowsFrequencyLimit)
                .isPresent();
    }

    /**
     * Получение настройки {@code impressionRateIntervalDays} кампании из настроек пакета.
     * Если ограничение действует на всё время проведения кампании, возвращается {@code null},
     * иначе - число дней, в течение которых действует ограничение.
     */
    @Nullable
    public static Integer extractImpressionRateIntervalDaysFromPackage(PricePackageWithoutClients pricePackage) {
        var sfr = pricePackage.getCampaignOptions().getShowsFrequencyLimit();
        var limitIsForCampaignTime = Boolean.TRUE.equals(sfr.getFrequencyLimitIsForCampaignTime());
        return limitIsForCampaignTime ? null : sfr.getFrequencyLimitDays();
    }

    /**
     * Устанавливает в кампании корректировки на тип инвентаря, если того требует пакет
     *
     * @param pricePackage
     * @return
     */
    public static List<BidModifier> enrichBidModifierInventory(@Nullable PricePackage pricePackage,
                                                               CampaignWithPricePackage campaign) {
        if (pricePackage == null
                || extractBidModifierInventory(pricePackage.getBidModifiers()) == null) {
            // Если на пакете нет корректировок инвентаря, то и не может быть на кампании.
            return emptyList();
        }

        var inventoryAdjustmentsList = extractInventoryAdjustmentsList(
                pricePackage, campaign);
        if (inventoryAdjustmentsList.isEmpty()) {
            // если нет adjustment-ов, то считаем нет и корректировок.
            return emptyList();
        }
        var campBidModifierInventory = extractBidModifierInventory(campaign.getBidModifiers());
        if (campBidModifierInventory == null) {
            campBidModifierInventory = new BidModifierInventory()
                    .withCampaignId(campaign.getId())
                    .withType(BidModifierType.INVENTORY_MULTIPLIER);
            if (campaign.getBidModifiers() == null) {
                campaign.setBidModifiers(new ArrayList<>());
            }
            campaign.getBidModifiers().add(campBidModifierInventory);
        }
        campBidModifierInventory.setInventoryAdjustments(inventoryAdjustmentsList);
        return campaign.getBidModifiers();
    }

    private static List<BidModifierInventoryAdjustment> extractInventoryAdjustmentsList(
            PricePackage pricePackage,
            CampaignWithPricePackage campaign) {

        List<BidModifierInventoryAdjustment> packageCheckedInventoryTypes =
                extractBidModifierInventory(pricePackage.getBidModifiers()).getInventoryAdjustments();

        Set<InventoryType> allowedOnPackageInventoryTypes = packageCheckedInventoryTypes.stream()
                .map(BidModifierInventoryAdjustment::getInventoryType)
                .collect(Collectors.toSet());

        // Если на пакете есть хотябы одна корректировка из расширенных, то считаем,
        // что имеем дело с расширенными корректировками.
        boolean usingExpandedInventoryBidModifiers = allowedOnPackageInventoryTypes.stream().anyMatch(
                el -> FunctionalUtils.flatMap(ACTUAL_INVENTORY_WIDE_TO_NARROW.values(), identity()).contains(el));

        Set<InventoryType> checkedInventoryTypes = packageCheckedInventoryTypes.stream()
                .filter(BidModifierInventoryAdjustment::getIsRequiredInPricePackage)
                .map(BidModifierInventoryAdjustment::getInventoryType)
                .collect(Collectors.toSet());

        var campBidModifierInventory = extractBidModifierInventory(campaign.getBidModifiers());
        var inventoryAdjustmentsList = campBidModifierInventory == null ? null
                : campBidModifierInventory.getInventoryAdjustments();
        if (inventoryAdjustmentsList != null) {
            // все корректировки с процентом > 0 считаем чекнутыми на фронте
            checkedInventoryTypes.addAll(inventoryAdjustmentsList.stream()
                    .filter(adj -> adj.getPercent() > 0)
                    .map(BidModifierInventoryAdjustment::getInventoryType)
                    .filter(allowedOnPackageInventoryTypes::contains)
                    .collect(Collectors.toSet()));
        }
        List<BidModifierInventoryAdjustment> inventoryAdjustmentsListForSave = new ArrayList<>();

        Set<InventoryType> inventoryTypeSet = new HashSet<>();

        ACTUAL_INVENTORY_WIDE_TO_NARROW.keySet().forEach(
                key -> {
                    List<InventoryType> values = ACTUAL_INVENTORY_WIDE_TO_NARROW.get(key);
                    if (!values.isEmpty()) {
                        // если выбраны все узкие типы или родительский широкий, тогда его и всех детей надо добавить
                        // в список отсутствия корректировок
                        if (checkedInventoryTypes.containsAll(values) || checkedInventoryTypes.contains(key)) {
                            inventoryTypeSet.add(key);
                            inventoryTypeSet.addAll(values);
                        }
                        // если есть выбраны узкие, но не все, то добавляем родительский тип к списку
                        else if (checkedInventoryTypes.stream().anyMatch(values::contains)) {
                            if (usingExpandedInventoryBidModifiers) {
                                // Если фича включена, тогда узкие типы добавляем, иначе только широкие используем
                                inventoryTypeSet.addAll(filterList(values, checkedInventoryTypes::contains));
                            }
                            inventoryTypeSet.add(key);
                        } else {
                            // если широкий не выбран, добавляем узкие без родительского
                            // требуется для дальнейшей инвертации
                            inventoryTypeSet.addAll(values);
                        }
                    }
                    // иначе просто смотрим выбран был или нет широкий тип(у которого нет узких).
                    else if (checkedInventoryTypes.contains(key)) {
                        inventoryTypeSet.add(key);
                    }
                }
        );

        Set<InventoryType> inventoryTypesForZero = filterToSet(
                setUnion(ACTUAL_INVENTORY_WIDE_TO_NARROW.keySet(),
                        FunctionalUtils.flatMap(ACTUAL_INVENTORY_WIDE_TO_NARROW.values(), identity())),
                inventoryType -> !inventoryTypeSet.contains(inventoryType));

        inventoryTypesForZero.forEach(inventoryType -> inventoryAdjustmentsListForSave.add(
                new BidModifierInventoryAdjustment()
                        .withInventoryType(inventoryType)
                        .withPercent(0)));

        return inventoryAdjustmentsListForSave;
    }

    public static BidModifierInventory extractBidModifierInventory(Collection<BidModifier> bidModifiers) {
        if (bidModifiers == null) {
            return null;
        }
        BidModifierInventory bidModifierInventory = (BidModifierInventory) bidModifiers.stream()
                .filter(it -> it.getType() == BidModifierType.INVENTORY_MULTIPLIER)
                .findAny().orElse(null);
        return bidModifierInventory;
    }
}
