package ru.yandex.direct.core.entity.campaign.service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.OptimizingCampaignRequestRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.ArchiveCampaignValidationService;
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperation;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperationName;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.dbschema.ppc.enums.OptimizingCampaignRequestsStatus;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.db.PpcPropertyNames.ARC_CAMPAIGN_SLEEP_COEF;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SENSITIVE_PROPERTIES_UNBOUNDED;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Приостановка и возобновление показов кампаний
 */
public class CampaignsArchiveOperation extends SimpleAbstractUpdateOperation<Campaign, Long> {
    private static final int CHUNK_SIZE = 2000;

    private final ArchiveCampaignValidationService validationService;
    private final CampaignRepository campaignRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final BidRepository bidRepository;
    private final OptimizingCampaignRequestRepository optimizingCampaignRequestRepository;
    private final CampOperationQueueRepository campOperationQueueRepository;
    private final KeywordRepository keywordRepository;
    private final DslContextProvider dslContextProvider;
    private final long operatorUid;
    private final ClientId clientId;
    private final int shard;
    private final PpcProperty<Double> sleepCoefficientProperty;
    private LocalDateTime operationDateTime;

    public CampaignsArchiveOperation(
            List<Long> campaignIds,
            ArchiveCampaignValidationService validationService,
            PpcPropertiesSupport ppcPropertiesSupport,
            CampaignRepository campaignRepository,
            BannerCommonRepository bannerCommonRepository,
            AggregatedStatusesRepository aggregatedStatusesRepository,
            BidRepository bidRepository,
            OptimizingCampaignRequestRepository optimizingCampaignRequestRepository,
            CampOperationQueueRepository campOperationQueueRepository,
            KeywordRepository keywordRepository,
            DslContextProvider dslContextProvider,
            long operatorUid, ClientId clientId, int shard
    ) {
        super(
                Applicability.PARTIAL,
                campaignIds.stream().map(cid ->
                        new ModelChanges<>(cid, Campaign.class)
                                .process(true, Campaign.STATUS_ARCHIVED)
                ).collect(toList()),
                id -> new Campaign().withId(id),
                SENSITIVE_PROPERTIES_UNBOUNDED
        );
        this.validationService = validationService;
        this.campaignRepository = campaignRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.bidRepository = bidRepository;
        this.optimizingCampaignRequestRepository = optimizingCampaignRequestRepository;
        this.campOperationQueueRepository = campOperationQueueRepository;
        this.keywordRepository = keywordRepository;
        this.dslContextProvider = dslContextProvider;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;
        this.sleepCoefficientProperty = ppcPropertiesSupport.get(ARC_CAMPAIGN_SLEEP_COEF, Duration.ofSeconds(60));
    }

    @Override
    protected ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChanges(
            List<ModelChanges<Campaign>> modelChanges
    ) {
        return validationService.validateModelChanges(modelChanges, operatorUid, clientId, shard);
    }

    @Override
    protected Collection<Campaign> getModels(Collection<Long> ids) {
        return campaignRepository.getCampaigns(shard, ids);
    }

    @Override
    protected ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<Campaign>>, Defect> preValidateResult,
            Map<Long, Campaign> models
    ) {
        return validationService.validateModelChangesBeforeApply(shard, preValidateResult, models);
    }

    @Override
    protected void beforeExecution(ExecutionStep<Campaign> executionStep) {
        this.operationDateTime = LocalDateTime.now();

        executionStep.getAppliedChangesForExecution().forEach(it -> {
            if (Boolean.FALSE.equals(it.getOldValue(Campaign.STATUS_ARCHIVED))) {
                it.modify(Campaign.STATUS_SHOW, false);
                it.modify(Campaign.STATUS_BS_SYNCED, StatusBsSynced.NO);
                it.modify(Campaign.LAST_CHANGE, operationDateTime);
            }
        });
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Campaign>> applicableAppliedChanges) {
        List<Long> campaignIds = filterAndMapList(applicableAppliedChanges, AppliedChanges::hasActuallyChangedProps,
                a -> a.getModel().getId());

        List<Long> heavyCampaignIds = bidRepository.getHeavyCampaignIds(shard, campaignIds);
        List<CampQueueOperation> campQueueOperations = mapList(heavyCampaignIds, this::queueCampOperation);
        campOperationQueueRepository.addCampaignQueueOperations(shard, campQueueOperations);

        Set<Long> nonHeavyCampaignIds = campaignIds.stream()
                .filter(id -> !heavyCampaignIds.contains(id)).collect(Collectors.toCollection(HashSet::new));
        bannerCommonRepository.resetStatusActiveAndArchiveStatusShowsForecast(shard, nonHeavyCampaignIds);
        List<Long> bidIds = bidRepository.getBidIdsByCampaignIds(shard, nonHeavyCampaignIds);
        for (List<Long> chunk : Iterables.partition(bidIds, CHUNK_SIZE)) {
            double sleepCoefficient = sleepCoefficientProperty.getOrDefault(1d);
            new RelaxedWorker(sleepCoefficient).runAndRelax(
                    () -> dslContextProvider
                            .ppcTransaction(shard, conf -> keywordRepository.archiveKeywords(conf, chunk)));
        }

        campaignRepository.updateCampaigns(shard, filterList(applicableAppliedChanges,
                a -> nonHeavyCampaignIds.contains(a.getModel().getId())));

        optimizingCampaignRequestRepository.closeRequestsFirstAid(shard, nonHeavyCampaignIds,
                OptimizingCampaignRequestsStatus.AcceptDeclined);

        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    private CampQueueOperation queueCampOperation(Long campaignId) {
        return new CampQueueOperation()
                .withCid(campaignId)
                .withCampQueueOperationName(CampQueueOperationName.ARC)
                .withParams("{\"UID\":\"" + operatorUid + "\"}");
    }

    @Override
    protected void afterExecution(ExecutionStep<Campaign> executionStep) {
        Collection<AppliedChanges<Campaign>> changes = executionStep.getAppliedChangesForExecution();
        List<Long> changedCampaignIds = changes.stream()
                .filter(AppliedChanges::hasActuallyChangedProps)
                .map(ac -> ac.getModel().getId())
                .collect(toList());

        aggregatedStatusesRepository.markCampaignStatusesAsObsolete(dslContextProvider.ppc(shard),
                this.operationDateTime, changedCampaignIds);
    }


}
