package ru.yandex.direct.core.entity.campaign.service;

import java.time.Duration;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.DeleteCampaignValidationService;
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperation;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperationName;
import ru.yandex.direct.core.entity.client.repository.ClientManagersRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция удаления кампаний.
 * Работает как в перле без флага "force".
 * По факту не удаляет, а помечает удаленной (statusEmpty в Yes).
 * <p>
 * Далее удаление происходит скриптом ppcClearEmptyCampaigns.pl
 */
public class CampaignsDeleteOperation extends AbstractOperationWithId {

    private static final String DELETE_CAMPAIGN_LOCK_PREFIX = "DEL_CAMP_";

    private final CampaignRepository campaignRepository;
    private final CampOperationQueueRepository campOperationQueueRepository;
    private final DeleteCampaignValidationService deleteCampaignValidationService;
    private final ClientManagersRepository clientManagersRepository;
    private final RbacService rbacService;
    private final DslContextProvider dslContextProvider;

    private final int shard;
    private final long operatorUid;
    private final ClientId clientId;

    public CampaignsDeleteOperation(Applicability applicability, List<Long> modelIds,
                                    CampaignRepository campaignRepository,
                                    CampOperationQueueRepository campOperationQueueRepository,
                                    DeleteCampaignValidationService deleteCampaignValidationService,
                                    ClientManagersRepository clientManagersRepository, RbacService rbacService,
                                    DslContextProvider dslContextProvider, long operatorUid,
                                    ClientId clientId, int shard) {
        super(applicability, modelIds);
        this.campaignRepository = campaignRepository;
        this.campOperationQueueRepository = campOperationQueueRepository;
        this.deleteCampaignValidationService = deleteCampaignValidationService;
        this.clientManagersRepository = clientManagersRepository;
        this.rbacService = rbacService;
        this.dslContextProvider = dslContextProvider;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> campaignIds) {
        return deleteCampaignValidationService.validate(campaignIds, operatorUid, clientId, shard);
    }

    @Override
    protected void execute(List<Long> campaignIds) {
        Set<Long> subCampaignIds = campaignRepository.getSubCampaignIdsWithMasterIds(shard, campaignIds).keySet();

        Set<Long> allCampaignIds = new HashSet<>(campaignIds);
        allCampaignIds.addAll(subCampaignIds);

        DSLContext context = dslContextProvider.ppc(shard);

        long chiefUid = rbacService.getChiefByClientId(clientId);
        String lockName = DELETE_CAMPAIGN_LOCK_PREFIX + chiefUid;

        try {
            Integer lockResult = context
                    .select(SqlUtils.mysqlGetLock(lockName, Duration.ofSeconds(60))).fetchOne().component1();
            checkState(Objects.equals(lockResult, 1), "failed to get mysql lock with name: " + lockName);

            // отвязываем менеджеров, у которых не останется кампаний после удаления текущих
            unbindManagers(allCampaignIds);

            campaignRepository.setStatusEmptyToYes(shard, allCampaignIds);

            List<CampQueueOperation> campQueueOperations = mapList(allCampaignIds, this::createCampQueueOperation);

            campOperationQueueRepository.addCampaignQueueOperations(shard, campQueueOperations);
        } finally {
            context.select(SqlUtils.mysqlReleaseLock(lockName)).execute();
        }
    }

    private CampQueueOperation createCampQueueOperation(Long campaignId) {
        return new CampQueueOperation()
                .withCid(campaignId)
                .withCampQueueOperationName(CampQueueOperationName.DEL)
                .withParams("{\"UID\":\"" + operatorUid + "\"}");
    }

    /**
     * Если после удаления текущих кампаний у менеджера не останется кампаний для клиента, удаляем связку
     * клиент-менеджер из ppc.client_managers
     *
     * @param campaignIds id удаляемых кампаний
     */
    private void unbindManagers(Collection<Long> campaignIds) {
        Set<Long> campaignIdsToDelete = Set.copyOf(campaignIds);

        // 1. получаем все кампании клиента
        List<CampaignSimple> clientCampaigns = campaignRepository.getCampaignsSimpleByClientId(shard, clientId);

        // 2. берем сервисируемые кампании и строим мапу byManagerUid
        Map<Long, List<CampaignSimple>> servicedCampaignsByManagerUid = StreamEx.of(clientCampaigns)
                .filter(campaign -> campaign.getManagerUserId() != null && campaign.getAgencyUserId() == null)
                .groupingBy(CampaignSimple::getManagerUserId);

        // 3. если для связки менеджер-клиент все кампании удаляются - нужно отвязать менеджера
        Set<Long> managerUidsToUnbind = EntryStream.of(servicedCampaignsByManagerUid)
                .mapValues(campaigns -> mapList(campaigns, CampaignSimple::getId))
                .filterValues(campaignIdsToDelete::containsAll)
                .keys()
                .toSet();

        clientManagersRepository.deleteFromClientManagers(shard, clientId.asLong(), managerUidsToUnbind);
    }
}
