package ru.yandex.direct.core.entity.campaign.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.SuspendResumeCampaignValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SENSITIVE_PROPERTIES_UNBOUNDED;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Приостановка и возобновление показов кампаний
 */
public class CampaignsSuspendResumeOperation extends SimpleAbstractUpdateOperation<Campaign, Long> {
    private final SuspendResumeCampaignValidationService validationService;
    private final CampaignRepository campaignRepository;
    private final ClientId clientId;
    private final long operatorUid;
    private final int shard;
    private final boolean resume;
    private LocalDateTime operationDateTime;

    public CampaignsSuspendResumeOperation(
            List<Long> campaignIds,
            boolean resume,
            SuspendResumeCampaignValidationService validationService,
            CampaignRepository campaignRepository,
            long operatorUid, ClientId clientId, int shard
    ) {
        super(
                Applicability.PARTIAL,
                campaignIds.stream().map(cid ->
                        new ModelChanges<>(cid, Campaign.class)
                                .process(resume, Campaign.STATUS_SHOW)
                ).collect(toList()),
                id -> new Campaign().withId(id),
                SENSITIVE_PROPERTIES_UNBOUNDED
        );
        this.validationService = validationService;
        this.campaignRepository = campaignRepository;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.shard = shard;
        this.resume = resume;
    }

    @Override
    protected ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChanges(
            List<ModelChanges<Campaign>> modelChanges
    ) {
        return validationService.validateModelChanges(modelChanges, operatorUid, clientId, shard);
    }

    @Override
    protected Collection<Campaign> getModels(Collection<Long> ids) {
        return campaignRepository.getCampaigns(shard, ids);
    }

    @Override
    protected ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<Campaign>>, Defect> preValidateResult,
            Map<Long, Campaign> models
    ) {
        return validationService.validateModelChangesBeforeApply(preValidateResult, models, resume);
    }

    @Override
    protected void beforeExecution(ExecutionStep<Campaign> executionStep) {
        this.operationDateTime = LocalDateTime.now();
        executionStep.getAppliedChangesForExecution().forEach(it -> {
            it.modify(Campaign.STATUS_BS_SYNCED, StatusBsSynced.NO);
            it.modify(Campaign.LAST_CHANGE, operationDateTime);
        });
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Campaign>> applicableAppliedChanges) {
        campaignRepository.updateCampaigns(shard, applicableAppliedChanges);
        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<Campaign> executionStep) {
        if (!resume) {
            Collection<AppliedChanges<Campaign>> changes = executionStep.getAppliedChangesForExecution();
            List<Long> changedCampaignIds = changes.stream()
                    .filter(AppliedChanges::hasActuallyChangedProps)
                    .map(ac -> ac.getModel().getId())
                    .collect(toList());
            campaignRepository.setCampOptionsStopTime(shard, changedCampaignIds, operationDateTime);
        }
    }

}

