package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateOptions;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.campaign.container.CampaignWithBidsDomainsAndPhones;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStub;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithNetworkSettings;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainerImpl;
import ru.yandex.direct.core.entity.campaign.service.validation.UnarchiveCampaignValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.type.disabled.DisabledFieldsDataContainer;
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperation;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperationName;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.moderation.client.ModerationClient;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.partition;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.AUTO_CONTEXT_LIMIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SENSITIVE_PROPERTIES_UNBOUNDED;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.THE_ONLY_VALID_CONTEXT_PRICE_COEF;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Разархивирование кампаний
 */
public class CampaignsUnarchiveOperation extends SimpleAbstractUpdateOperation<BaseCampaign, Long> {
    private static final int CHUNK_SIZE = 10;
    private static final Logger logger = LoggerFactory.getLogger(CampaignsUnarchiveOperation.class);

    private final UnarchiveCampaignValidationService validationService;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;
    private final BidRepository bidRepository;
    private final KeywordRepository keywordRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final CampOperationQueueRepository campOperationQueueRepository;
    private final ClientService clientService;
    private final ModerationClient moderationClient;
    private final ModerationRepository moderationRepository;
    private final BannerModerateService bannerModerateService;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final DslContextProvider dslContextProvider;

    private final long operatorUid;
    private final UidAndClientId uidAndclientId;
    private final int shard;
    private final RestrictedCampaignsUpdateOperationContainer container;
    private LocalDateTime operationDateTime;
    private List<Long> needRecalcPriceContextCids;

    public CampaignsUnarchiveOperation(
            List<Long> campaignIds,
            UnarchiveCampaignValidationService validationService,
            CampaignRepository campaignRepository,
            CampaignTypedRepository campaignTypedRepository,
            CampaignModifyRepository campaignModifyRepository,
            BidRepository bidRepository,
            KeywordRepository keywordRepository,
            BannerCommonRepository bannerCommonRepository,
            AggregatedStatusesRepository aggregatedStatusesRepository,
            CampOperationQueueRepository campOperationQueueRepository,
            RbacService rbacService,
            RequestBasedMetrikaClientFactory metrikaClientFactory,
            ClientService clientService,
            ModerationClient moderationClient,
            ModerationRepository moderationRepository,
            BannerModerateService bannerModerateService,
            BannerRelationsRepository bannerRelationsRepository,
            DslContextProvider dslContextProvider,
            long operatorUid,
            UidAndClientId uidAndclientId,
            int shard
    ) {
        super(
                Applicability.PARTIAL,
                mapList(campaignIds, cid -> new ModelChanges<>(cid, BaseCampaign.class)),
                id -> new CampaignStub().withId(id),
                SENSITIVE_PROPERTIES_UNBOUNDED
        );
        this.validationService = validationService;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
        this.bidRepository = bidRepository;
        this.keywordRepository = keywordRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.campOperationQueueRepository = campOperationQueueRepository;
        this.clientService = clientService;
        this.moderationClient = moderationClient;
        this.moderationRepository = moderationRepository;
        this.bannerModerateService = bannerModerateService;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.dslContextProvider = dslContextProvider;
        this.operatorUid = operatorUid;
        this.uidAndclientId = uidAndclientId;
        this.shard = shard;
        this.container = new RestrictedCampaignsUpdateOperationContainerImpl(
                shard,
                operatorUid,
                uidAndclientId.getClientId(),
                uidAndclientId.getUid(),
                rbacService.getChiefByClientId(uidAndclientId.getClientId()),
                metrikaClientFactory.createMetrikaClient(uidAndclientId.getClientId()),
                new CampaignOptions(),
                null,
                emptyMap(),
                new DisabledFieldsDataContainer(emptyList())
        );
    }

    @Override
    protected ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateModelChanges(
            List<ModelChanges<BaseCampaign>> modelChanges
    ) {
        return validationService.validateModelChanges(modelChanges, operatorUid, uidAndclientId.getClientId(), shard);
    }

    @Override
    protected Collection<BaseCampaign> getModels(Collection<Long> campaignIds) {
        return campaignTypedRepository.getSafely(shard, campaignIds,
                List.of(CommonCampaign.class, CampaignWithStrategy.class, CampaignWithNetworkSettings.class));
    }

    @Override
    protected ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> preValidateResult,
            Map<Long, BaseCampaign> models
    ) {
        return validationService.validateModelChangesBeforeApply(shard, uidAndclientId.getClientId(),
                preValidateResult, models);
    }

    @Override
    protected void beforeExecution(ExecutionStep<BaseCampaign> executionStep) {
        this.operationDateTime = LocalDateTime.now();
        this.needRecalcPriceContextCids = new ArrayList<>();

        executionStep.getAppliedChangesForExecution().forEach(it -> {
            AppliedChanges<CommonCampaign> acCommon = it.castModelUp(CommonCampaign.class);
            acCommon.modify(CommonCampaign.STATUS_ARCHIVED, false);
            if (!acCommon.changed(CommonCampaign.STATUS_ARCHIVED)) {
                return;
            }

            acCommon.modify(CommonCampaign.STATUS_BS_SYNCED, CampaignStatusBsSynced.NO);
            acCommon.modify(CommonCampaign.LAST_CHANGE, operationDateTime);

            if (!(acCommon.getModel() instanceof CampaignWithNetworkSettings)) {
                return;
            }
            AppliedChanges<CampaignWithNetworkSettings> acWithNS =
                    acCommon.castModelUp(CampaignWithNetworkSettings.class);
            CampaignWithNetworkSettings camp = acWithNS.getModel();

            // см https://a.yandex-team.ru/arcadia/direct/perl/protected/Common.pm?rev=r9287185#L3072-3077
            boolean needRecalcPriceContext = camp.getStrategy().getAutobudget() == CampaignsAutobudget.NO
                    && !camp.getStrategy().isDifferentPlaces()
                    && !camp.getContextPriceCoef().equals(THE_ONLY_VALID_CONTEXT_PRICE_COEF);
            boolean needUpdateNetworkParams = camp.getStrategy().getAutobudget() == CampaignsAutobudget.YES
                    && (!camp.getContextPriceCoef().equals(THE_ONLY_VALID_CONTEXT_PRICE_COEF)
                    || !camp.getContextLimit().equals(AUTO_CONTEXT_LIMIT));

            if (needRecalcPriceContext) {
                needRecalcPriceContextCids.add(camp.getId());
                var newStrategy = camp.getStrategy().copy();
                if (newStrategy.getPlatform() == CampaignsPlatform.BOTH) {
                    newStrategy.setStrategy(CampOptionsStrategy.DIFFERENT_PLACES);
                    acWithNS.modify(CampaignWithStrategy.STRATEGY, newStrategy);
                } else if (newStrategy.getStrategy() != null) {
                    newStrategy.setStrategy(null);
                    acWithNS.modify(CampaignWithStrategy.STRATEGY, newStrategy);
                }
            }
            if (needRecalcPriceContext || needUpdateNetworkParams) {
                acWithNS.modify(CampaignWithNetworkSettings.CONTEXT_LIMIT, AUTO_CONTEXT_LIMIT);
                acWithNS.modify(CampaignWithNetworkSettings.CONTEXT_PRICE_COEF, THE_ONLY_VALID_CONTEXT_PRICE_COEF);
            }
        });
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BaseCampaign>> applicableAppliedChanges) {
        // отфильтровываем кампании, которые не нужно разархивировать
        List<Long> changedCampaignIds = filterAndMapList(applicableAppliedChanges,
                AppliedChanges::hasActuallyChangedProps, a -> a.getModel().getId());

        // тяжелые кампании кладем в очередь для асинхронной разархивации
        List<Long> heavyCampaignIds = bidRepository.getHeavyCampaignIds(shard, changedCampaignIds);
        List<CampQueueOperation> campQueueOperations = mapList(heavyCampaignIds, this::createCampQueueOperation);
        campOperationQueueRepository.addCampaignQueueOperations(shard, campQueueOperations);

        // разархивация остальных кампаний
        List<AppliedChanges<BaseCampaign>> nonHeavyAppliedChanges = filterList(applicableAppliedChanges,
                a -> !heavyCampaignIds.contains(a.getModel().getId()));
        unarchiveCampaigns(nonHeavyAppliedChanges);

        aggregatedStatusesRepository.markCampaignStatusesAsObsolete(dslContextProvider.ppc(shard),
                this.operationDateTime, changedCampaignIds);

        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    private CampQueueOperation createCampQueueOperation(Long campaignId) {
        return new CampQueueOperation()
                .withCid(campaignId)
                .withCampQueueOperationName(CampQueueOperationName.UNARC)
                .withParams("{\"UID\":\"" + operatorUid + "\"}");
    }

    private void unarchiveCampaigns(List<AppliedChanges<BaseCampaign>> appliedChanges) {
        List<Long> campaignIds = mapList(appliedChanges, a -> a.getModel().getId());
        Client client = checkNotNull(clientService.getClient(uidAndclientId.getClientId()));
        BigDecimal minPrice = client.getWorkCurrency().getCurrency().getMinPrice();

        // Переносим bids_arc в bids
        for (List<Long> chunk : partition(campaignIds, CHUNK_SIZE)) {
            var needRecalcChunk = filterList(chunk, needRecalcPriceContextCids::contains);
            dslContextProvider.ppcTransaction(shard,
                    conf -> keywordRepository.unarchiveKeywords(conf, chunk, needRecalcChunk, minPrice));
        }

        // Для кампаний с автобюджетом и пустым приоритетом в bids заполняем его дефолтом
        List<Long> autobudgetCampaignIds = filterAndMapList(appliedChanges,
                a -> a.getModel() instanceof CampaignWithStrategy &&
                        ((CampaignWithStrategy) a.getModel()).getStrategy().getAutobudget() == CampaignsAutobudget.YES,
                a -> a.getModel().getId());
        bidRepository.setAutobudgetPriorityAutobudget(shard, autobudgetCampaignIds);

        // Если хотя бы к одному домену или телефону разархивируемой кампании есть записи на модерации
        // то отправим кампанию целиком на перемодерацию (контент сайта мог измениться, лицензии истечь и т.п.)
        List<CampaignWithBidsDomainsAndPhones> campaignsWithModerationData = campaignRepository
                .getCampaignsWithBidsDomainsAndPhones(shard, campaignIds);
        List<Long> campaignsToModerate = getCampaignsToModerate(campaignsWithModerationData);
        appliedChanges.stream()
                .filter(ac -> campaignsToModerate.contains(ac.getModel().getId()))
                .forEach(ac -> ac.castModelUp(CommonCampaign.class)
                        .modify(CommonCampaign.STATUS_MODERATE, CampaignStatusModerate.READY));

        List<Long> bannerIds = bannerRelationsRepository.getBannerIdsByCampaignIds(shard, campaignsToModerate);
        bannerModerateService.moderateBanners(uidAndclientId.getClientId(), operatorUid, bannerIds);
        campaignModifyRepository.updateCampaigns(container, appliedChanges);

        // Ess модерация требует чтобы появилось событие - переход объекта в Ready. После разархивации статус
        // объектов может остаться в Sending, и такие объекты не будут отправлены на модерацию
        for (List<CampaignWithBidsDomainsAndPhones> chunk : partition(campaignsWithModerationData, CHUNK_SIZE)) {
            bannerIds = flatMap(chunk, CampaignWithBidsDomainsAndPhones::getBids);
            bannerCommonRepository.fixStatusModerateForUnarchivedObjects(shard, bannerIds);
            bannerModerateService.moderateBanners(uidAndclientId.getClientId(), operatorUid, bannerIds,
                    new BannerModerateOptions(true));
        }

        // Обновляем banners.LastChange, чтобы не заархивировать по старости
        // также переотправляем все ресурсы в БК, т.к. ресурсы для баннеров из архивных кампаний они выкидывают
        bannerCommonRepository.unarchiveStatusBsSyncedAndStatusShowsForecast(shard, campaignIds);

        moderationRepository.insertIgnoreModExportCandidates(shard, campaignIds);
    }

    private List<Long> getCampaignsToModerate(List<CampaignWithBidsDomainsAndPhones> campaignsWithModerationData) {
        return campaignsWithModerationData.stream()
                .filter(it -> {
                    try {
                        // проверим, есть ли хотя бы к одному домену или телефону записи на модерации
                        return moderationClient.hasDomainsOrPhonesNotInWhitelist(it.getDomains(), it.getPhones(),
                                uidAndclientId.getUid(), uidAndclientId.getClientId().asLong());
                    } catch (Exception e) {
                        logger.warn("Error checking domains and phones for records in moderation: {}", e.getMessage());
                        // https://a.yandex-team.ru/arcadia/direct/perl/protected/Common.pm?rev=r9287185#L3133
                        // если по какой-то причине не можем проверить наличие записей на модерации, считаем, что их нет
                        return false;
                    }
                })
                .map(CampaignWithBidsDomainsAndPhones::getCid)
                .collect(toList());
    }
}
