package ru.yandex.direct.core.entity.campaign.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.UpdateCampaignValidationService;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SENSITIVE_PROPERTIES_UNBOUNDED;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.model.AppliedChanges.setter;
import static ru.yandex.direct.model.ModelChanges.isPropertyChanged;
import static ru.yandex.direct.model.ModelChanges.propertyModifier;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Реализация пошагового обновления минус-фраз на кампанию.
 * Обрабатывается только изменение минус-фраз.
 */
public class CampaignsUpdateOperation extends SimpleAbstractUpdateOperation<Campaign, Long> {

    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final UpdateCampaignValidationService updateCampaignValidationService;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final ClientId clientId;
    private final long operatorUid;
    private final int shard;
    private final MinusPhraseValidator.ValidationMode minusPhraseValidationMode;

    private List<Runnable> additionalTasks;

    public CampaignsUpdateOperation(Applicability applicability, List<ModelChanges<Campaign>> modelChanges,
                                    CampaignRepository campaignRepository,
                                    AdGroupRepository adGroupRepository,
                                    BannerCommonRepository bannerCommonRepository,
                                    UpdateCampaignValidationService updateCampaignValidationService,
                                    MinusKeywordPreparingTool minusKeywordPreparingTool,
                                    MinusPhraseValidator.ValidationMode minusPhraseValidationMode,
                                    long operatorUid, ClientId clientId, int shard) {
        super(applicability, modelChanges, id -> new Campaign().withId(id), SENSITIVE_PROPERTIES_UNBOUNDED);
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.updateCampaignValidationService = updateCampaignValidationService;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.shard = shard;
        this.minusPhraseValidationMode = minusPhraseValidationMode;
    }

    @Override
    protected ValidationResult<List<ModelChanges<Campaign>>, Defect> validateModelChanges(
            List<ModelChanges<Campaign>> modelChanges) {
        return updateCampaignValidationService.preValidate(modelChanges, minusPhraseValidationMode, operatorUid,
                clientId, shard
        );
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<Campaign> modelChangesValidatedStep) {
        prepareMinusKeywordsForSaving(modelChangesValidatedStep);
    }

    @Override
    protected Collection<Campaign> getModels(Collection<Long> ids) {
        return campaignRepository.getCampaigns(shard, ids);
    }

    @Override
    protected ValidationResult<List<Campaign>, Defect> validateAppliedChanges(
            ValidationResult<List<Campaign>, Defect> validationResult) {
        return updateCampaignValidationService.validate(validationResult);
    }

    @Override
    protected void beforeExecution(ExecutionStep<Campaign> executionStep) {
        Collection<AppliedChanges<Campaign>> appliedChangesCollection = executionStep.getAppliedChangesForExecution();
        modifyStatusesOnChangedMinusKeywords(appliedChangesCollection);
        additionalTasks = createAdditionalTasks(appliedChangesCollection, shard);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Campaign>> applicableAppliedChanges) {
        campaignRepository.updateCampaigns(shard, applicableAppliedChanges);
        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<Campaign> executionStep) {
        additionalTasks.forEach(Runnable::run);
    }

    /**
     * Подготовка минус-фраз в объектах ModelChanges к сохранению.
     * Проводится после первого этапа валидации для объектов ModelChanges без ошибок.
     * Изменяет валидные ModelChanges.
     */
    private void prepareMinusKeywordsForSaving(ModelChangesValidatedStep<Campaign> modelChangesValidatedStep) {
        StreamEx.of(modelChangesValidatedStep.getValidModelChanges())
                .filter(isPropertyChanged(Campaign.MINUS_KEYWORDS))
                .forEach(propertyModifier(Campaign.MINUS_KEYWORDS, minusKeywordPreparingTool::fullPrepareForSaving));
    }

    private void modifyStatusesOnChangedMinusKeywords(Collection<AppliedChanges<Campaign>> appliedChanges) {
        StreamEx.of(appliedChanges)
                .forEach(setter(Campaign.LAST_CHANGE, LocalDateTime.now()));

        StreamEx.of(appliedChanges)
                .filter(isChanged(Campaign.MINUS_KEYWORDS))
                .peek(setter(Campaign.STATUS_BS_SYNCED, StatusBsSynced.NO))
                .forEach(setter(Campaign.AUTOBUDGET_FORECAST_DATE, null));
    }

    private List<Runnable> createAdditionalTasks(Collection<AppliedChanges<Campaign>> appliedChanges, int shard) {
        List<Runnable> updateTasks = new ArrayList<>();

        List<AppliedChanges<Campaign>> appliedChangesWithChangedMinusKeywords = StreamEx.of(appliedChanges)
                .filter(isChanged(Campaign.MINUS_KEYWORDS))
                .toList();

        Set<Long> campaignIdsWithChangedMinusKeywords = StreamEx.of(appliedChangesWithChangedMinusKeywords)
                .map(a -> a.getModel().getId())
                .toSet();
        if (!campaignIdsWithChangedMinusKeywords.isEmpty()) {
            updateTasks.add(() -> adGroupRepository.updateStatusShowsForecastByCampaignIds(shard,
                    campaignIdsWithChangedMinusKeywords,
                    StatusShowsForecast.NEW));
        }

        Set<Long> campaignsToDropBannersStatusBsSynced = StreamEx.of(appliedChangesWithChangedMinusKeywords)
                .filter(a -> a.getModel().getType().equals(CampaignType.DYNAMIC)
                        || a.getModel().getType().equals(CampaignType.PERFORMANCE))
                .map(a -> a.getModel().getId())
                .toSet();
        if (!campaignsToDropBannersStatusBsSynced.isEmpty()) {
            updateTasks.add(() ->
                    bannerCommonRepository.updateStatusBsSyncedByCampaignIds(shard,
                            campaignsToDropBannersStatusBsSynced,
                            StatusBsSynced.NO));
        }

        return updateTasks;
    }
}

