package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.type.add.AddServicedCampaignService;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.AddServicedCampaignInfo;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.campaign.service.validation.type.disabled.DisabledFieldsDataContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.time.LocalDate.now;
import static ru.yandex.direct.common.util.RepositoryUtils.NOW_PLACEHOLDER;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CAMPAIGN_WARNING_BALANCE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_DAY_BUDGET;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_DAY_BUDGET_SHOW_MODE;

@Service
@ParametersAreNonnullByDefault
public class CommonCampaignService {
    private static final String AGENCY_WALLET_NAME = "Общий счет (агентский)";
    private static final String AGENCY_WALLET_NAME_EN = "Shared account (agent)";
    private static final String CLIENT_WALLET_NAME = "Общий счет";
    private static final String CLIENT_WALLET_NAME_EN = "Shared account";

    private final CampaignModifyRepository campaignModifyRepository;
    private final CampaignTypedRepository campaignTypedRepository;

    private final AddServicedCampaignService addServicedCampaignService;
    private final ProductService productService;
    private final UserService userService;

    @Autowired
    public CommonCampaignService(
            CampaignModifyRepository campaignModifyRepository,
            CampaignTypedRepository campaignTypedRepository,
            AddServicedCampaignService addServicedCampaignService,
            ProductService productService,
            UserService userService) {
        this.campaignModifyRepository = campaignModifyRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.addServicedCampaignService = addServicedCampaignService;
        this.productService = productService;
        this.userService = userService;
    }

    public long addWalletOnCampaignCreating(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                            Client client,
                                            @Nullable UidAndClientId agencyUidAndClientId,
                                            CurrencyCode currencyCode) {
        String name = getWalletName(agencyUidAndClientId);
        AddServicedCampaignInfo servicedInfo =
                addServicedCampaignService.getServicedInfoForClientCampaigns(addCampaignParametersContainer);

        WalletTypedCampaign wallet = createWallet(addCampaignParametersContainer, agencyUidAndClientId, currencyCode,
                client, name, servicedInfo);

        return campaignModifyRepository.addWallet(addCampaignParametersContainer, wallet);
    }

    private static String getWalletName(@Nullable UidAndClientId agencyUidAndClientId) {
        boolean en = LocaleContextHolder.getLocale().getLanguage().equals(new Locale("en").getLanguage());
        if (agencyUidAndClientId != null) {
            return en ? AGENCY_WALLET_NAME_EN : AGENCY_WALLET_NAME;
        }
        return en ? CLIENT_WALLET_NAME_EN : CLIENT_WALLET_NAME;
    }

    public void turnOnWalletStatusAggregated(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                             Long walletId) {
        List<WalletTypedCampaign> wallets = getWalletsByIds(addCampaignParametersContainer.getShard(),
                List.of(walletId));

        ModelChanges<WalletTypedCampaign> mc = new ModelChanges<>(walletId, WalletTypedCampaign.class);
        mc.process(true, WalletTypedCampaign.IS_SUM_AGGREGATED);

        AppliedChanges<WalletTypedCampaign> appliedChanges = mc.applyTo(wallets.get(0));

        RestrictedCampaignsUpdateOperationContainer updateCampaignParametersContainer =
                RestrictedCampaignsUpdateOperationContainer.create(addCampaignParametersContainer.getShard(),
                        addCampaignParametersContainer.getOperatorUid(),
                        addCampaignParametersContainer.getClientId(),
                        addCampaignParametersContainer.getClientUid(),
                        addCampaignParametersContainer.getChiefUid(),
                        new DisabledFieldsDataContainer(Collections.emptyList()));

        campaignModifyRepository.updateCampaigns(updateCampaignParametersContainer, List.of(appliedChanges));
    }

    public List<WalletTypedCampaign> getWalletsByIds(int shard, List<Long> walletIds) {
        var campaigns = campaignTypedRepository.getTypedCampaigns(shard, walletIds);

        return StreamEx.of(campaigns)
                .select(WalletTypedCampaign.class)
                .distinct(WalletTypedCampaign::getId)
                .toList();

    }

    private WalletTypedCampaign createWallet(
            RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
            @Nullable UidAndClientId agencyUidAndClientId,
            CurrencyCode currencyCode,
            Client client,
            String name,
            AddServicedCampaignInfo servicedInfo) {

        User subjectUser = userService.getUser(addCampaignParametersContainer.getClientUid());
        LocalDate now = now();

        WalletTypedCampaign wallet = new WalletTypedCampaign()
                .withUid(addCampaignParametersContainer.getChiefUid())
                .withClientId(addCampaignParametersContainer.getClientId().asLong())
                .withName(name)
                .withType(CampaignType.WALLET)
                .withCurrency(currencyCode)
                .withStatusActive(false)
                .withStatusModerate(CampaignStatusModerate.YES)
                .withStatusPostModerate(CampaignStatusPostmoderate.ACCEPTED)
                .withStatusBsSynced(CampaignStatusBsSynced.NO)
                .withStatusEmpty(false)
                .withStatusShow(true)
                .withStatusArchived(false)
                .withSum(BigDecimal.ZERO)
                .withSumToPay(BigDecimal.ZERO)
                .withSumLast(BigDecimal.ZERO)
                .withSumSpent(BigDecimal.ZERO)
                .withLastChange(NOW_PLACEHOLDER)
                .withStartDate(now)
                .withEnablePausedByDayBudgetEvent(true)
                .withEnableSendAccountNews(true)
                .withSmsTime(CampaignConstants.DEFAULT_SMS_TIME_INTERVAL)
                .withTimeZoneId(0L)
                .withAgencyId(0L)
                .withWalletId(0L)
                .withOrderId(0L)
                .withFio(subjectUser.getFio())
                .withWarningBalance(DEFAULT_CAMPAIGN_WARNING_BALANCE)
                .withSmsFlags(EnumSet.of(SmsFlag.PAUSED_BY_DAY_BUDGET_SMS))
                .withEmail(subjectUser.getEmail())
                .withIsServiceRequested(false)
                .withIsSumAggregated(client.getWorkCurrency() != CurrencyCode.YND_FIXED)
                .withDayBudget(DEFAULT_DAY_BUDGET)
                .withDayBudgetShowMode(DEFAULT_DAY_BUDGET_SHOW_MODE)
                .withDayBudgetDailyChangeCount(0);

        wallet.setProductId(productService.calculateProductId(client, wallet));

        if (agencyUidAndClientId != null) {
            wallet.setAgencyUid(agencyUidAndClientId.getUid());
            wallet.setAgencyId(agencyUidAndClientId.getClientId().asLong());
        } else if (servicedInfo.getIsServiced()) {
            wallet.setManagerUid(servicedInfo.getManagerUid());
        }

        setWalletFields(wallet);

        return wallet;
    }

    private void setWalletFields(WalletTypedCampaign wallet) {
        wallet.setWalletOnOffTime(NOW_PLACEHOLDER);
        wallet.setTotalSum(BigDecimal.ZERO);
        wallet.setTotalBalanceId(0L);
        wallet.setIsAutoPayOn(false);
        wallet.setTotalChipsCost(BigDecimal.ZERO);
        wallet.setIsSumAggregated(true);
    }

}
