package ru.yandex.direct.core.entity.campaign.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.copyentity.CopyConfig
import ru.yandex.direct.core.copyentity.CopyOperationFactory
import ru.yandex.direct.core.copyentity.CopyResult
import ru.yandex.direct.core.copyentity.model.CampaignCopyJobParams
import ru.yandex.direct.core.copyentity.model.CampaignCopyJobResult
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags
import ru.yandex.direct.core.copyentity.postprocessors.BannerStatusesCopier
import ru.yandex.direct.core.copyentity.postprocessors.KeywordStatusesCopier
import ru.yandex.direct.core.copyentity.prefilters.CampaignPrefilter
import ru.yandex.direct.core.entity.campaign.container.CampaignInfoForCopy
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignCopyRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignLimitsValidationService
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.alreadyInCampaignQueue
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.alreadyInCopyQueue
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.cannotCopyBannersWithRejectedCreatives
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.cannotCopyCanvasCreativesBetweenClients
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.cannotCopyCpcVideoBannersBetweenClients
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.cannotCopyDynamicFeedAdgroupsBetweenClients
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.cannotCopyPermalinksWithoutAccess
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.CAMPAIGNS_COPY
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository
import ru.yandex.direct.core.entity.organizations.service.OrganizationService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbqueue.LimitOffset
import ru.yandex.direct.dbqueue.model.DbQueueJob
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notInSet
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.ValidationUtils
import ru.yandex.direct.validation.util.check
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateObject

@Service
class CopyCampaignService(
    private val shardHelper: ShardHelper,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val campaignSubObjectAccessCheckerFactory: CampaignSubObjectAccessCheckerFactory,
    private val dbQueueRepository: DbQueueRepository,
    private val campOperationQueueRepository: CampOperationQueueRepository,
    private val campaignLimitsValidationService: CampaignLimitsValidationService,
    private val copyOperationFactory: CopyOperationFactory,
    private val bannerStatusesCopier: BannerStatusesCopier,
    private val keywordStatusesCopier: KeywordStatusesCopier,
    private val campaignPrefilter: CampaignPrefilter,
    private val campaignCopyRepository: CampaignCopyRepository,
    private val organizationService: OrganizationService,
    private val organizationRepository: OrganizationRepository,
) {

    companion object {
        /**
         * Флаги копирования кампаний, используемые при клиентском копировании (на гриде, а не на отдельной странице)
         */
        private val CLIENT_COPY_FLAGS = CopyCampaignFlags(
            isCopyStopped = true,
            isCopyArchived = true,
            isCopyKeywordStatuses = true,
            isCopyBannerStatuses = true,
            isCopyCampaignStatuses = true,
            isStopCopiedCampaigns = true,
        )
    }

    fun copyCampaigns(
        clientIdFrom: ClientId,
        clientIdTo: ClientId,
        operatorUid: Long,
        campaignIds: List<Long>,
        flags: CopyCampaignFlags,
    ): CopyResult<Long> {
        val copyConfig = CopyConfig(clientIdFrom, clientIdTo, operatorUid, BaseCampaign::class.java, campaignIds, flags)
        val operation = copyOperationFactory.build(copyConfig)

        val copyResult: CopyResult<Long> = operation.copy()
        copyResult.logFailedResultsForMonitoring()

        if (copyConfig.flags.isCopyBannerStatuses) {
            bannerStatusesCopier.copyBannerStatuses(copyConfig, copyResult)
        }

        if (copyConfig.flags.isCopyKeywordStatuses) {
            keywordStatusesCopier.copyKeywordStatuses(copyConfig, copyResult)
        }

        return copyResult
    }

    fun copyCampaignsScheduled(operatorUid: Long, clientId: ClientId, campaignIds: List<Long>): MassResult<Long> {
        val shard: Int = shardHelper.getShardByClientId(clientId)

        val campaigns: Map<Long, CommonCampaign> =
            campaignTypedRepository.getStrictly(shard, campaignIds, CommonCampaign::class.java)
                .associateBy { it.id }

        val vr = validateCampaignCopy(
            shard, shard,
            operatorUid,
            clientId, clientId,
            campaignIds,
            campaigns,
            CLIENT_COPY_FLAGS
        )

        val validCampaignIds = ValidationResult.getValidItems(vr)
        if (validCampaignIds.isEmpty()) {
            return MassResult.brokenMassAction(campaignIds, vr)
        }

        createJobs(
            shard, operatorUid, clientId,
            validCampaignIds.map { campaignId ->
                CampaignCopyJobParams(
                    clientIdFrom = clientId,
                    clientIdTo = clientId,
                    campaignId = campaignId,
                    flags = CLIENT_COPY_FLAGS,
                )
            }
        )

        return MassResult.successfulMassAction(campaignIds, vr)
    }

    fun copyCampaignsInterclientScheduled(
        operatorUid: Long,
        userFrom: User,
        userTo: User,
        campaignIds: List<Long>,
        flags: CopyCampaignFlags,
    ): MassResult<Long> {
        val shardFrom = shardHelper.getShardByClientId(userFrom.clientId)
        val shardTo = shardHelper.getShardByClientId(userTo.clientId)

        val campaigns: Map<Long, CommonCampaign> =
            campaignTypedRepository.getStrictly(shardFrom, campaignIds, CommonCampaign::class.java)
                .associateBy { it.id }

        val vr = validateCampaignCopy(
            shardFrom,
            shardTo,
            operatorUid,
            userFrom.clientId,
            userTo.clientId,
            campaignIds,
            campaigns,
            flags,
        )

        val validCampaignIds = ValidationResult.getValidItems(vr)
        if (validCampaignIds.isEmpty()) {
            return MassResult.brokenMassAction(campaignIds, vr)
        }

        createJobs(
            shardFrom,
            operatorUid,
            userFrom.clientId,
            validCampaignIds.map { campaignId ->
                CampaignCopyJobParams(
                    clientIdFrom = userFrom.clientId,
                    clientIdTo = userTo.clientId,
                    campaignId = campaignId,
                    flags = flags,
                    loginFrom = userFrom.login,
                    loginTo = userTo.login,
                )
            }
        )

        return MassResult.successfulMassAction(campaignIds, vr)
    }

    fun createJobs(
        shard: Int,
        operatorUid: Long,
        clientId: ClientId,
        params: List<CampaignCopyJobParams>,
    ): List<DbQueueJob<CampaignCopyJobParams, CampaignCopyJobResult>> =
        dbQueueRepository.insertJobs(shard, CAMPAIGNS_COPY, clientId, operatorUid, params)

    private fun validateCampaignCopy(
        shardFrom: Int,
        shardTo: Int,
        operatorUid: Long,
        clientIdFrom: ClientId,
        clientIdTo: ClientId,
        campaignIds: List<Long>,
        campaignsById: Map<Long, CommonCampaign>,
        flags: CopyCampaignFlags,
    ): ValidationResult<List<Long>, Defect<*>> {
        val accessValidator: CampaignSubObjectAccessValidator = campaignSubObjectAccessCheckerFactory
            .newCampaignChecker(operatorUid, clientIdFrom, campaignIds)
            .createValidator(CampaignAccessType.READ_WRITE)

        val vr = validateList(campaignIds) {
            checkEachBy(accessValidator)
        }

        val validCampaignIds: List<Long> = ValidationResult.getValidItems(vr)
        val validCampaigns: List<CommonCampaign> = validCampaignIds.map { id -> campaignsById[id]!! }
        if (validCampaignIds.isEmpty()) {
            return vr
        }

        val campaignIdsInCopyQueue: Set<Long> = getCampaignIdsInCopyQueue(shardFrom, clientIdFrom)
        val campaignIdsInCampQueue: Set<Long> =
            campOperationQueueRepository.getCampaignIdsInQueueOperations(shardFrom, validCampaignIds)

        val campaignInfoMap: Map<Long, CampaignInfoForCopy> = campaignCopyRepository
            .getCampaignInfoForCopy(shardTo, validCampaignIds)

        validateList(vr) {
            checkEach(notInSet(campaignIdsInCopyQueue), alreadyInCopyQueue(), When.isValid())
            checkEach(notInSet(campaignIdsInCampQueue), alreadyInCampaignQueue(), When.isValid())

            // Всё что можем - проверяем префильтром кампаний
            checkEachBy(
                transferCampaignIssues(
                    campaignPrefilter.campaignValidator(
                        flags.isCopyArchivedCampaigns,
                        clientIdFrom != clientIdTo
                    ),
                    campaignsById,
                ),
                When.isValid(),
            )

            // Остальное - отдельно
            checkEachBy(
                transferCampaignIssues(
                    campaignInfoValidator(clientIdFrom != clientIdTo),
                    campaignInfoMap,
                ),
                When.isValid(),
            )

            /*
             TODO Для копирования между клиентами:
              - Турецким клиентам недоступно копирование смартов?
                https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DoCmd.pm?rev=r9398066#L10921-10925
            */
        }

        if (clientIdFrom != clientIdTo) {
            val bannerPermalinks: Set<Long> = organizationRepository
                .getPermalinkIdsByCampaignId(shardFrom, validCampaignIds)
                .values.flatten().toSet()
            val campaignPermalinks: Set<Long> = organizationRepository
                .getDefaultPermalinkIdsByCampaignId(shardFrom, validCampaignIds)
                .values.toSet()
            val allPermalinks = bannerPermalinks union campaignPermalinks
            val availablePermalinks: Set<Long> = organizationService
                .getClientOrganizations(allPermalinks, clientIdTo).values
                .map { it.permalinkId }
                .toSet()

            validateList(vr) {
                check(cannotCopyPermalinksWithoutAccess()) {
                    allPermalinks == availablePermalinks
                }
            }
        }

        campaignLimitsValidationService.campaignsCountLimitCheck(shardTo, clientIdTo, validCampaigns, vr)

        return vr
    }

    private fun campaignInfoValidator(isCopyingBetweenClients: Boolean): Validator<CampaignInfoForCopy, Defect<*>> {
        return Validator { campaignInfo ->
            validateObject(campaignInfo) {
                check(cannotCopyDynamicFeedAdgroupsBetweenClients(), When.isTrue(isCopyingBetweenClients)) {
                    !it.hasFeedDynamicAdGroups
                }
                check(cannotCopyBannersWithRejectedCreatives()) {
                    !it.hasRejectedCreatives
                }
                check(cannotCopyCanvasCreativesBetweenClients(), When.isTrue(isCopyingBetweenClients)) {
                    !it.hasCanvasCreatives
                }
                check(cannotCopyCpcVideoBannersBetweenClients(), When.isTrue(isCopyingBetweenClients)) {
                    !it.hasCpcVideoBanners
                }
            }
        }
    }

    private fun getCampaignIdsInCopyQueue(shard: Int, clientId: ClientId): Set<Long> {
        val campaignCopyJobs = dbQueueRepository.getJobsByJobTypeAndClientIds(
            shard,
            CAMPAIGNS_COPY,
            listOf(clientId.asLong()),
            LimitOffset.maxLimited()
        )
        return campaignCopyJobs
            .map { it.args.campaignId }
            .toSet()
    }

    private fun <T> transferCampaignIssues(
        campaignValidator: Validator<T, Defect<*>>,
        campaignsById: Map<Long, T>
    ): Validator<Long, Defect<*>> {
        return Validator { id ->
            val campaignVr = campaignValidator.apply(campaignsById[id])
            val campaignIdVr = ValidationResult<Long, Defect<*>>(id)
            ValidationUtils.transferIssuesFromValidationToTopLevel(campaignVr, campaignIdVr)
            campaignIdVr
        }
    }
}
