package ru.yandex.direct.core.entity.campaign.service;

import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceStatusApprove;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIsNotArchived;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIsNotEmpty;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.cpmPriceCampaignStatusApproveIs;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.greaterThan;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.multipleConditionFilter;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereEqFilter;
import static ru.yandex.direct.multitype.repository.filter.SqlFunctions.SQL_MONDAY;
import static ru.yandex.direct.multitype.repository.filter.SqlFunctions.SQL_SUNDAY;
import static ru.yandex.direct.multitype.repository.filter.SqlFunctions.sqlWeekday;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.Predicates.not;

@Service
public class CpmPriceCampaignService {

    private final ShardHelper shardHelper;
    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final PricePackageRepository pricePackageRepository;
    private final ClientNdsService clientNdsService;
    private final ClientService clientService;
    private final CampaignService campaignService;

    @Autowired
    public CpmPriceCampaignService(ShardHelper shardHelper,
                                   CampaignTypedRepository campaignTypedRepository,
                                   AdGroupRepository adGroupRepository,
                                   PricePackageRepository pricePackageRepository,
                                   ClientNdsService clientNdsService,
                                   ClientService clientService,
                                   CampaignService campaignService
                                   ) {
        this.shardHelper = shardHelper;
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.clientNdsService = clientNdsService;
        this.clientService = clientService;
        this.campaignService = campaignService;
    }

    /**
     * Выбирает прайсовые кампании ожидающие апрува.
     * Это не архивные, не удалённые кампании, у которых statusApprove = New.
     * <p>При этом:
     * <li>либо не черновики</li>
     * <li>либо имеющие флаг "разрешенно раннее бронирование" и созданную дефолтную группу</li>
     */
    public Map<Integer, List<CpmPriceCampaign>> getCpmPriceCampaignsWaitingApproveByShard() {
        return shardHelper.forEachShardSequential(shard -> {

            List<CpmPriceCampaign> candidates = campaignTypedRepository.getSafely(shard, multipleConditionFilter(
                    campaignIsNotEmpty(),
                    campaignIsNotArchived(),
                    cpmPriceCampaignStatusApproveIs(CampaignsCpmPriceStatusApprove.New)
            ), CpmPriceCampaign.class);

            return filterCpmPriceCampaignsWaitingApprove(shard, candidates);
        });
    }

    /**
     * Выбирает прайсовые кампании ожидающие автоматического апрува:
     * - available_ad_group_types="cpm_yndx_frontpage"
     * - начинаются в понедельник и заканчиваются в воскресенье
     * - до старта кампании 1 день или больше
     * - объём больше, чем orderVolumeMin на пакете умножить на продолжительность кампании в неделях
     * - текущий statusApprove = New
     * - пользователь выполнил всё необходимое для того, чтобы кампанию можно было подтвердить (кампания не является
     *   черновиком либо ранняя бронь)
     */
    public Map<Integer, List<CpmPriceCampaign>> getCpmPriceCampaignsWaitingAutoApproveByShard() {
        var campaignsByShard = shardHelper.forEachShardSequential(shard -> {

            List<CpmPriceCampaign> candidates = campaignTypedRepository.getSafely(shard, multipleConditionFilter(
                    campaignIsNotEmpty(),
                    campaignIsNotArchived(),
                    cpmPriceCampaignStatusApproveIs(CampaignsCpmPriceStatusApprove.New),
                    greaterThan(CAMPAIGNS.START_TIME, DSL.currentLocalDate()),
                    whereEqFilter(sqlWeekday(CAMPAIGNS.START_TIME), SQL_MONDAY),
                    whereEqFilter(sqlWeekday(CAMPAIGNS.FINISH_TIME), SQL_SUNDAY)
            ), CpmPriceCampaign.class);

            return filterCpmPriceCampaignsWaitingApprove(shard, candidates);
        });

        var pricePackageIds = StreamEx.of(campaignsByShard.values())
                .flatMap(List::stream)
                .map(CpmPriceCampaign::getPricePackageId)
                .collect(Collectors.toSet());

        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);

        Set<Long> yndxFrontpageCids = campaignsByShard.values().stream()
                .flatMap(Collection::stream)
                .filter(campaign -> pricePackages.get(campaign.getPricePackageId()).getAvailableAdGroupTypes()
                        .equals(Set.of(AdGroupType.CPM_YNDX_FRONTPAGE)))
                .map(CampaignWithPricePackage::getId)
                .collect(Collectors.toSet());

        return EntryStream.of(campaignsByShard)
                .mapValues(campaigns -> campaigns.stream()
                        .filter(campaign -> yndxFrontpageCids.contains(campaign.getId()))
                        .filter(campaign -> campaignOrderVolumeInRange(campaign, pricePackages))
                        .collect(Collectors.toList()))
                .toMap();
    }

    /**
     * Фильтрует кампании и оставляет только те, которые ожидают выставления statusApprove = Yes. Т.е. пользователь
     * выполнил всё необходимое для того, чтобы кампанию можно было подтвердить.
     */
    private List<CpmPriceCampaign> filterCpmPriceCampaignsWaitingApprove(
            Integer shard,
            List<CpmPriceCampaign> candidates) {
        if (candidates.isEmpty()) {
            return emptyList();
        }
        List<Long> draftCampaignIdsToCheck = candidates.stream()
                .filter(c -> c.getStatusModerate() == CampaignStatusModerate.NEW
                        && c.getIsDraftApproveAllowed())
                .map(CpmPriceCampaign::getId)
                .collect(Collectors.toList());

        Set<Long> draftCampaignIdsToInclude = adGroupRepository
                .getDefaultPriceSalesAdGroupIdByCampaignId(shard, draftCampaignIdsToCheck)
                .keySet();

        return candidates.stream()
                .filter(campaign -> (campaign.getStatusModerate() != CampaignStatusModerate.NEW)
                        || draftCampaignIdsToInclude.contains(campaign.getId()))
                .collect(Collectors.toList());
    }

    /**
     * Фильтрует кампании и оставляет только те, для которых не запрещено выставить statusApprove = Yes (выставление
     * statusApprove = Yes ничего не сломает).
     */
    public <T extends CampaignWithPricePackage> List<T> filterCpmPriceCampaignsEligibleForApprove(
            Integer shard,
            Collection<T> candidates) {
        Set<Long> campaignIds = candidates.stream()
                .map(CampaignWithPricePackage::getId)
                .collect(Collectors.toSet());

        Map<Long, List<Long>> adGroupIdsByCampaignIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard,
                campaignIds);
        Set<Long> allAdgroupIds = adGroupIdsByCampaignIds.values().stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
        Set<Long> archivedAdGroupIds= adGroupRepository.getArchivedAdGroupIds(shard, allAdgroupIds);
        Set<Long> campaignsWithNotArchivedGroups = EntryStream.of(adGroupIdsByCampaignIds)
                .filterValues(adGroupIds -> adGroupIds.stream().anyMatch(not(archivedAdGroupIds::contains)))
                .keys()
                .toSet();

        // В Internal Tools выводим на апрув кампании только с statusApprove = New, а здесь разрешаем и
        // statusApprove = No на всякий случай.
        // Вдруг промахнулись и поменяли statusApprove на No вместо Yes, например? Оставляем возможность вручную
        // вбить cid и поменять на statusApprove = Yes.
        Set<PriceFlightStatusApprove> allowedStatusApprove = Set.of(PriceFlightStatusApprove.NEW,
                PriceFlightStatusApprove.NO);

        return candidates.stream()
                .filter(campaign -> campaignsWithNotArchivedGroups.contains(campaign.getId()))
                .filter(campaign -> !campaign.getStatusEmpty())
                .filter(campaign -> !campaign.getStatusArchived())
                .filter(campaign -> allowedStatusApprove.contains(campaign.getFlightStatusApprove()))
                .collect(Collectors.toList());
    }

    private boolean campaignOrderVolumeInRange(CpmPriceCampaign campaign, Map<Long, PricePackage> pricePackages) {
        long weeks = campaign.getStartDate().until(campaign.getEndDate().plusDays(1), ChronoUnit.WEEKS);
        Long orderVolumeMin = pricePackages.get(campaign.getPricePackageId()).getOrderVolumeMin();
        return campaign.getFlightOrderVolume() >= orderVolumeMin * weeks;
    }

    public Map<Long, WalletRestMoney> getWalletRestMoneyMap(Map<Integer, List<CpmPriceCampaign>> campaignsByShard) {
        List<CpmPriceCampaign> campaigns = flatToList(campaignsByShard.values());
        Map<Long, Client> clients = getClients(campaigns);
        Map<Long, ClientNds> clientsNds =
                listToMap(clientNdsService.massGetEffectiveClientNds(clients.values()), ClientNds::getClientId);

        Map<Long, WalletRestMoney> walletsRestMoney =
                getWalletsRestMoneyWithSubtractedNds(campaignsByShard, clientsNds);
        return walletsRestMoney;
    }

    private Map<Long, Client> getClients(List<CpmPriceCampaign> campaigns) {
        Set<ClientId> clientIds = listToSet(campaigns, campaign -> ClientId.fromLong(campaign.getClientId()));
        List<Client> clients = clientService.massGetClient(clientIds);
        return listToMap(clients, Client::getClientId);
    }

    private Map<Long, WalletRestMoney> getWalletsRestMoneyWithSubtractedNds(
            Map<Integer, List<CpmPriceCampaign>> campaignsByShard, Map<Long, ClientNds> clientsNds) {
        return EntryStream.of(campaignsByShard)
                .mapKeyValue((shard, campaignsInShard) ->
                        getWalletsRestMoneyWithSubtractedNds(shard, campaignsInShard, clientsNds))
                .flatMapToEntry(identity())
                .toMap();
    }

    private Map<Long, WalletRestMoney> getWalletsRestMoneyWithSubtractedNds(
            int shard, List<CpmPriceCampaign> campaignsInShard, Map<Long, ClientNds> clientsNds) {
        Map<Long, ClientNds> walletIdToClientNds = new HashMap<>();
        for (CpmPriceCampaign cpmPriceCampaign : campaignsInShard) {
            if (cpmPriceCampaign.getWalletId() > 0) {
                walletIdToClientNds.computeIfAbsent(
                        cpmPriceCampaign.getWalletId(),
                        walletId -> clientsNds.get(cpmPriceCampaign.getClientId()));
            }
        }

        Map<Long, WalletRestMoney> walletsRestMoney =
                campaignService.getWalletsRestMoneyByWalletCampaignIds(shard, walletIdToClientNds.keySet());
        EntryStream.of(walletsRestMoney)
                .forKeyValue((walletId, restMoney) -> {
                    ClientNds clientNds = walletIdToClientNds.get(walletId);
                    if (clientNds == null) {
                        return;
                    }
                    restMoney.setRest(restMoney.getRest().subtractNds(clientNds.getNds()));
                });

        return walletsRestMoney;
    }

    /**
     * Получить дефолтную группу прайсовой кампании.
     * Если по какой-то причине у кампании несколько дефолтных групп, то возвращаем первую попавшуюся.
     * Если дефолтной группы нет - null.
     *
     * @param cid - идентификатор кампании
     * @return pid
     */
    public AdGroup getDefaultPriceSalesAdGroupByCampaignId(Long cid) {
        int shard = shardHelper.getShardByCampaignId(cid);
        Long pid = adGroupRepository
                .getDefaultPriceSalesAdGroupIdByCampaignId(shard, List.of(cid)).get(cid);
        if (pid == null) {
            return null;
        }
        return StreamEx.of(adGroupRepository.getAdGroups(shard, List.of(pid))).findAny().orElse(null);
    }
}
