package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.i18n.Translatable;

import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CLICK;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_ROI;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_WEEK_BUNDLE;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.AUTOBUDGET_STRATEGIES;

@Service
public class MailTextCreatorService {

    private final TranslationService translationService;
    private final StrategyTranslationService strategyTranslationService;

    @Autowired
    public MailTextCreatorService(TranslationService translationService,
                                  StrategyTranslationService strategyTranslationService) {
        this.translationService = translationService;
        this.strategyTranslationService = strategyTranslationService;
    }

    private static final MailTranslations TRANSLATIONS = MailTranslations.INSTANCE;

    public String getTextForEndDateNullValue() {
        return String.format("<%s>", translationService.translate(TRANSLATIONS.valueNotSpecified()));
    }

    public String makeText(DbStrategy oldStrategy, DbStrategy newStrategy, CurrencyCode currency) {
        boolean isDifferentPlaces = newStrategy.isDifferentPlaces();
        Map<CampaignsPlatform, String> prefix = new LinkedHashMap<>();
        if (isDifferentPlaces) {
            prefix.put(CampaignsPlatform.SEARCH, String.format("\t%s: ", translate(TRANSLATIONS.search())));
            prefix.put(CampaignsPlatform.CONTEXT, String.format("\t%s: ", translate(TRANSLATIONS.net())));
        } else {
            prefix.put(CampaignsPlatform.SEARCH, "");
        }

        String strategyName = strategyNameHuman(newStrategy);
        List<String> result = new ArrayList<>();
        if (oldStrategy != null && newStrategy.getPlatform() == oldStrategy.getPlatform()) {
            result.add(translate(TRANSLATIONS.strategyParamChanged(strategyName)));
        } else {
            result.add(translate(TRANSLATIONS.strategyOn(strategyName)));
        }
        for (Map.Entry<CampaignsPlatform, String> prefixEntry : prefix.entrySet()) {
            if (!(isDifferentPlaces || AUTOBUDGET_STRATEGIES.contains(newStrategy.getStrategyName()))) {
                continue;
            }
            if (prefixEntry.getKey() == CampaignsPlatform.SEARCH && newStrategy.isSearchStop()) {
                result.add(prefixEntry.getValue() + translate(TRANSLATIONS.stop()));
                continue;
            }
            if (prefixEntry.getKey() == CampaignsPlatform.CONTEXT && newStrategy.isNetStop()) {
                result.add(prefixEntry.getValue() + translate(TRANSLATIONS.stop()));
                continue;
            }
            if (AUTOBUDGET_STRATEGIES.contains(newStrategy.getStrategyName())) {
                String autoBudgetNotify = autobudgetNotify(prefixEntry.getValue(), newStrategy, currency);
                if (StringUtils.isNotBlank(autoBudgetNotify)) {
                    result.add(autoBudgetNotify);
                }
            } else {
                result.add(prefixEntry.getValue() + oneStrategyNameHuman(newStrategy.getStrategyName()));
            }
        }
        return String.join("\n", result);
    }

    private String strategyNameHuman(DbStrategy strategy) {
        if (strategy.isDifferentPlaces()) {
            return translate(TRANSLATIONS.strategyBoth());
        }
        return oneStrategyNameHuman(strategy.getStrategyName());
    }

    private String autobudgetNotify(String prefix, DbStrategy strategy, CurrencyCode currency) {
        List<String> result = new ArrayList<>();
        if (strategy.getStrategyName() == AUTOBUDGET) {
            if (strategy.getStrategyData().getGoalId() != null) {
                if (strategy.getStrategyData().getGoalId() > 0) {
                    result.add(translate(TRANSLATIONS.maximumGoal(String.valueOf(strategy.getStrategyData().getGoalId()))));
                } else {
                    result.add(translate(TRANSLATIONS.maximumGoals()));
                }
            }
            String sumStr = formatMoney(strategy.getStrategyData().getSum(), currency);
            if (isBidSet(strategy)) {
                String bidSumStr = formatMoney(strategy.getStrategyData().getBid(), currency);
                result.add(translate(TRANSLATIONS.spentMax(sumStr, bidSumStr)));
            } else {
                result.add(translate(TRANSLATIONS.spentWeekly(sumStr)));
            }

        } else if (strategy.getStrategyName() == AUTOBUDGET_WEEK_BUNDLE) {
            if (isBidSet(strategy)) {
                String bidSumStr = formatMoney(strategy.getStrategyData().getBid(), currency);
                result.add(translate(TRANSLATIONS.clicksWeeklyMax(strategy.getStrategyData().getLimitClicks(),
                        bidSumStr)));
            } else if (isAvgBidSet(strategy)) {
                String avgBidSum = formatMoney(strategy.getStrategyData().getAvgBid(), currency);
                result.add(translate(TRANSLATIONS.clicksWeeklyAvg(strategy.getStrategyData().getLimitClicks(),
                        avgBidSum)));
            } else {
                result.add(translate(TRANSLATIONS.clicksWeekly(strategy.getStrategyData().getLimitClicks())));
            }
        } else if (strategy.getStrategyName() == AUTOBUDGET_AVG_CLICK) {
            String avgBidSum = formatMoney(strategy.getStrategyData().getAvgBid(), currency);
            if (isSumSet(strategy)) {
                String sumStr = formatMoney(strategy.getStrategyData().getSum(), currency);
                result.add(translate(TRANSLATIONS.holdPriceInBudget(avgBidSum, sumStr)));
            } else {
                result.add(translate(TRANSLATIONS.holdPrice(avgBidSum)));
            }

        } else if (strategy.getStrategyName() == AUTOBUDGET_AVG_CPA) {
            // TODO // same as in perl
        } else if (strategy.getStrategyName() == AUTOBUDGET_ROI) {
            String description = translate(TRANSLATIONS.holdProfitability(strategy.getStrategyData().getRoiCoef()));
            if (strategy.getStrategyData().getGoalId() > 0) {
                description += " " + translate(TRANSLATIONS.goal(strategy.getStrategyData().getGoalId()));
            } else {
                description += " " + translate(TRANSLATIONS.goals());
            }
            if (strategy.getStrategyData().getReserveReturn() != null) {
                description += ", " + translate(TRANSLATIONS.reverseReturn(strategy.getStrategyData().getReserveReturn()));
            }
            if (isSumSet(strategy)) {
                description += ", " + translate(TRANSLATIONS.weekBudget(formatMoney(strategy.getStrategyData().getSum(), currency)));
            }
            if (isBidSet(strategy)) {
                description += ", " + translate(TRANSLATIONS.maxBid(formatMoney(strategy.getStrategyData().getBid(),
                        currency)));
            }
            if (isProfitabilitySet(strategy)) {
                description += ", " + translate(TRANSLATIONS.percent(strategy.getStrategyData().getProfitability()));
            }
            result.add(description);
        }
        final String delimiter;
        StringBuilder answer = new StringBuilder();
        if (StringUtils.isNotEmpty(prefix)) {
            answer.append(prefix).append(oneStrategyNameHuman(strategy.getStrategyName()));
            delimiter = "\t\t";
        } else {
            delimiter = "\t";
        }

        boolean notFirstLoop = false;
        for (String s : result) {
            if (notFirstLoop || StringUtils.isNotEmpty(prefix)) {
                answer.append("\n");
            }
            answer.append(delimiter).append(s);
            notFirstLoop = true;
        }
        return answer.toString();
    }

    private static boolean isBidSet(DbStrategy strategy) {
        return strategy.getStrategyData().getBid() != null && strategy.getStrategyData().getBid().compareTo(BigDecimal.ZERO) > 0;
    }

    private static boolean isAvgBidSet(DbStrategy strategy) {
        return strategy.getStrategyData().getAvgBid() != null && strategy.getStrategyData().getAvgBid().compareTo(BigDecimal.ZERO) > 0;
    }

    private static boolean isSumSet(DbStrategy strategy) {
        return strategy.getStrategyData().getSum() != null && strategy.getStrategyData().getSum().compareTo(BigDecimal.ZERO) > 0;
    }

    private static boolean isProfitabilitySet(DbStrategy strategy) {
        return strategy.getStrategyData().getProfitability() != null && strategy.getStrategyData().getProfitability().compareTo(BigDecimal.ZERO) > 0;
    }

    public String formatMoney(BigDecimal sum, CurrencyCode currencyCode) {
        Money money = Money.valueOf(sum, currencyCode);
        return getMoneyWithoutCurrency(money) + " " + translate(currencyCode.getTranslation().shortForm());
    }

    public String getMoneyWithoutCurrency(Money money) {
        DecimalFormatSymbols symbols = DecimalFormatSymbols.getInstance();
        symbols.setGroupingSeparator(' ');
        symbols.setDecimalSeparator('.');
        DecimalFormat formatter =  new DecimalFormat("###,##0.00", symbols);
        return formatter.format(money.roundToCentDown().bigDecimalValue());
    }

    private String oneStrategyNameHuman(StrategyName platform) {
        return strategyTranslationService.getTranslation(platform);
    }

    private String translate(Translatable t) {
        return translationService.translate(t);
    }
}
