package ru.yandex.direct.core.entity.campaign.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.request.GetExistentCountersRequest;
import ru.yandex.direct.metrika.client.model.request.GoalType;
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;
import ru.yandex.direct.metrika.client.model.response.UserCounters;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtendedResponse;

import static java.util.Collections.emptyList;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService.getConditions;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Адаптер для использования {@link MetrikaClient} в рамках одного запроса, для заданных представителей клиента.
 * Если данные ранее не были получены для клиента, то выполнится реальный вызов метрики и данные запишутся в кэш.
 */
public class RequestBasedMetrikaClientAdapter {

    private final MetrikaClient metrikaClient;
    private final List<Long> clientRepresentativesUids;
    private final Set<String> enabledFeatures;
    private List<Long> campaignsCounterIds;
    /**
     * Флаг говорит о том, что есть кампании, для которых могут быть использованы цели без доступа.
     * Нужен для того, чтобы получить из Метрики данные и по недоступным счетчикам и целям.
     * Проверка на использование целей без доступа для конкретной кампании делается в
     * для добавления кампаний:
     * {@link ru.yandex.direct.core.entity.metrika.service.campaigngoals.CommonCampaignGoalsService#getAvailableGoalsForCampaignType}
     * для обновления кампаний:
     * {@link ru.yandex.direct.core.entity.metrika.service.campaigngoals.CommonCampaignGoalsService#getAvailableGoalsByCampaignId}
     */
    private boolean shouldFetchUnavailableGoals;

    private final Map<CacheKey, Object> cache = new HashMap<>();

    @VisibleForTesting
    public RequestBasedMetrikaClientAdapter(MetrikaClient metrikaClient,
                                            Collection<Long> clientRepresentativesUids,
                                            Set<String> enabledFeatures) {
        this(metrikaClient, clientRepresentativesUids, enabledFeatures, null, false);
    }

    public RequestBasedMetrikaClientAdapter(MetrikaClient metrikaClient,
                                            Collection<Long> clientRepresentativesUids,
                                            Set<String> enabledFeatures,
                                            @Nullable Collection<? extends BaseCampaign> campaigns,
                                            boolean shouldFetchUnavailableGoals) {
        this.metrikaClient = metrikaClient;
        this.clientRepresentativesUids = new ArrayList<>(clientRepresentativesUids);
        this.enabledFeatures = enabledFeatures;
        this.shouldFetchUnavailableGoals = shouldFetchUnavailableGoals;
        if (campaigns != null) {
            this.campaignsCounterIds = CampaignConverter.extractCounterIdsFromCampaigns(campaigns);
        }
    }

    private RequestBasedMetrikaClientAdapter(MetrikaClient metrikaClient,
                                             Collection<Long> clientRepresentativesUids,
                                             List<Long> counterIds,
                                             Set<String> enabledFeatures,
                                             boolean shouldFetchUnavailableGoals) {
        this.metrikaClient = metrikaClient;
        this.clientRepresentativesUids = new ArrayList<>(clientRepresentativesUids);
        this.enabledFeatures = enabledFeatures;
        this.shouldFetchUnavailableGoals = shouldFetchUnavailableGoals;
        this.campaignsCounterIds = counterIds;
    }

    //todo: rename campaign to more abstract entity
    public static RequestBasedMetrikaClientAdapter getRequestBasedMetrikaClientAdapterForStrategies(
            MetrikaClient metrikaClient,
            Collection<Long> clientRepresentativesUids,
            Set<FeatureName> enabledFeatures,
            Collection<? extends BaseStrategy> strategies,
            boolean shouldFetchUnavailableGoals) {

        var counterIds = extractCounterIdsFromStrategies(strategies);
        return new RequestBasedMetrikaClientAdapter(
                metrikaClient,
                clientRepresentativesUids,
                counterIds,
                listToSet(enabledFeatures, FeatureName::getName),
                shouldFetchUnavailableGoals);
    }

    public static List<Long> extractCounterIdsFromStrategies(Collection<? extends BaseStrategy> strategies) {
        return StreamEx.of(strategies)
                .flatCollection(StrategyModelUtils::metrikaCounterIds)
                .distinct()
                .toList();
    }

    /**
     * Получить список всех доступных аб-сегментов для клиента.
     */
    @SuppressWarnings("unchecked")
    public Map<Long, List<RetargetingCondition>> getAbSegmentGoals() {
        return (Map<Long, List<RetargetingCondition>>) cache.computeIfAbsent(
                CacheKey.AB_SEGMENT_GOALS_BY_UIDS_KEY,
                (key) -> metrikaClient.getGoals(clientRepresentativesUids, GoalType.AB_SEGMENT)
                        .getUidToConditions()
        );
    }

    /**
     * Получить доступные цели для клиента по списку goalIds.
     *
     * @param goalIds фильтр по goalId
     * @return мапа uid представителя на goals
     */
    public Map<Long, List<RetargetingCondition>> getGoalsByUid(Collection<Long> goalIds) {
        GoalsContainer container = (GoalsContainer) cache.computeIfAbsent(CacheKey.GOALS_BY_UIDS_KEY,
                t -> new GoalsContainer());
        var unCheckedGoalIds = filterList(goalIds, goalId -> !container.checkedGoalIds.contains(goalId));
        if (!unCheckedGoalIds.isEmpty()) {
            var goalTypeToConditions = StreamEx.of(unCheckedGoalIds)
                    .mapToEntry(Goal::computeType, Function.identity())
                    .sortedBy(Map.Entry::getKey)
                    .collapseKeys()
                    .mapToValue((type, ids) -> getConditions(metrikaClient, clientRepresentativesUids, type, ids))
                    .toMap();

            goalTypeToConditions.forEach((goalType, uidToConditions) -> {
                var containerConditions = container.goalTypeToCondition.get(goalType);
                var result = mergeConditions(containerConditions, uidToConditions);
                container.goalTypeToCondition.put(goalType, result);
            });
            container.checkedGoalIds.addAll(unCheckedGoalIds);
        }

        Set<Long> goalIdsSet = new HashSet<>(goalIds);
        Map<Long, List<RetargetingCondition>> result = new HashMap<>();
        for (var uidToConditions : container.goalTypeToCondition.values()) {
            var filteredResult = EntryStream.of(uidToConditions)
                    .mapValues(conditions -> {
                        var filteredList = filterList(conditions, c -> goalIdsSet.contains(c.getId()));
                        if (filteredList.isEmpty()) {
                            return null;
                        }
                        return filteredList;
                    })
                    .nonNullValues()
                    .toMap();
            result = mergeConditions(result, filteredResult);
        }

        return result;
    }

    private Map<Long, List<RetargetingCondition>> mergeConditions(@Nullable Map<Long, List<RetargetingCondition>> first,
                                                                  Map<Long, List<RetargetingCondition>> second) {
        if (first == null) {
            return second;
        }
        Map<Long, List<RetargetingCondition>> result = new HashMap<>();
        first.forEach((uid, conditions) -> {
            List<RetargetingCondition> resultConditions = result.computeIfAbsent(uid, t -> new ArrayList<>());
            resultConditions.addAll(conditions);
        });
        second.forEach((uid, conditions) -> {
            List<RetargetingCondition> resultConditions = result.computeIfAbsent(uid, t -> new ArrayList<>());
            resultConditions.addAll(conditions);
        });
        return result;
    }

    /**
     * Получить список счетчиков для клиента.
     */
    @SuppressWarnings("unchecked")
    public List<UserCounters> getUsersCountersNumByCampaignCounterIds() {
        if (campaignsCounterIds == null || campaignsCounterIds.isEmpty()) {
            return emptyList();
        }
        return (List<UserCounters>) cache.computeIfAbsent(
                CacheKey.USER_COUNTERS_NUM_KEY, (key) -> metrikaClient
                        .getUsersCountersNum2(clientRepresentativesUids, campaignsCounterIds)
                        .getUsers()
        );
    }

    /**
     * Получить список счетчиков на кампании для клиента с доп информацией по ним, такой как: counterSource,
     * ecommerce etc.
     */
    @SuppressWarnings("unchecked")
    public List<UserCountersExtended> getUsersCountersNumExtendedByCampaignCounterIds() {
        if (campaignsCounterIds == null || campaignsCounterIds.isEmpty()) {
            return emptyList();
        }
        var cacheData = (UserCountersExtendedResponse) cache.computeIfAbsent(
                CacheKey.USERS_COUNTER_NUM_EXTENDED_KEY,
                (key) -> metrikaClient.getUsersCountersNumExtended2(clientRepresentativesUids,
                        new UserCountersExtendedFilter().withCounterIds(new ArrayList<>(campaignsCounterIds))));
        return cacheData.getUsers();
    }

    /**
     * Получить список счетчиков доступных для клиента по заданным ид.
     * Внимание! Сперва необходимо установить <code>counterIds</code>.
     */
    @SuppressWarnings("unchecked")
    public List<GetExistentCountersResponseItem> getCampaignCountersIfNeeded() {
        return (List<GetExistentCountersResponseItem>) cache.computeIfAbsent(
                CacheKey.EXISTENT_COUNTERS, (key) -> this.fetchCampaignCountersIfNeeded()
        );
    }

    /**
     * Получить информацию по всем целям для счетчиков клиента.
     */
    @SuppressWarnings("unchecked")
    public Map<Integer, List<CounterGoal>> getCountersGoals() {
        var goals = cache.get(CacheKey.MASS_COUNTERS_GOALS);
        if (goals == null) {
            var counterIds = extractUserCounterIds();
            goals = fetchMassCountersGoals(counterIds);
            cache.put(CacheKey.MASS_COUNTERS_GOALS, goals);
        }
        return (Map<Integer, List<CounterGoal>>) goals;
    }

    public void setCampaignsCounterIds(Collection<? extends BaseCampaign> campaigns) {
        setCampaignsCounterIds(campaigns, emptyList());
    }

    /**
     * Установить новое значение <code>campaignsCounterIds</code> поместив туда счётчики из <code>campaigns</code>
     * объединив их с дополнительными счётчиками <code>extraCounterIds</code>
     * И сбросить кэш {@link #getCampaignCountersIfNeeded()}, если ранее значение установлено не было.
     * Метод необходим для операций апдейта, где изначально счетчики недоступны и для операции добавления для случая
     * когда counterId не передан, а загружается из метрики.
     */
    public void setCampaignsCounterIds(Collection<? extends BaseCampaign> campaigns,
                                       Collection<Long> extraCounterIds) {
        if (campaigns == null) {
            return;
        }

        this.campaignsCounterIds = StreamEx.of(extraCounterIds)
                .append(CampaignConverter.extractCounterIdsFromCampaigns(campaigns))
                .append(nvl(campaignsCounterIds, List.of()))
                .distinct()
                .toList();
        cache.remove(CacheKey.EXISTENT_COUNTERS);
        cache.remove(CacheKey.USER_COUNTERS_NUM_KEY);
        cache.remove(CacheKey.USERS_COUNTER_NUM_EXTENDED_KEY);
        cache.remove(CacheKey.MASS_COUNTERS_GOALS);
    }

    public boolean isShouldFetchUnavailableGoals() {
        return shouldFetchUnavailableGoals;
    }

    public void setShouldFetchUnavailableGoals(boolean shouldFetchUnavailableGoals) {
        this.shouldFetchUnavailableGoals = shouldFetchUnavailableGoals;
    }

    private Set<Integer> extractUserCounterIds() {
        var userCounterExtendedIds = flatMapToSet(getUsersCountersNumExtendedByCampaignCounterIds(),
                this::extractUserCounterIds);

        return StreamEx.of(getCampaignCountersIfNeeded())
                .map(GetExistentCountersResponseItem::getCounterId)
                .map(Long::intValue)
                .append(userCounterExtendedIds)
                .toSet();
    }

    private List<Integer> extractUserCounterIds(UserCountersExtended userCounters) {
        return mapList(userCounters.getCounters(), CounterInfoDirect::getId);
    }

    private List<GetExistentCountersResponseItem> fetchCampaignCountersIfNeeded() {
        var isGoalsFromAllOrgsAllowed = enabledFeatures.contains(FeatureName.GOALS_FROM_ALL_ORGS_ALLOWED.getName());
        boolean shouldNotFetchCampaignCounters = !shouldFetchUnavailableGoals && !isGoalsFromAllOrgsAllowed;
        if (shouldNotFetchCampaignCounters || isEmpty(campaignsCounterIds)) {
            return emptyList();
        }

        var request = new GetExistentCountersRequest().withCounterIds(campaignsCounterIds);
        return metrikaClient.getExistentCounters(request).getResponseItems();
    }

    private Map<Integer, List<CounterGoal>> fetchMassCountersGoals(Set<Integer> counterIds) {
        if (isEmpty(counterIds)) {
            return Collections.emptyMap();
        }

        return metrikaClient.getMassCountersGoalsFromMetrika(counterIds);
    }

    private static class GoalsContainer {
        Map<ru.yandex.direct.core.entity.retargeting.model.GoalType, Map<Long, List<RetargetingCondition>>> goalTypeToCondition
                = new HashMap<>();
        Set<Long> checkedGoalIds = new HashSet<>();
    }

    private enum CacheKey {
        AB_SEGMENT_GOALS_BY_UIDS_KEY,
        GOALS_BY_UIDS_KEY,
        USER_COUNTERS_NUM_KEY,
        USERS_COUNTER_NUM_EXTENDED_KEY,
        EXISTENT_COUNTERS,
        MASS_COUNTERS_GOALS,
    }
}
