package ru.yandex.direct.core.entity.campaign.service;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.shouldFetchUnavailableGoals;

@Component
@ParametersAreNonnullByDefault
public class RequestBasedMetrikaClientFactory {

    private final MetrikaClient metrikaClient;
    private final RbacService rbacService;
    private final FeatureService featureService;

    @Autowired
    public RequestBasedMetrikaClientFactory(MetrikaClient metrikaClient,
                                            RbacService rbacService,
                                            FeatureService featureService) {
        this.metrikaClient = metrikaClient;
        this.rbacService = rbacService;
        this.featureService = featureService;
    }

    public RequestBasedMetrikaClientAdapter createMetrikaClient(ClientId clientId) {
        return createMetrikaClient(clientId, null);
    }

    public RequestBasedMetrikaClientAdapter createMetrikaClient(ClientId clientId,
                                                                @Nullable List<? extends BaseCampaign> campaigns) {
        var clientRepresentativesUids =
                rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        var enabledFeatures = featureService.getEnabledForClientId(clientId);
        var shouldFetchUnavailableGoals = campaigns != null &&
                campaigns.stream().anyMatch(campaign -> shouldFetchUnavailableGoals(campaign, enabledFeatures));
        return new RequestBasedMetrikaClientAdapter(metrikaClient, clientRepresentativesUids, enabledFeatures,
                campaigns, shouldFetchUnavailableGoals);
    }

    /**
     * То же самое, что createMetrikaClient, но не проставляет campaignCounterIds, чтобы проставить их позже.
     */
    public RequestBasedMetrikaClientAdapter createMetrikaClientWithoutCampaignCounterIds(
            ClientId clientId,
            @Nullable List<? extends BaseCampaign> campaigns) {
        var clientRepresentativesUids =
                rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        var enabledFeatures = featureService.getEnabledForClientId(clientId);
        var shouldFetchUnavailableGoals = campaigns != null &&
                campaigns.stream().anyMatch(campaign -> shouldFetchUnavailableGoals(campaign, enabledFeatures));
        return new RequestBasedMetrikaClientAdapter(metrikaClient, clientRepresentativesUids, enabledFeatures,
                null, shouldFetchUnavailableGoals);
    }

}
