package ru.yandex.direct.core.entity.campaign.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.service.type.add.CampaignAddOperationSupportFacade;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainerImpl;
import ru.yandex.direct.core.entity.campaign.service.validation.AddRestrictedCampaignValidationService;
import ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.getGoalIdToCounterIdForCampaignsWithoutCounterIds;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignValidationUtils.eraseSpecificWarningsIfErrors;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.buildSubValidationResult;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.util.ValidationUtils.cloneValidationResultSubNodesWithIssues;

@ParametersAreNonnullByDefault
/*
  Операция добавления кампаний.
  Работает "частично".
 */
public class RestrictedCampaignsAddOperation {

    private final List<? extends BaseCampaign> campaignsToAdd;

    private final CampaignModifyRepository campaignModifyRepository;

    private final StrategyTypedRepository strategyTypedRepository;

    private final AddRestrictedCampaignValidationService addRestrictedCampaignValidationService;

    private final CampaignAddOperationSupportFacade campaignAddOperationSupportFacade;

    private final DslContextProvider ppcDslContextProvider;

    private final GoalUtilsService goalUtilsService;

    private ValidationResult<? extends List<? extends BaseCampaign>, Defect> validationResult;
    private List<? extends BaseCampaign> validItems;

    private final RestrictedCampaignsAddOperationContainerImpl container;
    private boolean prepared;

    public RestrictedCampaignsAddOperation(
            List<? extends BaseCampaign> campaignsToAdd,
            int shard,
            UidAndClientId uidAndClientId,
            Long operatorUid,
            CampaignModifyRepository campaignModifyRepository,
            StrategyTypedRepository strategyTypedRepository,
            AddRestrictedCampaignValidationService addRestrictedCampaignValidationService,
            CampaignAddOperationSupportFacade campaignAddOperationSupportFacade,
            DslContextProvider ppcDslContextProvider,
            RbacService rbacService,
            CampaignOptions options,
            RequestBasedMetrikaClientFactory metrikaClientFactory,
            GoalUtilsService goalUtilsService) {
        this.campaignsToAdd = campaignsToAdd;
        this.campaignModifyRepository = campaignModifyRepository;
        this.strategyTypedRepository = strategyTypedRepository;
        this.addRestrictedCampaignValidationService = addRestrictedCampaignValidationService;
        this.campaignAddOperationSupportFacade = campaignAddOperationSupportFacade;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.prepared = false;
        this.goalUtilsService = goalUtilsService;

        var metrikaClientAdapter =
                metrikaClientFactory.createMetrikaClientWithoutCampaignCounterIds(uidAndClientId.getClientId(),
                        campaignsToAdd);

        container = new RestrictedCampaignsAddOperationContainerImpl(
                shard,
                operatorUid,
                uidAndClientId.getClientId(),
                uidAndClientId.getUid(),
                rbacService.getChiefByClientId(uidAndClientId.getClientId()),
                campaignsToAdd,
                options,
                metrikaClientAdapter,
                getClientPackageStrategyFromOperationCampaignsById(shard, uidAndClientId));
    }

    public Optional<MassResult<Long>> prepare() {
        prepared = true;

        var preValidationResult = preValidate(campaignsToAdd);
        List<BaseCampaign> preValidItems = ValidationResult.getValidItems(preValidationResult);

        onPreValidated(preValidItems);

        validationResult = eraseSpecificWarningsIfErrors(validate(preValidationResult));
        validItems = ValidationResult.getValidItems(validationResult);

        if (validationResult.hasErrors() || validItems.isEmpty()) {
            return Optional.of(MassResult.brokenMassAction(mapList(validationResult.getValue(), v -> null),
                    validationResult));
        }

        onModelsValidated(validItems);

        return Optional.empty();
    }

    private ValidationResult<? extends List<? extends BaseCampaign>, Defect> preValidate(
            List<? extends BaseCampaign> itemsToAdd) {
        return addRestrictedCampaignValidationService.preValidate(container, container.getClientUid(), itemsToAdd);
    }

    private ValidationResult<? extends List<? extends BaseCampaign>, Defect> validate(
            ValidationResult<? extends List<? extends BaseCampaign>, Defect> preValidateResult) {

        var preValidateResultWithoutValidSubNodes = cloneValidationResultSubNodesWithIssues(preValidateResult);

        var validItemsWithIndex = ValidationResult.getValidItemsWithIndex(preValidateResultWithoutValidSubNodes);
        // везде порядок сохранится, т.к. внутри LinkedHashMap
        List<BaseCampaign> validItems = List.copyOf(validItemsWithIndex.values());
        List<Integer> validItemsIndexes = EntryStream.of(validItemsWithIndex)
                .keys()
                .toList();

        ValidationResult<List<BaseCampaign>, Defect> vrForValidItems = buildSubValidationResult(
                preValidateResultWithoutValidSubNodes, validItemsIndexes, validItems);

        addRestrictedCampaignValidationService.validate(container, vrForValidItems);

        return preValidateResultWithoutValidSubNodes;
    }

    public MassResult<Long> prepareAndApply() {
        return prepare().orElseGet(this::apply);
    }

    public MassResult<Long> apply() {
        checkState(prepared, "prepare() must be called before apply()");
        beforeExecution(validItems);
        execute(validItems);
        afterExecution(validItems);

        return MassResult.successfulMassAction(getResult(validationResult), validationResult);
    }

    private void onPreValidated(List<? extends BaseCampaign> preValidItems) {
        Map<Long, Long> goalIdToCounterIdForCampaignsWithoutCounterIds =
                container.getOptions().isFetchMetrikaCountersForGoals()
                        ? getGoalIdToCounterIdForCampaignsWithoutCounterIds(preValidItems, container.getMetrikaClient())
                        : emptyMap();
        // в metrikaClient устанавливаем вообще все счётчики (с валидных кампаний и добавляем не указанные счётчики,
        // т.е. загруженные из метрики)
        container.getMetrikaClient().setCampaignsCounterIds(preValidItems,
                goalIdToCounterIdForCampaignsWithoutCounterIds.values());
        // в контейнер сохраняем только не указанные счётчики
        container.setGoalIdToCounterIdForCampaignsWithoutCounterIds(goalIdToCounterIdForCampaignsWithoutCounterIds);

        campaignAddOperationSupportFacade.onPreValidated(container, preValidItems);
    }

    private void onModelsValidated(List<? extends BaseCampaign> validItems) {
        campaignAddOperationSupportFacade.onModelsValidated(container, validItems);
    }

    private void beforeExecution(List<? extends BaseCampaign> validItems) {
        campaignAddOperationSupportFacade.beforeExecution(container, validItems);
    }

    private void execute(List<? extends BaseCampaign> validItems) {
        ppcDslContextProvider.ppcTransaction(container.getShard(), executeInTransaction(validItems));
    }

    private TransactionalRunnable executeInTransaction(List<? extends BaseCampaign> validItems) {
        return configuration -> {
            DSLContext dsl = configuration.dsl();
            campaignModifyRepository.addCampaigns(dsl, container, validItems);
            campaignAddOperationSupportFacade
                    .updateRelatedEntitiesInTransaction(dsl, container, validItems);
        };
    }

    private void afterExecution(List<? extends BaseCampaign> validItems) {
        campaignAddOperationSupportFacade.afterExecution(container, validItems);
    }

    private List<Long> getResult(ValidationResult<? extends List<? extends BaseCampaign>, Defect> vr) {
        Map<Integer, BaseCampaign> validItemsWithIndex = ValidationResult.getValidItemsWithIndex(vr);

        return IntStreamEx.ofIndices(vr.getValue())
                .boxed()
                .map(validItemsWithIndex::get)
                .map(c -> ifNotNull(c, BaseCampaign::getId))
                .toList();
    }

    private Map<Long, BaseStrategy> getClientPackageStrategyFromOperationCampaignsById(int shard,
                                                                                       UidAndClientId uidAndClientId) {
        var filledStrategyIds = campaignsToAdd.stream()
                .filter(c -> c instanceof CampaignWithPackageStrategy)
                .map(c -> ((CampaignWithPackageStrategy) c).getStrategyId())
                .filter(CommonUtils::isValidId)
                .collect(Collectors.toList());

        return strategyTypedRepository.getIdToModelTyped(shard, uidAndClientId.getClientId(), filledStrategyIds);
    }
}
