package ru.yandex.direct.core.entity.campaign.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.amazonaws.util.CollectionUtils;
import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignStub;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.type.update.CampaignUpdateOperationSupportFacade;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainerImpl;
import ru.yandex.direct.core.entity.campaign.service.validation.UpdateRestrictedCampaignValidationService;
import ru.yandex.direct.core.entity.campaign.service.validation.type.disabled.DisabledFieldsDataContainer;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.getStrategyGoalId;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.shouldFetchUnavailableGoals;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SPECIAL_GOAL_IDS;
import static ru.yandex.direct.core.entity.campaign.service.validation.type.bean.utils.CampaignValidationUtils.eraseSpecificWarningsIfErrors;
import static ru.yandex.direct.model.ModelUtilsKt.getPropertyValue;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.CollectionUtils.unionToSet;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.util.ValidationUtils.cloneValidationResultSubNodesWithIssues;
import static ru.yandex.direct.validation.util.ValidationUtils.modelChangesValidationToModelValidationForSubtypes;

@ParametersAreNonnullByDefault
public class RestrictedCampaignsUpdateOperation {
    private Map<Long, BaseCampaign> campaignModelById;
    private final List<? extends ModelChanges<? extends BaseCampaign>> modelChanges;
    private List<? extends AppliedChanges<? extends BaseCampaign>> appliedChanges;

    private final CampaignModifyRepository campaignModifyRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final StrategyTypedRepository strategyTypedRepository;
    private final UpdateRestrictedCampaignValidationService updateRestrictedCampaignValidationService;

    private final CampaignUpdateOperationSupportFacade campaignUpdateOperationSupportFacade;

    private final CampaignAdditionalActionsService campaignAdditionalActionsService;
    private final DslContextProvider ppcDslContextProvider;
    private final RestrictedCampaignsUpdateOperationContainerImpl container;
    private final Set<String> enabledFeatures;
    private final Applicability applicability;

    @SuppressWarnings("checkstyle:parameternumber")
    public RestrictedCampaignsUpdateOperation(
            List<? extends ModelChanges<? extends BaseCampaign>> modelChanges,
            Long operatorUid,
            UidClientIdShard uidClientIdShard,
            CampaignModifyRepository campaignModifyRepository,
            CampaignTypedRepository campaignTypedRepository,
            StrategyTypedRepository strategyTypedRepository,
            UpdateRestrictedCampaignValidationService updateRestrictedCampaignValidationService,
            CampaignUpdateOperationSupportFacade campaignUpdateOperationSupportFacade,
            CampaignAdditionalActionsService campaignAdditionalActionsService,
            DslContextProvider ppcDslContextProvider,
            RbacService rbacService,
            RequestBasedMetrikaClientFactory metrikaClientFactory,
            FeatureService featureService,
            Applicability applicability,
            CampaignOptions options) {
        this.modelChanges = modelChanges;

        this.campaignModifyRepository = campaignModifyRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.strategyTypedRepository = strategyTypedRepository;
        this.updateRestrictedCampaignValidationService = updateRestrictedCampaignValidationService;
        this.campaignUpdateOperationSupportFacade = campaignUpdateOperationSupportFacade;
        this.campaignAdditionalActionsService = campaignAdditionalActionsService;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.applicability = applicability;

        var metrikaDataAdapter =
                metrikaClientFactory.createMetrikaClient(uidClientIdShard.getClientId());
        var campaignIds = mapList(modelChanges, mc -> mc.getId());

        container = new RestrictedCampaignsUpdateOperationContainerImpl(
                uidClientIdShard.getShard(),
                operatorUid,
                uidClientIdShard.getClientId(),
                uidClientIdShard.getUid(),
                rbacService.getChiefByClientId(uidClientIdShard.getClientId()),
                metrikaDataAdapter,
                options,
                campaignIds,
                getClientStrategiesToLinkToAndCampaignsOldStrategiesById(campaignIds, uidClientIdShard),
                new DisabledFieldsDataContainer(campaignIds)
        );
        enabledFeatures = featureService.getEnabledForClientId(uidClientIdShard.getClientId());
    }


    private ValidationResult<? extends List<? extends BaseCampaign>, Defect> fullValidate() {
        var vr = preValidate();
        onModelChangesValidated(vr);
        var vrBeforeApply = validateBeforeApply(vr);

        List<? extends ModelChanges<? extends BaseCampaign>> validItemsBeforeApply =
                ValidationResult.getValidItems(vrBeforeApply);
        ValidationResult<? extends List<? extends BaseCampaign>, Defect> validationResult =
                validate(vrBeforeApply, validItemsBeforeApply);

        campaignUpdateOperationSupportFacade.onAppliedChangesValidated(container, appliedChanges);

        return eraseSpecificWarningsIfErrors(validationResult);
    }

    public ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> preValidate() {
        List<Long> modelIds = mapList(modelChanges, ModelChanges::getId);
        Map<Long, CampaignsType> clientCampaignIdsWithDbType = campaignTypedRepository
                .getClientCampaignIdsWithType(container.getShard(), container.getClientId(),
                        modelIds);
        Map<Long, CampaignType> clientCampaignIdsWithCoreType = EntryStream.of(clientCampaignIdsWithDbType)
                .mapValues(CampaignType::fromSource)
                .toMap();

        fillContainerRuntimeClass(container, clientCampaignIdsWithCoreType, modelChanges);
        return updateRestrictedCampaignValidationService.preValidate(container, modelChanges,
                clientCampaignIdsWithCoreType.keySet());
    }

    private void fillContainerRuntimeClass(RestrictedCampaignsUpdateOperationContainerImpl container,
                                           Map<Long, CampaignType> clientCampaignIdsWithType,
                                           List<? extends ModelChanges<? extends BaseCampaign>> modelChanges) {
        for (var mc : modelChanges) {
            CampaignType campaignType = clientCampaignIdsWithType.get(mc.getId());
            if (campaignType != null) {
                container.setCampaignType(mc.getId(), campaignType);
            }
        }
    }

    private void onModelChangesValidated(ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> vr) {
        campaignUpdateOperationSupportFacade.onModelChangesValidated(container, ValidationResult.getValidItems(vr));
    }

    private ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> validateBeforeApply(
            ValidationResult<List<ModelChanges<BaseCampaign>>, Defect> vr) {
        List<ModelChanges<BaseCampaign>> modelChanges = ValidationResult.getValidItems(vr);
        List<Long> mcCounterIds = StreamEx.of(modelChanges)
                .filter(m -> m.isPropChanged(CampaignWithMetrikaCounters.METRIKA_COUNTERS))
                .map(m -> m.castModel(CampaignWithMetrikaCounters.class)
                        .getChangedProp(CampaignWithMetrikaCounters.METRIKA_COUNTERS)).filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .toList();
        List<Long> validItemIds = mapList(modelChanges, ModelChanges::getId);
        campaignModelById = getCampaignModels(validItemIds);

        Map<Long, Long> goalIdToCounterIdForCampaignsWithoutCounterIds;
        if (container.getOptions().isFetchMetrikaCountersForGoals()) {
            Set<Long> goalIdsToFetchCounters = new HashSet<>();
            for (var mc : modelChanges) {
                var campaign = campaignModelById.get(mc.getId());
                var metrikaCounters = getMetrikaCounters(mc, campaign);
                if (CollectionUtils.isNullOrEmpty(metrikaCounters)) {
                    Long goalId = getGoalId(mc, campaign);
                    if (goalId != null && !SPECIAL_GOAL_IDS.contains(goalId)) {
                        goalIdsToFetchCounters.add(goalId);
                    }
                }
                // Можно указывать goalId в meaningfulGoals даже если он относятся не к счётчику, указанному на
                // кампании (при условии, что этот goalId привязан к кампании).
                // Поэтому надо загрузить счётчики для meaningfulGoals, даже если на кампании указаны счётчики.
                // Покрыто аква-тестом:
                // StrategyAverageRoiGoalIdWithWrongMetrikaCountersTest#goalAddedToCampaign_updatePriorityGoal_Success
                List<Long> userMeaningfulGoalsIds = getMeaningfulGoalIdsExceptSpecial(mc, campaign);
                goalIdsToFetchCounters.addAll(userMeaningfulGoalsIds);
            }
            goalIdToCounterIdForCampaignsWithoutCounterIds = CampaignStrategyUtils.getGoalIdToCounterIdMap(
                    goalIdsToFetchCounters, container.getMetrikaClient());
        } else {
            goalIdToCounterIdForCampaignsWithoutCounterIds = emptyMap();
        }

        container.setGoalIdToCounterIdForCampaignsWithoutCounterIds(goalIdToCounterIdForCampaignsWithoutCounterIds);
        container.getMetrikaClient().setCampaignsCounterIds(campaignModelById.values(),
                unionToSet(mcCounterIds, goalIdToCounterIdForCampaignsWithoutCounterIds.values()));

        var shouldFetchUnavailableGoals = campaignModelById.values().stream()
                .anyMatch(campaign -> shouldFetchUnavailableGoals(campaign, enabledFeatures));
        container.getMetrikaClient().setShouldFetchUnavailableGoals(shouldFetchUnavailableGoals);
        return updateRestrictedCampaignValidationService.validateBeforeApply(container, vr, campaignModelById);
    }

    @Nullable
    private static List<Long> getMetrikaCounters(ModelChanges<BaseCampaign> mc, BaseCampaign campaign) {
        if (campaign instanceof CampaignWithMetrikaCounters) {
            return getPropertyValue(
                    mc.castModel(CampaignWithMetrikaCounters.class),
                    (CampaignWithMetrikaCounters) campaign,
                    CampaignWithMetrikaCounters.METRIKA_COUNTERS);
        }
        return null;
    }

    private static Long getGoalId(ModelChanges<BaseCampaign> mc, BaseCampaign campaign) {
        if (campaign instanceof CampaignWithCustomStrategy) {
            var strategy = getPropertyValue(
                    mc.castModel(CampaignWithCustomStrategy.class),
                    (CampaignWithCustomStrategy) campaign,
                    CampaignWithCustomStrategy.STRATEGY);
            return getStrategyGoalId(strategy);
        }
        return null;
    }

    /**
     * Возвращает meaningfulGoals, кроме specialGoalIds (ENGAGED_SESSION_GOAL_ID etc)
     */
    private static List<Long> getMeaningfulGoalIdsExceptSpecial(
            ModelChanges<BaseCampaign> mc,
            BaseCampaign campaign) {
        if (campaign instanceof CampaignWithMeaningfulGoals) {
            var meaningfulGoals = getPropertyValue(
                    mc.castModel(CampaignWithMeaningfulGoals.class),
                    (CampaignWithMeaningfulGoals) campaign,
                    CampaignWithMeaningfulGoals.MEANINGFUL_GOALS);
            return StreamEx.ofNullable(meaningfulGoals)
                    .flatMap(Collection::stream)
                    .map(MeaningfulGoal::getGoalId)
                    .nonNull()
                    .filter(not(SPECIAL_GOAL_IDS::contains))
                    .toList();
        }
        return emptyList();
    }

    private Map<Long, BaseCampaign> getCampaignModels(Collection<Long> ids) {
        List<? extends BaseCampaign> typedCampaigns =
                campaignTypedRepository.getTypedCampaigns(container.getShard(), ids);

        return listToMap(typedCampaigns, ModelWithId::getId);
    }

    private ValidationResult<? extends List<? extends BaseCampaign>, Defect> validate(
            ValidationResult<? extends List<? extends ModelChanges<? extends BaseCampaign>>, Defect> vrBeforeApply,
            List<? extends ModelChanges<? extends BaseCampaign>> validItemsBeforeApply) {

        Set<Long> idsOfValidModelChanges = listToSet(validItemsBeforeApply, ModelChanges::getId);
        Map<Integer, ? extends AppliedChanges<? extends BaseCampaign>> appliedChangesByIndex =
                getAppliedChangesByIndex(idsOfValidModelChanges);
        appliedChanges = EntryStream.of(appliedChangesByIndex)
                .sortedBy(Map.Entry::getKey)
                .values()
                .toList();

        var vrBeforeApplyWithoutValidSubNodes = cloneValidationResultSubNodesWithIssues(vrBeforeApply);
        var modelVr = convertModelChangesValidation(vrBeforeApplyWithoutValidSubNodes, appliedChanges);

        return updateRestrictedCampaignValidationService.validate(container, modelVr, appliedChangesByIndex);
    }

    private Map<Integer, ? extends AppliedChanges<? extends BaseCampaign>> getAppliedChangesByIndex(
            Set<Long> idsOfValidModelChanges) {
        return EntryStream.of(modelChanges)
                .filterValues(changes -> idsOfValidModelChanges.contains(changes.getId()))
                .mapValues(changes -> changes.castModelUp(BaseCampaign.class)
                        .applyTo(campaignModelById.get(changes.getId())))
                .toMap();
    }

    private ValidationResult<? extends List<? extends BaseCampaign>, Defect> convertModelChangesValidation(
            ValidationResult<? extends List<? extends ModelChanges<? extends BaseCampaign>>, Defect> vrBeforeApply,
            List<? extends AppliedChanges<? extends BaseCampaign>> appliedChanges) {
        campaignUpdateOperationSupportFacade.onChangesApplied(container, appliedChanges);

        List<? extends BaseCampaign> models = mapList(appliedChanges, AppliedChanges::getModel);
        return modelChangesValidationToModelValidationForSubtypes(vrBeforeApply, models,
                id -> new CampaignStub().withId(id));
    }

    public MassResult<Long> apply() {
        var vr = fullValidate();
        List<? extends BaseCampaign> validItems = ValidationResult.getValidItems(vr);

        if (vr.hasErrors() || validItems.isEmpty() || (isFull(applicability) && vr.hasAnyErrors())) {
            return MassResult.brokenMassAction(mapList(vr.getValue(), v -> null), vr);
        }

        Set<Long> idsOfValidModelChanges = listToSet(validItems, BaseCampaign::getId);

        beforeExecution(idsOfValidModelChanges);
        execute(idsOfValidModelChanges);
        afterExecution(idsOfValidModelChanges);
        return MassResult.successfulMassAction(getResult(vr), vr);
    }

    private void execute(Set<Long> idsOfValidModelChanges) {
        var validAppliedChanges = extractAppliedChangesByIds(idsOfValidModelChanges);
        CampaignAdditionalActionsContainer additionalActionsContainer = new CampaignAdditionalActionsContainer();
        campaignUpdateOperationSupportFacade.addToAdditionalActionsContainer(additionalActionsContainer,
                container, validAppliedChanges);

        ppcDslContextProvider.ppcTransaction(container.getShard(),
                createTransactionalUpdateTask(validAppliedChanges, container, additionalActionsContainer));

        campaignUpdateOperationSupportFacade.updateRelatedEntitiesOutOfTransaction(container,
                validAppliedChanges);
        campaignUpdateOperationSupportFacade.updateRelatedEntitiesOutOfTransactionWithModelChanges(container,
                modelChanges, validAppliedChanges);
    }

    private TransactionalRunnable createTransactionalUpdateTask(
            List<AppliedChanges<? extends BaseCampaign>> validAppliedChanges,
            RestrictedCampaignsUpdateOperationContainer updateParameters,
            CampaignAdditionalActionsContainer additionalActionsContainer) {
        return configuration -> {
            DSLContext dsl = configuration.dsl();
            campaignModifyRepository.updateCampaigns(dsl, updateParameters, validAppliedChanges);

            campaignUpdateOperationSupportFacade
                    .updateRelatedEntitiesInTransaction(dsl, updateParameters, validAppliedChanges);

            campaignAdditionalActionsService.processAdditionalActionsContainer(dsl, container.getOperatorUid(),
                    additionalActionsContainer);
        };
    }

    private List<Long> getResult(ValidationResult<? extends List<? extends BaseCampaign>, Defect> vr) {
        Map<Integer, BaseCampaign> validItemsWithIndex = ValidationResult.getValidItemsWithIndex(vr);

        return IntStreamEx.ofIndices(vr.getValue())
                .boxed()
                .map(validItemsWithIndex::get)
                .map(c -> ifNotNull(c, BaseCampaign::getId))
                .toList();
    }

    private void afterExecution(Set<Long> idsOfValidModelChanges) {
        var validAppliedChanges = extractAppliedChangesByIds(idsOfValidModelChanges);
        campaignUpdateOperationSupportFacade.afterExecution(validAppliedChanges, container);
    }

    private void beforeExecution(Set<Long> idsOfValidModelChanges) {
        var validAppliedChanges = extractAppliedChangesByIds(idsOfValidModelChanges);
        campaignUpdateOperationSupportFacade.beforeExecution(validAppliedChanges, container);
    }

    @Nonnull
    private List<AppliedChanges<? extends BaseCampaign>> extractAppliedChangesByIds(Set<Long> idsOfValidModelChanges) {
        return filterList(appliedChanges, x -> idsOfValidModelChanges.contains(x.getOldValue(BaseCampaign.ID)));
    }

    private Map<Long, BaseStrategy> getClientStrategiesToLinkToAndCampaignsOldStrategiesById(List<Long> campaignIds,
                                                                                             UidClientIdShard uidClientIdShard) {
        var filledStrategyIdsFromModelChanges = modelChanges.stream()
                .filter(mc -> CampaignWithPackageStrategy.class.isAssignableFrom(mc.getModelType()))
                .map(mc -> mc.castModel(CampaignWithPackageStrategy.class))
                .map(mc -> mc.getPropIfChanged(CampaignWithPackageStrategy.STRATEGY_ID))
                .filter(CommonUtils::isValidId)
                .collect(Collectors.toList());

        Map<Long, ? extends BaseCampaign> typedCampaigns =
                campaignTypedRepository.getTypedCampaignsMap(uidClientIdShard.getShard(), campaignIds);
        var changingCampaignsStrategyIdsWithFilledStrategyIdsFromModelChanges =
                StreamEx.of(typedCampaigns.values())
                        .select(CampaignWithPackageStrategy.class)
                        .map(CampaignWithPackageStrategy::getStrategyId)
                        .filter(CommonUtils::isValidId)
                        .append(filledStrategyIdsFromModelChanges)
                        .toList();

        return strategyTypedRepository.getIdToModelTyped(uidClientIdShard.getShard(), uidClientIdShard.getClientId(),
                changingCampaignsStrategyIdsWithFilledStrategyIdsFromModelChanges);
    }
}
