package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatus;
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatusInfo;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.service.ProductionCalendarProviderService;
import ru.yandex.direct.libs.timetarget.HoursCoef;
import ru.yandex.direct.libs.timetarget.ProductionCalendar;
import ru.yandex.direct.libs.timetarget.TimeTarget;

/**
 * Сервис для конвертации временных таргетингов кампании в статус на текущее время
 */
@Service
public class TimeTargetStatusService {
    private final ProductionCalendarProviderService productionCalendarProviderService;

    @Autowired
    public TimeTargetStatusService(ProductionCalendarProviderService productionCalendarProviderService) {
        this.productionCalendarProviderService = productionCalendarProviderService;
    }

    /**
     * Получить статус кампании с переданным временным таргетингом в заданной географической зоне в конкретный момент времени
     *
     * @param timeTarget     описание временного таргетинга
     * @param geoTimezone    географическая зона
     * @param currentInstant момент времени
     */
    public TimeTargetStatusInfo getTimeTargetStatus(TimeTarget timeTarget, GeoTimezone geoTimezone,
                                                    Instant currentInstant) {
        if (timeTarget == null || timeTarget.getWeekdayCoefs().isEmpty()) {
            // По умолчанию показы разрешены в любые часы
            return new TimeTargetStatusInfo()
                    .withStatus(TimeTargetStatus.ACTIVE);
        }
        // Сначала смотрим на текущее время в искомой таймзоне и проверяем на активность прямо сейчас
        ZoneId zoneId = geoTimezone.getTimezone();
        ZonedDateTime currentDateTime = currentInstant.atZone(zoneId);
        int year = currentDateTime.getYear();
        ProductionCalendar productionCalendar =
                productionCalendarProviderService.getProductionCalendar(year, geoTimezone.getRegionId());
        double coef = timeTarget.calcTimeTargetCoef(currentDateTime.toLocalDateTime(), productionCalendar);
        if (coef > 0) {
            // Временной таргетинг сейчас активен
            return new TimeTargetStatusInfo()
                    .withStatus(TimeTargetStatus.ACTIVE)
                    .withCoef(BigDecimal.valueOf(coef));
        }
        ZonedDateTime selectedDateTime = getNextShowDateTime(currentDateTime, timeTarget, geoTimezone, currentInstant);
        if (selectedDateTime == null) {
            // В течение месяца не нашлось ни одной подходящей даты
            return new TimeTargetStatusInfo()
                    .withStatus(TimeTargetStatus.LATER_THAN_WEEK);
        }
        // Смотрим, сколько дней пройдёт до выбранной даты
        long days = selectedDateTime.toLocalDate().toEpochDay() - currentDateTime.toLocalDate().toEpochDay();
        TimeTargetStatus state;
        if (days >= 7) {
            state = TimeTargetStatus.LATER_THAN_WEEK;
        } else if (days > 1) {
            state = TimeTargetStatus.THIS_WEEK;
        } else if (days == 1) {
            state = TimeTargetStatus.TOMORROW;
        } else {
            state = TimeTargetStatus.TODAY;
        }
        productionCalendar = productionCalendarProviderService
                .getProductionCalendar(selectedDateTime.getYear(), geoTimezone.getRegionId());
        coef = timeTarget.calcTimeTargetCoef(selectedDateTime.toLocalDateTime(), productionCalendar);
        return new TimeTargetStatusInfo()
                .withStatus(state)
                .withActivationTime(selectedDateTime.toOffsetDateTime())
                .withCoef(BigDecimal.valueOf(coef));
    }

    @Nullable
    private ZonedDateTime getNextShowDateTime(ZonedDateTime currentDateTime, TimeTarget timeTarget,
                                              GeoTimezone geoTimezone, Instant currentInstant) {
        // Нам нужно подобрать дату и время, в которые начнутся показы. Дальше, чем на месяц мы не заглядываем.
        // Код построен таким образом, чтобы быстро отобрать даты и часы, в которые показы принципиально возможны,
        // и только в этом случае используется дорогая работа с таймзонами. Отобранные даты/часы могут не подойти
        // по двум причинам: это время находится в прошлом, либо в принципе не может существовать из-за перевода
        // часов. Пример гипотетической ситуации: сейчас на часах 22:15, однако в 23:00 часы будут переведены на час
        // назад, и на часах снова станет 22:00. Если бы мы по каким-то причинам не могли показываться в 22:15, то
        // следующий момент времени будет в 22:00, через 45 минут. Однако на самом деле мы этот случай уже проверили
        // в самом начале функции, таким образом первый час, который вообще имеет смысл проверять - это 23:00 сегодня,
        // если предположить, что переводов часов более чем на час не бывает. Переводы часов достаточно редки, так что
        // в типичной ситуации первый же час, который мы выберем, будет нам подходить. При этом у нас будет до 30
        // шагов перебора дат (очень быстрая арифметика, т.к. LocalDate представляет из себя простой счётчик дней),
        // а в пределах одного дня до 24 возможных шагов, один из них которых c высокой вероятностью подойдёт. При
        // этом на работу с таймзонами должно уйти максимум 1-2 шага.
        // NOTE: Не учитывается возможность, что в 22:30 время перескочит на 21:30, однако мы всё-равно учитываем
        // только время начала показов вида hh:00, и проверка 21:00 не будет иметь большого смысла.
        // NOTE: Есть сомнения, насколько полезным здесь будет кеширование. К тому же тип TimeTarget сейчас не
        // хешируемый и не сравниваемый, а хеширование/сравнение/сериализация может оказаться дороже, чем вся логика
        // ниже.
        int year = currentDateTime.getYear();
        ZoneId zoneId = geoTimezone.getTimezone();

        ProductionCalendar productionCalendar =
                productionCalendarProviderService.getProductionCalendar(year, geoTimezone.getRegionId());

        LocalDate candidateDate = currentDateTime.toLocalDate();
        LocalDate borderDate = candidateDate.plusDays(31); // не смотрим дальше, чем 31 день
        int firstValidHour = currentDateTime.getHour() + 1; // в первый день смотрим только на следующий час
        while (candidateDate.isBefore(borderDate)) {
            int candidateYear = candidateDate.getYear();
            if (year != candidateYear) {
                // В декабре можно дойти до января будущего года
                year = candidateYear;
                productionCalendar =
                        productionCalendarProviderService.getProductionCalendar(year, geoTimezone.getRegionId());
            }

            HoursCoef hoursCoef = timeTarget.getHoursCoef(candidateDate, productionCalendar);
            if (hoursCoef != null) {
                // Выбираем часы, в которые разрешены показы
                for (int hour = firstValidHour; hour < 24; ++hour) {
                    if (hoursCoef.getCoefForHour(hour) > 0) {
                        // Формируем дату и время в указанный час и смотрим по каким оффсетам оно может существовать
                        LocalDateTime candidateDateTime = LocalDateTime.of(candidateDate, LocalTime.of(hour, 0));
                        for (ZoneOffset zoneOffset : zoneId.getRules().getValidOffsets(candidateDateTime)) {
                            Instant candidateInstant = candidateDateTime.toInstant(zoneOffset);
                            if (candidateInstant.isAfter(currentInstant)) {
                                // Данное условие может не срабатывать в сегодняшний до момента перевода часов
                                return candidateDateTime.atOffset(zoneOffset).atZoneSameInstant(zoneId);
                            }
                        }
                    }
                }
            }
            candidateDate = candidateDate.plusDays(1);
            firstValidHour = 0;
        }
        return null;
    }
}
