package ru.yandex.direct.core.entity.campaign.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbutil.model.ClientId
import java.math.BigDecimal

@Service
class WalletHasMoneyChecker(
    private val campaignRepository: CampaignRepository,
    private val clientRepository: ClientRepository
) {
    fun calcHasMoney(
        shard: Int,
        campaigns: List<CommonCampaign>,
        wallets: List<WalletTypedCampaign>
    ): Map<Long, Boolean> {
        val clientsAutoOverdraftInfo = clientRepository.getClientsAutoOverdraftInfo(
            shard,
            campaigns.map { ClientId.fromLong(it.clientId) }.distinct()
        ).associateBy { it.clientId }

        val walletsMap = wallets.associateBy { it.id }
        val walletsDebt = campaignRepository.getWalletsDebt(shard, walletsMap.keys)

        return campaigns.associate {
            it.id to calcHasMoneyForWallet(
                campaign = it,
                wallet = walletsMap[it.walletId],
                clientsAutoOverdraftInfo = clientsAutoOverdraftInfo,
                walletDebt = walletsDebt[it.walletId]
            )
        }
    }

    fun calcHasMoneyForStrategies(
        shard: Int,
        strategies: List<CommonStrategy>,
        wallets: List<WalletTypedCampaign>
    ): Map<Long, Boolean> {
        val clientsAutoOverdraftInfo = clientRepository.getClientsAutoOverdraftInfo(
            shard,
            strategies.map { ClientId.fromLong(it.clientId) }.distinct()
        ).associateBy { it.clientId }
        val walletsMap = wallets.associateBy { it.id }
        val walletsDebt = campaignRepository.getWalletsDebt(shard, walletsMap.keys)

        return strategies.associate {
            it.id to calcHasMoneyForWallet(
                wallet = walletsMap[it.walletId],
                clientsAutoOverdraftInfo = clientsAutoOverdraftInfo,
                walletDebt = walletsDebt[it.walletId]
            )
        }
    }

    private fun calcHasMoneyForWallet(
        campaign: CommonCampaign,
        wallet: WalletTypedCampaign?,
        clientsAutoOverdraftInfo: Map<Long, ClientAutoOverdraftInfo>,
        walletDebt: BigDecimal?
    ): Boolean {
        if (campaign.sum - campaign.sumSpent > Currencies.EPSILON) {
            return true
        }
        return calcHasMoneyForWallet(wallet, clientsAutoOverdraftInfo, walletDebt)
    }

    private fun calcHasMoneyForWallet(
        wallet: WalletTypedCampaign?,
        clientsAutoOverdraftInfo: Map<Long, ClientAutoOverdraftInfo>,
        walletDebt: BigDecimal?
    ): Boolean {
        if (wallet == null) {
            return false
        }
        val overdraftAddition = clientsAutoOverdraftInfo[wallet.clientId]?.let {
            AutoOverdraftUtils.calculateAutoOverdraftAddition(
                wallet.currency, wallet.sum, walletDebt?.negate() ?: BigDecimal.ZERO, it
            )
        } ?: BigDecimal.ZERO

        return wallet.sum - (walletDebt ?: BigDecimal.ZERO) + overdraftAddition > Currencies.EPSILON
    }
}
