package ru.yandex.direct.core.entity.campaign.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.container.WalletsWithCampaigns;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.currency.Money;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Инкапсуляция логики расчёта остатка на общем счёте и создания экземпляров {@link WalletRestMoney}.
 */
@ParametersAreNonnullByDefault
class WalletRestMoneyCalculator {

    private final WalletsWithCampaigns walletsWithCampaigns;

    /**
     * Создать немутабельный экземпляр для расчёта.
     */
    WalletRestMoneyCalculator(WalletsWithCampaigns walletsWithCampaigns) {
        this.walletsWithCampaigns = walletsWithCampaigns;
    }

    /**
     * Рассчитать остаток на общем счёте.
     * <p>
     * Берётся остаток на кампании-кошельке, и из него вычитаются абсолютные значения
     * "задолженностей" (отрицательных остатоков) на его кампаниях.
     * Таким образом учитывается компенсация.
     * Положительные собственные остатки на кампаниях не учитываются.
     * <p>
     * Можно запрашивать результат по кампаниям, не переданным в конструктор. В таком случае
     * подразумевается, что кампания не под общим счётом, будет возвращено значение
     * с нулевым остатком – см. {@link WalletRestMoney}.
     *
     * @param campaigns кампании
     * @return Мапа, где по каждому id переданных кампаний будет содержаться экземпляр {@link WalletRestMoney},
     * который в свою очередь хранит информацию об остатке на общем счёте. Если кампания не подключена
     * к общему счёту, соответствующее ей значение будет не {@code null} и будет хранить нулевой остаток.
     * @see <a href="https://wiki.yandex-team.ru/users/hmepas/campaignssum/">Описание на wiki</a>
     */
    Map<Long, WalletRestMoney> getWalletRestMoneyByCampaignIds(Collection<Campaign> campaigns) {
        Map<Long, BigDecimal> restByWalletId = StreamEx.of(campaigns)
                .map(Campaign::getWalletId)
                .nonNull().distinct()
                .filter(walletsWithCampaigns::containsWalletWithId)
                .mapToEntry(walletsWithCampaigns::getWallet)
                .mapValues(this::calculateWalletRest)
                .toMap();

        Map<Long, WalletRestMoney> resultByWalletId = EntryStream.of(restByWalletId)
                .mapToValue(this::toRestMoney).toMap();

        Map<Long, WalletRestMoney> resultByCampaignId = new HashMap<>();

        for (Campaign campaign : campaigns) {
            // если у кампании нет кошелька, мб она либо сама кошелёк, либо её нет
            // в #walletsWithCampaigns – тогда предполагаем, что кампания без ОС, ставим нулевой остаток
            Long walletId = nvl(campaign.getWalletId(), campaign.getId());

            WalletRestMoney walletRest = resultByWalletId.getOrDefault(walletId, zeroRestMoney(campaign));

            resultByCampaignId.put(campaign.getId(), walletRest);
        }
        return resultByCampaignId;
    }

    Map<Long, WalletRestMoney> getWalletRestMoneyByWalletCampaignIds() {
        return StreamEx.of(walletsWithCampaigns.getAllWallets())
                .mapToEntry(WalletCampaign::getId, this::calculateWalletRest)
                .mapToValue(this::toRestMoney)
                .toMap();
    }

    /**
     * Посчитать остаток на кампании-кошельке, используя данные о привязанных к этому кошельку кампаниях.
     *
     * @param wallet кампания-кошелёк
     * @return Остаток с учётом компенсаций в счёт кампаний с отрицательными собственными остатками.
     * Результат не будет превышать значение собственного остатка на кампании-кошельке.
     */
    private BigDecimal calculateWalletRest(WalletCampaign wallet) {
        BigDecimal walletRest = calculateSelfRest(wallet);
        for (WalletCampaign campaignUnderWallet : walletsWithCampaigns.getCampaignsBoundTo(wallet)) {
            BigDecimal campaignRest = calculateSelfRest(campaignUnderWallet);
            if (campaignRest.compareTo(BigDecimal.ZERO) < 0) {
                walletRest = walletRest.add(campaignRest);
            }
        }
        return walletRest;
    }

    /**
     * Посчитать собственный остаток <i>на обычной кампании или на кошельке</i>.
     *
     * @param campaign кампания, обычная или кошелёк
     * @return Собственный остаток: {@link Campaign#SUM} - {@link Campaign#SUM_SPENT}.
     */
    private BigDecimal calculateSelfRest(WalletCampaign campaign) {
        return campaign.getSum().subtract(campaign.getSumSpent());
    }

    private WalletRestMoney toRestMoney(long walletId, BigDecimal restSum) {
        checkState(walletsWithCampaigns.containsWalletWithId(walletId), "No wallet with id %s found.", walletId);
        return new WalletRestMoney()
                .withWalletId(walletId)
                .withRest(Money.valueOf(restSum, walletsWithCampaigns.getWallet(walletId).getCurrency()));
    }

    private static WalletRestMoney zeroRestMoney(Campaign campaign) {
        return new WalletRestMoney()
                .withWalletId(null)
                .withRest(Money.valueOf(BigDecimal.ZERO, campaign.getCurrency()));
    }
}
