package ru.yandex.direct.core.entity.campaign.service;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.base.Preconditions.checkState;

@Service
public class WalletService {
    private static final String WALLET_ENABLE_LOCK_PREFIX = "WALLET_ENABLE_LOCK";

    private final UserService userService;

    private final WalletRepository walletRepository;

    private final ShardHelper shardHelper;

    private final ClientService clientService;

    private final CampaignRepository campaignRepository;

    private final DslContextProvider ppcDslContextProvider;

    private final CommonCampaignService commonCampaignService;

    private final RbacService rbacService;

    @Autowired
    public WalletService(UserService userService, WalletRepository walletRepository,
                         ShardHelper shardHelper, ClientService clientService, CampaignRepository campaignRepository,
                         DslContextProvider ppcDslContextProvider, CommonCampaignService commonCampaignService,
                         RbacService rbacService) {
        this.userService = userService;
        this.walletRepository = walletRepository;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.campaignRepository = campaignRepository;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.commonCampaignService = commonCampaignService;
        this.rbacService = rbacService;

    }

    @Nonnull
    public List<Wallet> massGetWallets(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .flatMapKeyValue(
                        (shard, ids) -> walletRepository.getAllWalletExistingCampByClientId(shard, ids).stream())
                .toList();
    }

    public Wallet getWalletForNewCampaigns(
            RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
            @Nullable UidAndClientId agencyUidAndClientId) {
        int shard = shardHelper.getShardByClientId(addCampaignParametersContainer.getClientId());
        @Nullable List<Long> agencyRepresentativeUids = null;
        if (agencyUidAndClientId != null) {
            agencyRepresentativeUids = userService.massGetUidsByClientIds(Set.of(agencyUidAndClientId.getClientId()))
                    .get(agencyUidAndClientId.getClientId());
        }
        return walletRepository.getWalletForCampaigns(shard, addCampaignParametersContainer.getChiefUid(),
                agencyRepresentativeUids);
    }

    /**
     * Создание новому клиенту общего счёта. Исходим из того, что у него нет кампаний
     * Возвращает id кошелька
     */
    public Long createWalletForNewClient(ClientId clientId, Long uid) {
        int shard = shardHelper.getShardByClientId(clientId);
        long chiefUid = rbacService.getChiefByClientId(clientId);
        RestrictedCampaignsAddOperationContainer addCampaignParametersContainer =
                RestrictedCampaignsAddOperationContainer.create(
                shard, uid, clientId, uid, chiefUid);

        return createWalletIfNeeded(addCampaignParametersContainer);
    }

    /**
     * Создаёт общий счёт при необходимости и возвращает id созданного/уже существующего кошелька
     * При создании  смотрит, не выставлен ли clients_options.client_flags.create_without_wallet
     * и есть ли уже общий счёт
     */
    public Long createWalletIfNeeded(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer) {
        Client client = clientService.getClient(addCampaignParametersContainer.getClientId());

        UidAndClientId agencyUidAndClientId =
                userService.getAgencyUidAndClientId(addCampaignParametersContainer.getOperatorUid(),
                        client.getAgencyUserId(), client.getAgencyClientId());

        String lockName = getLockName(client, agencyUidAndClientId);

        boolean isFirstCampaignsUnderWallet =
                campaignRepository.isFirstCampaignsUnderWalletRegardlessOfCampaignTypes(
                        addCampaignParametersContainer.getShard(),
                        addCampaignParametersContainer.getClientId(), Collections.emptySet());

        if (isFirstCampaignsUnderWallet && !client.getSharedAccountDisabled()) {
            return createWalletOrGetWalletId(addCampaignParametersContainer, client, agencyUidAndClientId, lockName);
        }
        return null;
    }

    private Long createWalletOrGetWalletId(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                           Client client,
                                           UidAndClientId agencyUidAndClientId,
                                           String lockName) {
        DSLContext context = ppcDslContextProvider.ppc(addCampaignParametersContainer.getShard());
        try {
            Integer lockResult = context
                    .select(SqlUtils.mysqlGetLock(lockName, Duration.ofSeconds(0))).fetchOne().component1();
            checkState(Objects.equals(lockResult, 1), "failed to get mysql lock with name: " + lockName);

            Wallet wallet = getWalletForNewCampaigns(addCampaignParametersContainer,
                    agencyUidAndClientId);
            boolean hasNoWallet = wallet == null;
            //Если у пользователя не было кампаний, у пользователя явно не указано -- не создавать общий счет
            //у пользователя еще нет кошелька
            //то при создании кампании создадим кошелек.
            if (hasNoWallet) {
                return commonCampaignService.addWalletOnCampaignCreating(addCampaignParametersContainer,
                        client,
                        agencyUidAndClientId,
                        client.getWorkCurrency());
            } else {
                return wallet.getWalletCampaignId();
            }
        } finally {
            context.select(SqlUtils.mysqlReleaseLock(lockName)).execute();
        }
    }

    /**
     * Есть ли у пользователя включенный общий счёт
     */
    public boolean hasEnabledSharedWallet(long uid) {
        return hasEnabledSharedWallet(userService.getUser(uid));
    }

    /**
     * Есть ли у пользователя включенный общий счёт
     */
    public boolean hasEnabledSharedWallet(@Nonnull User user) {
        int shard = shardHelper.getShardByClientId(user.getClientId());
        List<Long> agencyUids = null;
        if (user.getAgencyClientId() != null) {
            ClientId agencyClientId = ClientId.fromLong(user.getAgencyClientId());
            agencyUids = userService.massGetUidsByClientIds(List.of(agencyClientId)).get(agencyClientId);
        }
        Wallet wallet = walletRepository.getWalletForCampaigns(shard, user.getChiefUid(), agencyUids);
        return wallet != null && (wallet.getEnabled() != null && wallet.getEnabled() ||
                campaignRepository.isFirstCampaignsUnderWalletRegardlessOfCampaignTypes(
                        shard, user.getClientId(), Collections.emptySet()));
    }

    private static String getLockName(Client client,
                                      @Nullable UidAndClientId agencyUidAndClientId) {
        return StreamEx
                .of(WALLET_ENABLE_LOCK_PREFIX,
                        client.getClientId(),
                        agencyUidAndClientId != null ? agencyUidAndClientId.getClientId() : 0,
                        client.getWorkCurrency())
                .joining("_");
    }

}
