package ru.yandex.direct.core.entity.campaign.service.accesschecker;

import java.util.Objects;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheckDefaultImpl;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public abstract class AbstractCampaignAccessibiltyChecker<T extends CampaignForAccessCheckDefaultImpl> implements CampaignAccessibiltyChecker<T> {

    /**
     * Список поддерживаемых типов кампаний
     * Для того что не входит в данный список будет возвращаться NotSupported
     */
    private final Set<CampaignType> editableTypes;
    /**
     * Список допустимых типов кампаний
     * Для того что не входит в данный список будет возвращаться NotFound
     */
    protected final Set<CampaignType> allowableTypes;
    /**
     * Список источников, кампании которых можно менять
     * Для того что не входит в данный список будет возвращаться NoRight
     */
    private final Set<CampaignSource> editableSources;
    /**
     * Список источников, кампании которых доступны
     * Для того что не входит в данный список будет возвращаться NotFound
     */
    protected final Set<CampaignSource> allowableSources;

    public AbstractCampaignAccessibiltyChecker(
            Set<CampaignType> editableTypes, Set<CampaignType> allowableTypes,
            Set<CampaignSource> editableSources, Set<CampaignSource> allowableSources) {
        //нужно для корректной выборки из репозитория, при изменении требования надо править toAllowableCampaignsRepositoryFilter
        checkState(CollectionUtils.isNotEmpty(allowableTypes), "empty allowable types does not make sense");
        checkState(CollectionUtils.isNotEmpty(allowableSources), "empty allowable sources does not make sense");
        this.editableTypes = Objects.requireNonNull(editableTypes, "editableTypes");
        this.allowableTypes = Objects.requireNonNull(allowableTypes, "allowableTypes");
        this.editableSources = Objects.requireNonNull(editableSources, "editableSource");
        this.allowableSources = Objects.requireNonNull(allowableSources, "allowableSources");
    }

    protected boolean isEditableType(T campaign) {
        return editableTypes.contains(campaign.getType());
    }

    protected boolean isAllowableType(T campaign) {
        return allowableTypes.contains(campaign.getType());
    }

    @Override
    public boolean isAllowable(T campaign) {
        return isAllowableType(campaign) && allowableSources.contains(nvl(campaign.getSource(), CampaignSource.DIRECT));
    }

    @Override
    public boolean isEditable(T campaign) {
        return isEditableType(campaign) && editableSources.contains(nvl(campaign.getSource(), CampaignSource.DIRECT));
    }
}
