package ru.yandex.direct.core.entity.campaign.service.accesschecker;

import java.util.Objects;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.result.Defect;

/**
 * Дефекты которые могут возникнуть при валидации доступа к кампании
 */
@ParametersAreNonnullByDefault
public final class CampaignAccessDefects {

    private final Function<Long, Defect> typeNotAllowable;
    private final Function<Long, Defect> notVisible;
    private final Function<Long, Defect> typeNotSupported;
    private final Function<Long, Defect> noRights;
    private final Function<Long, Defect> archivedModification;

    private CampaignAccessDefects(
            Function<Long, Defect> typeNotAllowable,
            Function<Long, Defect> notVisible,
            Function<Long, Defect> typeNotSupported,
            Function<Long, Defect> noRights,
            Function<Long, Defect> archivedModification) {
        this.typeNotAllowable = Objects.requireNonNull(typeNotAllowable, "typeNotAllowable");
        this.notVisible = Objects.requireNonNull(notVisible, "notVisible");
        this.typeNotSupported = Objects.requireNonNull(typeNotSupported, "typeNotSupported");
        this.noRights = Objects.requireNonNull(noRights, "noRights");
        this.archivedModification = Objects.requireNonNull(archivedModification, "archivedModification");
    }

    /**
     * Объект находится в кампании с недопустимым типом objectId -> Defect
     */
    Function<Long, Defect> getTypeNotAllowable() {
        return typeNotAllowable;
    }

    /**
     * Объект находится в кампании недоступной для чтения objectId -> Defect
     */
    Function<Long, Defect> getNotVisible() {
        return notVisible;
    }

    /**
     * Объект находится в неподдерживаемом типе кампании objectId -> Defect
     */
    Function<Long, Defect> getTypeNotSupported() {
        return typeNotSupported;
    }

    /**
     * Объект находится в кампании недоступной для записи objectId -> Defect
     */
    Function<Long, Defect> getNoRights() {
        return noRights;
    }

    /**
     * Делается попытка модификации объекта в архивной кампании objectId -> Defect
     */
    Function<Long, Defect> getArchivedModification() {
        return archivedModification;
    }

    /**
     * Создать на основе данного объекта Builder. Необходимо в сценариях когда нужно на основе данного
     * объекта создать новый, в котором необходимо поменять не все поля
     */
    public Builder toBuilder() {
        return new Builder()
                .withTypeNotAllowable(typeNotAllowable)
                .withNotVisible(notVisible)
                .withTypeNotSupported(typeNotSupported)
                .withNoRights(noRights)
                .withArchivedModification(archivedModification);
    }

    public static class Builder {
        private Function<Long, Defect> typeNotAllowable;
        private Function<Long, Defect> notVisible;
        private Function<Long, Defect> typeNotSupported;
        private Function<Long, Defect> noRights;
        private Function<Long, Defect> archivedModification;

        public Builder withTypeNotAllowable(Function<Long, Defect> typeNotAllowable) {
            this.typeNotAllowable = typeNotAllowable;
            return this;
        }

        public Builder withTypeNotAllowable(Defect<Void> typeNotAllowable) {
            this.typeNotAllowable = id -> typeNotAllowable;
            return this;
        }

        public Builder withNotVisible(Function<Long, Defect> notVisible) {
            this.notVisible = notVisible;
            return this;
        }

        public Builder withNotVisible(Defect notVisible) {
            this.notVisible = id -> notVisible;
            return this;
        }

        public Builder withTypeNotSupported(Function<Long, Defect> typeNotSupported) {
            this.typeNotSupported = typeNotSupported;
            return this;
        }

        public Builder withTypeNotSupported(Defect typeNotSupported) {
            this.typeNotSupported = id -> typeNotSupported;
            return this;
        }

        public Builder withNoRights(Function<Long, Defect> noRights) {
            this.noRights = noRights;
            return this;
        }

        public Builder withNoRights(Defect noRights) {
            this.noRights = id -> noRights;
            return this;
        }

        public Builder withArchivedModification(Function<Long, Defect> archivedModification) {
            this.archivedModification = archivedModification;
            return this;
        }

        public Builder withArchivedModification(Defect archivedModification) {
            this.archivedModification = id -> archivedModification;
            return this;
        }

        public CampaignAccessDefects build() {
            return new CampaignAccessDefects(
                    typeNotAllowable, notVisible, typeNotSupported, noRights, archivedModification);
        }
    }
}
