package ru.yandex.direct.core.entity.campaign.service.accesschecker;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacService;

import static java.util.Collections.unmodifiableMap;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.core.entity.campaign.CampaignUtils.CPM_TYPES;
import static ru.yandex.direct.utils.CommonUtils.memoize;

/**
 * Объект осуществляет проверки суб-объектов кампаний (и самих кампаний тоже)
 **/
public class CampaignSubObjectAccessChecker<T extends CampaignForAccessCheck> {
    private final RbacService rbacService;
    /**
     * Отображение идентификатора проверяемого объекта в кампанию для проверки доступности
     */
    private final Supplier<Map<Long, T>> lazySubObjectIdToCamp;
    private final Supplier<Set<Long>> lazyCampaignIds;
    /**
     * Множество идентификаторов доступных для чтения объектов
     */
    private final Supplier<Set<Long>> lazyVisibleSubObjects;
    /**
     * Множество идентификаторов доступных для чтения объектов с точки зрения accessChecker
     */
    private final Supplier<Set<Long>> lazyAllowableSubObjects;
    /**
     * Множество идентификаторов доступных для записи объектов в том числе и в соответствии с accessChecker
     */
    private final Supplier<Set<Long>> lazyWritableAndEditableSubObjects;

    /**
     * Контейнер, в который валидаторы смогут складывать затронутые кампании
     */
    private final AffectedCampaignIdsContainer affectedCampaignIdsContainer;

    /**
     * Модель доступа
     */
    private final CampaignAccessibiltyChecker<T> campaignAccessibiltyChecker;

    CampaignSubObjectAccessChecker(RbacService rbacService,
                                   Long operatorUid, UidClientIdShard client, Collection<Long> subObjectIds,
                                   CampaignSubObjectRetriever<T> retriever,
                                   AffectedCampaignIdsContainer affectedCampaignIdsContainer,
                                   CampaignAccessibiltyChecker<T> campaignAccessibiltyChecker,
                                   FeatureService featureService) {
        this(rbacService, operatorUid, client.getClientId(), client.getShard(), subObjectIds, retriever,
                affectedCampaignIdsContainer, campaignAccessibiltyChecker, featureService);
    }

    CampaignSubObjectAccessChecker(RbacService rbacService,
                                   Long operatorUid, ClientId clientId, int shard, Collection<Long> subObjectIds,
                                   CampaignSubObjectRetriever<T> retriever,
                                   AffectedCampaignIdsContainer affectedCampaignIdsContainer,
                                   CampaignAccessibiltyChecker<T> campaignAccessibiltyChecker,
                                   FeatureService featureService) {
        this.rbacService = rbacService;
        this.affectedCampaignIdsContainer = affectedCampaignIdsContainer;
        this.campaignAccessibiltyChecker = campaignAccessibiltyChecker;

        lazySubObjectIdToCamp = memoize(() -> retriever.get(shard,
                campaignAccessibiltyChecker.toAllCampaignsRepositoryAdapter(clientId),
                subObjectIds));
        lazyCampaignIds = memoize(() -> getCampaignIds(lazySubObjectIdToCamp.get()));
        lazyVisibleSubObjects = memoize(() -> {
            Set<Long> visibleCampaignIds = filterVisibleCampaignIds(operatorUid, lazyCampaignIds.get());
            return filterSubObjectsByCampaign(CampaignForAccessCheck::getId, visibleCampaignIds::contains);
        });
        lazyAllowableSubObjects = memoize(() -> filterSubObjectsByCampaign(
                Function.identity(),
                campaignAccessibiltyChecker::isAllowable)
        );
        lazyWritableAndEditableSubObjects = memoize(() -> {
            Set<Long> writableCampaignIds = filterWritableCampaignIds(operatorUid, lazyCampaignIds.get());
            return filterSubObjectsByCampaign(
                    Function.identity(),
                    camp -> writableCampaignIds.contains(camp.getId()) && campaignAccessibiltyChecker.isEditable(camp) &&
                            (!CPM_TYPES.contains(camp.getType()) || !featureService.isEnabledForClientId(clientId,
                                    FeatureName.IS_CPM_BANNER_CAMPAIGN_DISABLED))
            );
        });
    }

    /**
     * Создает новый валидатор доступа к кампаниям
     */
    public CampaignSubObjectAccessValidator createValidator(CampaignAccessType desiredAccess) {
        return new CampaignSubObjectAccessValidator(this, desiredAccess, affectedCampaignIdsContainer);
    }

    /**
     * Создает новый валидатор доступа к кампаниям с указанными дефектами
     */
    public CampaignSubObjectAccessValidator createValidator(CampaignAccessType desiredAccess,
                                                            CampaignAccessDefects accessDefects) {
        return new CampaignSubObjectAccessValidator(this, desiredAccess, accessDefects, affectedCampaignIdsContainer);
    }

    /**
     * Создает новый валидатор доступа к объявлениям и их кампаниям
     */
    public CampaignSubObjectAccessValidator createAdsValidator(CampaignAccessType desiredAccess) {
        return new CampaignSubObjectAccessValidator(this, desiredAccess, AccessDefectPresets.AD_ACCESS_DEFECTS,
                affectedCampaignIdsContainer);
    }

    /**
     * Создает новый валидатор доступа к группам объявлений и их кампаниям
     */
    public CampaignSubObjectAccessValidator createAdGroupValidator(CampaignAccessType desiredAccess) {
        return createAdGroupValidator(desiredAccess, AccessDefectPresets.AD_GROUP_ACCESS_DEFECTS_FOR_RETARGETING);
    }

    /**
     * Создает новый валидатор доступа к группам объявлений и их кампаниям
     */
    public CampaignSubObjectAccessValidator createAdGroupValidator(
            CampaignAccessType desiredAccess,
            CampaignAccessDefects accessDefects) {
        return new CampaignSubObjectAccessValidator(this, desiredAccess, accessDefects,
                affectedCampaignIdsContainer);
    }

    public CampaignSubObjectAccessConstraint createBidModifierConstraint(CampaignAccessType desiredAccess,
                                                                         Map<Long, Long> campaignIdsByMultiplierIds) {
        return new CampaignSubObjectAccessConstraint(
                this, campaignIdsByMultiplierIds::get, AccessDefectPresets.BID_MODIFIER_ACCESS_DEFECTS,
                desiredAccess, affectedCampaignIdsContainer);
    }

    public Map<Long, CampaignForAccessCheck> getSubObjectIdToCamp() {
        return unmodifiableMap(lazySubObjectIdToCamp.get());
    }

    /**
     * Получить идентификаторы объектов, принадлежащих видимым оператору кампаниям
     */
    Set<Long> getVisible() {
        return lazyVisibleSubObjects.get();
    }

    /**
     * Получить идентификаторы объектов, доступных в соответствии с accessChecker
     */
    Set<Long> getAllowable() {
        return lazyAllowableSubObjects.get();
    }

    public Set<Long> getUnsupported(CampaignAccessibiltyChecker<? super T> accessibiltyChecker) {
        return filterSubObjectsByCampaign(Function.identity(), not(accessibiltyChecker::isAllowable));
    }

    /**
     * Получить идентификаторы объектов, принадлежащих кампаниям, которые оператор имеет право модифицировать в том
     * числе и в соответствии с accessChecker
     */
    Set<Long> getWritableAndEditable() {
        return lazyWritableAndEditableSubObjects.get();
    }

    /**
     * Проверить, что объект принадлежит видимой оператору кампании
     */
    public boolean objectInVisibleCampaign(Long objectId) {
        return getVisible().contains(objectId);
    }

    /**
     * Проверить, что объект принадлежит кампании, доступной для модификации в том числе и в соответсвии с accessChecker
     */
    public boolean objectInWritableAndEditableCampaign(Long objectId) {
        return getWritableAndEditable().contains(objectId);
    }

    /**
     * Проверить, что объект принадлежит архивной кампании
     */
    public boolean objectInArchivedCampaign(Long objectId) {
        return getCampaignFor(objectId)
                .map(CampaignForAccessCheck::getArchived)
                .orElse(false);
    }

    boolean objectInAllowableCampaign(Long objectId) {
        return getAllowable().contains(objectId);
    }

    Optional<T> getCampaignFor(Long objectId) {
        return Optional.ofNullable(lazySubObjectIdToCamp.get().get(objectId));
    }

    private static Set<Long> getCampaignIds(Map<Long, ? extends CampaignForAccessCheck> subObjectToCampaign) {
        return EntryStream.of(subObjectToCampaign).values().map(CampaignForAccessCheck::getId).toSet();
    }

    private Set<Long> filterVisibleCampaignIds(Long operatorUid, Collection<Long> campaignIds) {
        return rbacService.getVisibleCampaigns(operatorUid, campaignIds);
    }

    private Set<Long> filterWritableCampaignIds(Long operatorUid, Collection<Long> campaignIds) {
        return rbacService.getWritableCampaigns(operatorUid, campaignIds);
    }

    private <U> Set<Long> filterSubObjectsByCampaign(Function<T, U> getter, Predicate<U> predicate) {
        return EntryStream.of(lazySubObjectIdToCamp.get()).mapValues(getter).filterValues(predicate).keys().toSet();
    }
}
