package ru.yandex.direct.core.entity.campaign.service.accesschecker;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.repository.CampaignAccessCheckRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacService;


@ParametersAreNonnullByDefault
@Component
public class CampaignSubObjectAccessCheckerFactory {
    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final CampaignAccessCheckRepository campaignAccessCheckRepository;
    private final AffectedCampaignIdsContainer affectedCampaignIdsContainer;
    private final RequestCampaignAccessibilityCheckerProvider accessibilityCheckerProvider;
    private final FeatureService featureService;
    @Autowired
    public CampaignSubObjectAccessCheckerFactory(ShardHelper shardHelper,
                                                 RbacService rbacService,
                                                 CampaignAccessCheckRepository campaignAccessCheckRepository,
                                                 AffectedCampaignIdsContainer affectedCampaignIdsContainer,
                                                 RequestCampaignAccessibilityCheckerProvider accessibilityCheckerProvider,
                                                 FeatureService featureService) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.campaignAccessCheckRepository = campaignAccessCheckRepository;
        this.affectedCampaignIdsContainer = affectedCampaignIdsContainer;
        this.accessibilityCheckerProvider = accessibilityCheckerProvider;
        this.featureService = featureService;
    }

    public CampaignSubObjectAccessChecker newCampaignChecker(Long operatorUid, ClientId clientId,
                                                             Collection<Long> campaignIds) {
        return create(operatorUid, clientId, campaignIds,
                campaignAccessCheckRepository::getCampaignsForAccessCheckByCampaignIds);
    }

    public CampaignSubObjectAccessChecker newCampaignChecker(Long operatorUid, UidClientIdShard client,
                                                             Collection<Long> campaignIds) {
        return create(operatorUid, client, campaignIds,
                campaignAccessCheckRepository::getCampaignsForAccessCheckByCampaignIds);
    }

    public CampaignSubObjectAccessChecker newAdsChecker(Long operatorUid, ClientId clientId,
                                                        Collection<Long> adsIds) {
        return create(operatorUid, clientId, adsIds,
                campaignAccessCheckRepository::getCampaignsForAccessCheckByBannerIds);
    }

    public CampaignSubObjectAccessChecker newDynamicTextAdTargetChecker(Long operatorUid, ClientId clientId,
                                                                        Collection<Long> dynamicTextAdTargetIds) {
        return create(operatorUid, clientId, dynamicTextAdTargetIds,
                campaignAccessCheckRepository::getCampaignsByDynCondIdsFromDomainTypeAdgroup);
    }

    public CampaignSubObjectAccessChecker newDynamicFeedAdTargetChecker(Long operatorUid, ClientId clientId,
                                                                        Collection<Long> dynamicFeedAdTargetIds) {
        return create(operatorUid, clientId, dynamicFeedAdTargetIds,
                campaignAccessCheckRepository::getCampaignsByDynCondIdsFromFeedTypeAdgroup);
    }

    public CampaignSubObjectAccessChecker newDynamicAdTargetChecker(Long operatorUid, ClientId clientId,
                                                                    Collection<Long> dynamicAdTargetIds) {
        return create(operatorUid, clientId, dynamicAdTargetIds,
                campaignAccessCheckRepository::getCampaignsByDynCondIds);
    }

    public CampaignSubObjectAccessChecker newAdGroupChecker(Long operatorUid, ClientId clientId,
                                                            Collection<Long> adGroupIds) {
        return create(operatorUid, clientId, adGroupIds,
                campaignAccessCheckRepository::getCampaignsForAccessCheckByAdGroupIds);
    }

    public CampaignSubObjectAccessChecker newAdGroupChecker(Long operatorUid, UidClientIdShard client,
                                                            Collection<Long> adGroupIds) {
        return create(operatorUid, client, adGroupIds,
                campaignAccessCheckRepository::getCampaignsForAccessCheckByAdGroupIds);
    }

    public CampaignSubObjectAccessChecker newBidChecker(Long operatorUid, ClientId clientId, Collection<Long> bidIds) {
        return create(operatorUid, clientId, bidIds, campaignAccessCheckRepository::getCampaignsWithTypeByBidIds);
    }

    public CampaignSubObjectAccessChecker newVcardChecker(Long operatorUid, ClientId clientId, List<Long> vcardIds) {
        return create(operatorUid, clientId, vcardIds, campaignAccessCheckRepository::getCampaignsWithTypeByVcardIds);
    }

    public CampaignSubObjectAccessChecker newRelevanceMatchChecker(Long operatorUid, ClientId clientId,
                                                                   List<Long> relevanceMatchIds) {
        return create(operatorUid, clientId, relevanceMatchIds,
                campaignAccessCheckRepository::getCampaignsWithTypeByRelevanceMatchIds);
    }

    public CampaignSubObjectAccessChecker newOfferRetargetingChecker(Long operatorUid, ClientId clientId,
                                                                        List<Long> offerRetargetingIds) {
        return create(operatorUid, clientId, offerRetargetingIds,
                campaignAccessCheckRepository::getCampaignsWithTypeByOfferRetargetingIds);
    }

    private <T extends CampaignForAccessCheck>
    CampaignSubObjectAccessChecker<T> create(Long operatorUid, ClientId clientId, Collection<Long> subObjectIds,
                                             CampaignSubObjectRetriever<T> retriever) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new CampaignSubObjectAccessChecker(rbacService, operatorUid, clientId, shard, subObjectIds, retriever,
                affectedCampaignIdsContainer, accessibilityCheckerProvider.get(), featureService);
    }

    private <T extends CampaignForAccessCheck>
    CampaignSubObjectAccessChecker create(Long operatorUid, UidClientIdShard client,
                                          Collection<Long> subObjectIds, CampaignSubObjectRetriever<T> retriever) {
        return new CampaignSubObjectAccessChecker(rbacService, operatorUid, client, subObjectIds, retriever,
                affectedCampaignIdsContainer, accessibilityCheckerProvider.get(), featureService);
    }
}
