package ru.yandex.direct.core.entity.campaign.service.accesschecker;

import java.util.Optional;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

/**
 * Проверяет доступность кампаний или подобъектов (группы, баннеры...) с помощью CampaignSubObjectAccessChecker
 */
@ParametersAreNonnullByDefault
public class CampaignSubObjectAccessConstraint implements Constraint<Long, Defect> {
    private final CampaignSubObjectAccessChecker checker;
    private final Function<Long, Long> checkedObjectIdProvider;
    private final CampaignAccessDefects accessDefects;
    private final CampaignAccessType desiredAccess;
    private final AffectedCampaignIdsContainer affectedCampaignIdsContainer;

    /**
     * @param checker                      Объект с помощью которого будет выполняться проверка
     * @param checkedObjectIdProvider      Отображение идентификатора объекта, для которого требуется проверка,
     *                                     в идентификатор объекта (возможно другого типа) для которого будет
     *                                     выполняться проверка
     * @param accessDefects                Дефекты доступа
     * @param desiredAccess                Требуемый доступ к компании
     * @param affectedCampaignIdsContainer Контейнер для логирования затронутых кампаний
     */

    public CampaignSubObjectAccessConstraint(
            CampaignSubObjectAccessChecker checker,
            Function<Long, Long> checkedObjectIdProvider,
            CampaignAccessDefects accessDefects,
            CampaignAccessType desiredAccess,
            AffectedCampaignIdsContainer affectedCampaignIdsContainer) {
        this.checker = checker;
        this.checkedObjectIdProvider = checkedObjectIdProvider;
        this.accessDefects = accessDefects;
        this.desiredAccess = desiredAccess;
        this.affectedCampaignIdsContainer = affectedCampaignIdsContainer;
    }

    @Override
    public Defect apply(Long objectId) {
        if (objectId == null) {
            // Constraint'ы по умолчанию толерантны к отсутствию значения
            return null;
        }

        Long checkedObjectId = checkedObjectIdProvider.apply(objectId);

        if (!checker.objectInVisibleCampaign(checkedObjectId)) {
            return accessDefects.getNotVisible().apply(objectId);
        }

        if (!checker.objectInAllowableCampaign(checkedObjectId)) {
            return accessDefects.getTypeNotAllowable().apply(objectId);
        }

        if (desiredAccess == CampaignAccessType.READ_WRITE) {
            if (!checker.objectInWritableAndEditableCampaign(checkedObjectId)) {
                return accessDefects.getNoRights().apply(objectId);
            }

            if (checker.objectInArchivedCampaign(checkedObjectId)) {
                return accessDefects.getArchivedModification().apply(objectId);
            }
        }

        Optional<? extends CampaignForAccessCheck> campaign = checker.getCampaignFor(checkedObjectId);
        campaign
                .map(CampaignForAccessCheck::getId)
                .ifPresent(affectedCampaignIdsContainer::add);
        return null;
    }

    public CampaignSubObjectAccessChecker getChecker() {
        return checker;
    }

    Function<Long, Long> getCheckedObjectIdProvider() {
        return checkedObjectIdProvider;
    }

    CampaignAccessDefects getAccessDefects() {
        return accessDefects;
    }

    CampaignAccessType getDesiredAccess() {
        return desiredAccess;
    }
}
